package com.mapr.baseutils;

/**
  The purpose of this class is to set JVM properties that may be needed
  by the runtime in order to function. Normally these would be
  set via -D options to the JVM, but that puts a burden on custom
  Java application development so we set them here when we can.
*/

import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.security.JNISecurity;
import org.apache.log4j.Logger;


public class JVMProperties {
  static private final Logger LOG = Logger.getLogger(JVMProperties.class);

  static private String MAPRHOME = BaseUtilsHelper.getPathToMaprHome();
  static private String defaultCluster = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();

  static private boolean securityEnabled;

  static {
    JNISecurity.Initialize();
    securityEnabled = JNISecurity.IsSecurityEnabled(defaultCluster);

    if (securityEnabled) {
      setPropIfUnset("javax.net.ssl.trustStore", MAPRHOME + "/conf/ssl_truststore");
      //this property suppresses the JVM's default URL provider which
      //automatically handles SPNEGO HTTP challenges interfering with 
      //the behavior we want in our authenticator. Setting basic makes it
      //look for basic auth challenges (ignoring SPNEGO).
      //http://docs.oracle.com/javase/6/docs/technotes/guides/net/http-auth.html
      setPropIfUnset("http.auth.preference", "basic");
      setPropIfUnset("zookeeper.saslprovider", "com.mapr.security.maprsasl.MaprSaslProvider");
    } else {
      setPropIfUnset("zookeeper.saslprovider", "com.mapr.security.simplesasl.SimpleSaslProvider");
      setPropIfUnset("zookeeper.sasl.clientconfig", "Client_simple");
    }

    setPropIfUnset("java.security.auth.login.config", MAPRHOME + "/conf/mapr.login.conf");
  }

  static private void setPropIfUnset(String property, String value) {
    String current = System.getProperty(property);
    if (current == null) {
      if (LOG.isDebugEnabled()) 
        LOG.debug("Setting JVM property " + property + " to " + value);
      System.setProperty(property, value);
    } else {
      if (LOG.isDebugEnabled()) 
        LOG.debug("JVM property " + property + " already set to " + current + ", leaving unchanged");
    }
  }

  static public void init() {
    //nothing to do since all work is done in static initializer
  }
}

