package com.mapr.baseutils;

import java.util.regex.Pattern;

public class BinaryString {

  public static final Pattern UUID_PATTERN = Pattern.compile("\\p{XDigit}{8}-\\p{XDigit}{4}-\\p{XDigit}{4}-\\p{XDigit}{4}-\\p{XDigit}{12}");

  public static String toUUIDString(final byte[] uuid) {
    if (uuid == null) {
      throw new NullPointerException();
    } else if (uuid.length != 16) {
      throw new IllegalArgumentException("Expected 16 byte UUID, found " + uuid.length);
    }
    return String.format("%02x%02x%02x%02x-%02x%02x-%02x%02x-%02x%02x-%02x%02x%02x%02x%02x%02x",
        uuid[0], uuid[1], uuid[2], uuid[3], uuid[4], uuid[5], uuid[6], uuid[7],
        uuid[8], uuid[9], uuid[10], uuid[11], uuid[12], uuid[13], uuid[14], uuid[15]);
  }

  public static byte[] fromUUIDString(final String uuid) {
    if (uuid == null) {
      throw new NullPointerException();
    } else if (!UUID_PATTERN.matcher(uuid).matches()) {
      throw new IllegalArgumentException("Invalid UUID string: " + uuid );
    }
    byte[] uuidStrBytes = uuid.getBytes();
    byte[] result = new byte[16];
    for (int i = 0, j = 0; i < uuidStrBytes.length; i++) {
      if (uuidStrBytes[i] == '-') continue;
      int val = Character.digit(uuidStrBytes[i++], 16) << 4;
      val |= Character.digit(uuidStrBytes[i], 16);
      result[j++] = (byte) val;
    }
    return result;
  }

  /**
   * Taken from Hbase: base/util/Bytes.java
   * Removed offset and length, start at 0 and end at last position
   * Write a printable representation of a byte array. Non-printable
   * characters are hex escaped in the format \\x%02X, eg:
   * \x00 \x05 etc
   *
   * @param b array to write out
   * @return string output
   */
  public static String toStringBinary(final byte [] b) {
    StringBuilder result = new StringBuilder();
    int off = 0;
    int len = b.length;
    for (int i = off; i < off + len ; ++i ) {
      int ch = b[i] & 0xFF;
      if ((ch >= '0' && ch <= '9') ||
          (ch >= 'A' && ch <= 'Z') ||
          (ch >= 'a' && ch <= 'z') ||
          " `~!@#$%^&*()-_=+[]{}|;:',.<>/?".indexOf(ch) >= 0 ) {
        result.append((char)ch);
      } else {
        result.append(String.format("\\x%02X", ch));
      }
    }
    return result.toString();
  }

  public static String toStringHex(final byte [] b) {
    StringBuilder result = new StringBuilder();
    int off = 0;
    int len = b.length;
    for (int i = off; i < off + len ; ++i ) {
      int ch = b[i] & 0xFF;
      result.append(String.format("\\x%02X", ch));
    }
    return result.toString();
  }

  private static boolean isHexDigit(char c) {
    return (c >= 'A' && c <= 'F') || (c >= '0' && c <= '9');
  }

  /**
   * Takes a ASCII digit in the range A-F0-9 and returns
   * the corresponding integer/ordinal value.
   * @param ch  The hex digit.
   * @return The converted hex value as a byte.
   */
  public static byte toBinaryFromHex(byte ch) {
    if ( ch >= 'A' && ch <= 'F' )
      return (byte) ((byte)10 + (byte) (ch - 'A'));
    // else
    return (byte) (ch - '0');
  }

  public static byte [] toBytesBinary(String in) {
    // this may be bigger than we need, but let's be safe.
    byte [] b = new byte[in.length()];
    int size = 0;
    for (int i = 0; i < in.length(); ++i) {
      char ch = in.charAt(i);
      if (ch == '\\' && in.length() > i+1 && in.charAt(i+1) == 'x') {
        // ok, take next 2 hex digits.
        char hd1 = in.charAt(i+2);
        char hd2 = in.charAt(i+3);

        // they need to be A-F0-9:
        if (!isHexDigit(hd1) ||
            !isHexDigit(hd2)) {
          // bogus escape code, ignore:
          continue;
        }
        // turn hex ASCII digit -> number
        byte d = (byte) ((toBinaryFromHex((byte)hd1) << 4) + 
                          toBinaryFromHex((byte)hd2));
        b[size++] = d;
        i += 3; // skip 3
      } else {
        b[size++] = (byte) ch;
      }
    }
    // resize:
    byte [] b2 = new byte[size];
    System.arraycopy(b, 0, b2, 0, size);
    return b2;
  }
}
