package com.mapr.baseutils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

import org.apache.log4j.Logger;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.security.JNISecurity;

public class BaseUtilsHelper {

  private static final String MAPR_ENV_VAR = "MAPR_HOME";
  private static final String MAPR_PROPERTY_HOME = "mapr.home.dir";
  private static final String MAPR_HOME_PATH_DEFAULT = "/opt/mapr";
  public static final String HOST_NAME_FILE_PATH = getPathToMaprHome() + "/hostname";

  private static final Logger LOG = Logger.getLogger(BaseUtilsHelper.class);
  
  public static String clusterId = null;

  private static String hostname = null;
  
  public static synchronized String getClusterId()  {
    if ( clusterId != null ) {
      return clusterId;
    } else {
      String maprHome = BaseUtilsHelper.getPathToMaprHome();

      if ( maprHome != null ) {
        BufferedReader fis = null;
        try {
          fis = new BufferedReader(new FileReader(maprHome + "/conf/clusterid"));
          String clusterString = fis.readLine();
          String [] clusterNames = clusterString.split("\\s");
          clusterId = clusterNames[0];
          fis.close();
        } catch(FileNotFoundException fne) {
          LOG.error("Exception while trying to get info from clusterid file", fne);
        } catch (IOException ioe) {
          LOG.error("Exception while trying to read file: " + maprHome + "/conf/clusterid", ioe);
        }  
      }
      return clusterId;
    }  
  }

  public static String getPathToMaprHome()
  {
      String maprHome = System.getenv(MAPR_ENV_VAR);

      if (maprHome == null)
      {
          maprHome = System.getProperty(MAPR_PROPERTY_HOME);
          if (maprHome == null)
          {
              maprHome = MAPR_HOME_PATH_DEFAULT;
          }
      }
      return maprHome;
  }

  /** 
   * Reads the hostname from /opt/mapr/hostname file.
   *
   * @return the hostname
   */
  public synchronized static String getMapRHostName() {
    if (hostname == null) {
        hostname = readMapRHostNameFromFile(HOST_NAME_FILE_PATH);
    }   
    return hostname;
  }

  private static String readMapRHostNameFromFile(String filePath) {
    if (!new File(filePath).exists()) {
      LOG.debug(filePath + " does not exist. Assuming client-only installation.."); 
      return "MAPR_CLIENT";
    }

    BufferedReader breader = null;
    FileReader freader = null;
    try {
      freader = new FileReader(filePath);
      breader = new BufferedReader(freader);
      return breader.readLine();
    } catch (Exception e) {
      /* On any exception while reading hostname return null */
      LOG.warn("Error while reading " + filePath, e); 
    } finally {
      try {
        if (breader != null) {
          breader.close();
        }   
      } catch (Throwable t) {
        LOG.error("Failed to close buffered reader", t); 
      }   

      try {
        if (freader != null) {
          freader.close();
        }   
      } catch (Throwable t) {
        LOG.error("Failed to close " + filePath, t); 
      }   
    }
    return null;
  }

  // SHOULD BE USED ONLY BY TEST CODE
  public synchronized static void setMapRHostName(String host) {
     if (hostname == null) {
      hostname = host;
     }
  }
  public static String getUrlScheme() {
    return JNISecurity.IsSecurityEnabled(CLDBRpcCommonUtils.getInstance()
        .getCurrentClusterName()) 
        ? "https://" 
        : "http://";
  }


  public static String getStandardUiPortProperty() {
    return "service.ui.port";
  }

  public static String getUIPortProperty() {
    return JNISecurity.IsSecurityEnabled(CLDBRpcCommonUtils.getInstance()
        .getCurrentClusterName())
        ? "service.ui.secure.port"
        : getStandardUiPortProperty();
   }
}
