/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.audit;

import java.net.*;
import java.io.*;

import org.apache.log4j.Logger;

public class ExpandAuditLogs {

  private static final Logger LOG = Logger.getLogger(ExpandAuditLogs.class);

  private static void printUsage() {
    StringBuilder builder = new StringBuilder();
    builder.append("usage: expandaudit\n");
    builder.append("            [-volumename volume name]\n");
    builder.append("            [-volumeid volume ids. Either volume name");
    builder.append(" or id must be specified]\n");
    builder.append("            -o output directory\n");
    builder.append("            [-d Specify for deleted volumes only]\n");
    builder.append("            [-cluster cluster name]\n");
    builder.append("            [-t number of threads used for parallel");
    builder.append(" expansion across cluster nodes. default 10]\n");
    builder.append ("For deleted volumes, user specified volume name will be used during expansion\n");
    System.out.println(builder.toString());
    LOG.info(builder.toString());
  }

  static void LogErrorAndExit(String string)
  {
    LOG.error(string);
    System.out.println(string);
    printUsage();
    System.exit(1);
  }

  static void LogInfo(String string)
  {
    LOG.info(string);
    System.out.println(string);
  }

  public static void main(String args[]) {

     if (args.length < 2) {
       printUsage();
       System.exit(1);
     }

    String outputDir = "";
    String inputDir = "/var/mapr/local/";
    int volId = 0;
    String volName= "";
    String cluster = "";
    boolean vollistCheck = true;
    boolean expandToolLogs = false;
    int numThreads = 10;
    boolean deleted = false;
    boolean verbose = false;

    for (int i = 0; i < args.length; i++)  {

      if (args[i].equals("-d")) deleted = true;
      else if (args[i].equals("-v")) verbose = true;
      else if (args[i].equals("-o")) {
        if (i != args.length - 1) {
          outputDir = args[i + 1];
          i++;
        }
        else LogErrorAndExit("-o option requires an output directory where " +
                             "expanded audit logs stored");
       } else if (args[i].equals("-volumeid")) {
         if (i !=  args.length - 1) {
           try {
              volId = Integer.parseInt(args[i + 1]);
              i++;
            }
            catch (NumberFormatException e) {
              LogErrorAndExit("Invalid value for -volumeid option = " + args[i+1]);
            }
          } else LogErrorAndExit("-volumeid option requires a volume Id");
       } else if (args[i].equals("-volumename")) {
         if (i != args.length - 1) {
           volName  = args[i + 1];
           i++;
         }
         else LogErrorAndExit("-volumename option requires a volume name");
       } else if (args[i].equals("-i")) {
         if (i != args.length - 1) {
           inputDir  = args[i + 1];
           i++;
         }
         else LogErrorAndExit("-i option requires an input directory for " + 
                "cluster audit logs location");
       } else if (args[i].equals("-cluster")) {
         if (i != args.length - 1) {
           cluster  = args[i + 1];
           i++;
         }
         else LogErrorAndExit("-cluster option requires cluster name");
       } else if (args[i].equals("-t")) {
         if (i != args.length - 1) {
           try {
              numThreads = Integer.parseInt(args[i + 1]);
              i++;
            }
            catch (NumberFormatException e) {
              LogErrorAndExit("Invalid value for -t option = " + args[i+1]);
            }
            if (numThreads <= 0)
              LogErrorAndExit("Invalid number of threads for -t option = " +
                              numThreads);
         }
         else LogErrorAndExit("-t option requires to specify number of " + 
                              "threads in thread pool");
       } else if (args[i].equals("-vl")) {
         vollistCheck = false;
       } else if (args[i].equals("-expandtoollogs")) {
         expandToolLogs = true;
       } else {
         LogErrorAndExit("Unexpected arg = " + args[i]);
       }
    }

    if ((volId == 0) && (volName.length() == 0))
      LogErrorAndExit("Atleast one of the args - volume id or " +
                      "name must be specified");
    if (!deleted && (volId != 0) && (volName.length() != 0))
      LogErrorAndExit("Only one of the args - volume id or " +
                      "name must be specified");
    if (outputDir.length() == 0)
      LogErrorAndExit("output directory must be specified");

    ExpandAuditLogCluster expandAuditLogCluster = new ExpandAuditLogCluster();

    try {
      expandAuditLogCluster.initialize(outputDir, inputDir, volId, volName,
                                vollistCheck, cluster, numThreads, deleted,
                                verbose, expandToolLogs);
      expandAuditLogCluster.Phase1Expand();
      LogInfo("Phase1 Audit Log Expansion Done. Starting Phase2");
      expandAuditLogCluster.Phase2Expand();
      expandAuditLogCluster.ShutDown();
      LogInfo("Phase2 Audit Log Expansion Done. Exiting");
    } catch (Exception e) {
      e.printStackTrace();
    }
    // @TODO Program hangs here, without explicit exit - Debug ?
    System.exit(0);
  }
}
