/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.audit;

import java.net.*;
import java.io.*;
import org.apache.hadoop.fs.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.conf.*;
import org.apache.hadoop.util.*;
import org.apache.hadoop.io.*;
// import org.json.simple.JSONObject;
// import org.json.simple.JSONValue;
// import org.json.simple.parser.*;
import net.minidev.json.*;
import net.minidev.json.parser.*;
import com.mapr.fs.MapRFileSystem;

import java.util.concurrent.Executors;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ExecutionException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import org.apache.log4j.Logger;

public class ExpandAuditLogCluster {

  private static final String MAPRFS_URI = "maprfs:///";
  private MapRFileSystem fs; // need MapRFileSystem for getMountPath
  private int volId;
  private String volIdStr;
  private String volName;
  private String cluster;
  private boolean vollistCheck;
  private boolean expandToolLogs;
  private boolean deleted;
  private boolean verbose;
  private String inputDir, outputDir;
  private ExecutorService expandAuditLogClusterPool;
  private Path outputPath;

  private ConcurrentFIFOHashMap<String, String> fidPathMap = null;
  private ConcurrentHashMap<String, String> failedAttrFid = null;
  private ConcurrentHashMap<Integer, String> userMap =
    new ConcurrentHashMap<Integer, String>();

  private static final Logger LOG =
    Logger.getLogger(ExpandAuditLogCluster.class);

  final static String FSAUDITPREFIX = "FSAudit";
  final static String DBAUDITPREFIX = "DBAudit" ;
  final static String EXPANDAUDITLOGPREFIX= "ExpandAudit";
  final static String JSONSUFFIX = ".json";
  final static String VOLUMEID  = "volumeId";
  final static String CHILDFID  = "childFid";
  final static String TABLEFID  = "tableFid";
  final static String PARENTFID  = "parentFid";
  final static String CHILDNAME  = "childName";
  final static String SRCNAME  = "srcName";

  // result of parsing a single audit record
  private enum LogParseResult {
    JSONPARSEFAILED,
    VOLIDMISMATCH,
    FIDEXPANDED,
    FIDEXPANDFAILED,
    READONLY
  }

  // Result of parsing a single audit log file
  private enum FileParseResult {
    NONE,         // No logs in output
    PENDING,      // Pending fidexpansion in output
    COMPLETED     // All fids expanded
  }

  ContainerFactory orderedKeyFactory = new ContainerFactory() {
    @Override public List createArrayContainer() {return new LinkedList();}
    @Override public Map createObjectContainer() {return new LinkedHashMap();}
  };

  CreateRecordHandling createRecordHandling;
  DeleteRecordHandling deleteRecordHandling;

  private class Metrics {
    long nCount;
    long nSize;

    Metrics() 
    {
      nCount = 0;
      nSize = 0;
    }
  }

  long numRecords = 0, processingTime = 0, convertedSize = 0;
  //@TODO delete debug data for resolving perf issue
  long parsingTime = 0;

  void LogErrorAndExit(String string)
  {
    LOG.error(string);
    System.out.println(string);
    System.exit(1);
  }

  void LogErrorAndContinue(String string)
  {
    LOG.error(string);
    System.out.println(string);
  }

  void LogInfo(String string)
  {
    LOG.info(string);
    System.out.println(string);
  }

  public void initialize(String outputDir, String inputDir, int volId,
    String volName, boolean vollistCheck, String cluster, int threadCount,
    boolean deleted, boolean verbose, boolean expandToolLogs) throws Exception
  {
    Configuration conf = new Configuration();
    String uri = MAPRFS_URI;
    if (cluster.length() > 0) uri = uri + "mapr/" + cluster;
    conf.set("fs.default.name", uri);
    fs = new MapRFileSystem();
    fs.initialize(URI.create(uri), conf, true);

    this.volId = volId;
    this.volName = volName;
    this.outputDir = outputDir;
    this.inputDir = inputDir;
    this.vollistCheck = vollistCheck;
    this.deleted = deleted;
    this.verbose = verbose;
    this.cluster = cluster;
    this.expandToolLogs = expandToolLogs;

    LogInfo("ExpandAuditLogs with args:");
    if (volId != 0) LogInfo("volId = " + volId);
    if (volName != null) LogInfo("volumeName = " + volName);
    LogInfo("Output Directory for expanded audit logs = " + outputDir);
    LogInfo("Input Directory for audit logs = " + inputDir);

    String hostName = null;
    try {
      InetAddress myIP = InetAddress.getLocalHost();
      // getHostname returns DNS hostname
      if (myIP != null) hostName = myIP.getHostName();
    } catch (UnknownHostException e) {
    }
    if (hostName != null) LogInfo("HostName = " +  hostName);
    else LogInfo("Unable to retrieve HostName");
    LogInfo("Please look in mapr logs location for errlog");
    if (verbose) LOG.info("Number of threads = " +  threadCount);

    int fidPathMapSize = 64 * 1024;
    Runtime runtime = Runtime.getRuntime();
    // 8 bits for hash element size, 4 bits for 1/16th memory
    if ((int)(runtime.freeMemory() >> 12) > fidPathMapSize)
      fidPathMapSize = (int)(runtime.freeMemory() >> 12);

    if (verbose) LOG.info("Setting fidcache size = " + fidPathMapSize);
    fidPathMap = new ConcurrentFIFOHashMap<String, String>(fidPathMapSize);
    failedAttrFid = new ConcurrentHashMap<String, String>(fidPathMapSize * 10);
    createRecordHandling = new CreateRecordHandling(fs);
    deleteRecordHandling = new DeleteRecordHandling(fidPathMap, fs);

    expandAuditLogClusterPool = Executors.newFixedThreadPool(threadCount);

    StringBuilder sb = new StringBuilder();
    if (volId != 0) {
      sb.append("maprcli volume list");
      if (cluster.length() > 0) sb.append(" -cluster " + cluster);
      sb.append(" -filter volumeid==" + this.volId);
      sb.append(" -columns volumename -noheader");
      String cldbVolName = ExecuteCommand(sb.toString());
      if (cldbVolName.length() > 0) {
        if (deleted) LogErrorAndExit("-d option specified, but volumeid = " +
                       volId + " exists with volumename = " + cldbVolName);
        LogInfo("Using CLDB Extracted VolumeName = " + cldbVolName);
        this.volName = cldbVolName;
      } else if (this.volName.length() > 0 && deleted) {
          LogInfo("Using user specified volumeName = " + this.volName+
            "for VolumeId = " + this.volId);
      } else if (this.volName.length() == 0) {
        if (!deleted)
          LogErrorAndExit("Unable to Extract VolumeName for specified " +
            "VolumeId = " + volId + ". If deleted volume, please " +
            "re-run the tool with -d option");
      }
      this.volIdStr = String.valueOf(volId);
    } else {
      sb.append("maprcli volume info");
      if (cluster.length() > 0) sb.append(" -cluster " + cluster);
      sb.append(" -name " + this.volName);
      sb.append(" -columns volumeid -noheader");
      String volIdStr = ExecuteCommand(sb.toString());
      try {
        this.volId = Integer.parseInt(volIdStr);
      } catch (NumberFormatException e) {
        String userMsg =
          "Unable to convert specified VolumeName = " + this.volName + " to " +
          "VolumeId. Probably volume is deleted or cldb not accessible." + 
          " VolumeName from CLDB = " + volIdStr + "\n";
        LogErrorAndExit(userMsg);
      }
      this.volIdStr = volIdStr;
      LogInfo("Using CLDB Extracted VolumeId = " + this.volIdStr);
    }
  }

  void ShutDown()
  {
    expandAuditLogClusterPool.shutdown();
  }

  public void Phase1Expand() throws Exception
  {
    long beginTime = System.currentTimeMillis(); 

    Path outputDirPath = new Path(outputDir);
    if (!fs.exists(outputDirPath) || !fs.isDirectory(outputDirPath))
      LogErrorAndExit("Output Directory = " + outputDir + " is invalid");

    outputPath = new Path(outputDir + "/" + volId + "/");
    if (!fs.exists(outputPath) && !fs.mkdirs(outputPath))
      LogErrorAndExit("create output directory = " +
                      outputPath.toString() + " failed");

    Path clusterNodesDir = new Path(inputDir);
    if (!fs.exists(clusterNodesDir) || !fs.isDirectory(clusterNodesDir))
      LogErrorAndExit("Invalid Input cluster audit log " +
        "location = " + inputDir);

    FileStatus[] fileStatus = fs.listStatus(clusterNodesDir);
    Path[] paths = FileUtil.stat2Paths(fileStatus);
    Collection<Future> futures = new LinkedList<Future>();

    for(Path path : paths) { // looping over all nodes
      String pathStr = path.toString();
      String nodeName =
        pathStr.substring(pathStr.lastIndexOf('/') + 1, pathStr.length());
      Path auditPath = new Path(path + "/audit/");
      if (fs.exists(auditPath)) {
        LogInfo("Processing audit logs from Node = " + nodeName +
                 "\naudit logs path = " + auditPath);
        Callable<Void> workItem = 
          new ExpandAuditLogNodePhase1ExecutorService(this, nodeName, auditPath);
        futures.add(expandAuditLogClusterPool.submit(workItem));
        FileStatus[] instances = fs.listStatus(auditPath);
        for (FileStatus s: instances) {
          if (s.isDirectory()) {
            Path[] instancePathArr = FileUtil.stat2Paths(new FileStatus[] {s});
            Path instancePath = instancePathArr[0];
            LogInfo("Processing audit logs from Node = " + nodeName +
                    "\naudit logs path = " + instancePath);
            nodeName = nodeName + "/" + instancePath.getName();
            Callable<Void> instanceWorkItem = 
              new ExpandAuditLogNodePhase1ExecutorService(this, nodeName,
                                                          instancePath);
            futures.add(expandAuditLogClusterPool.submit(instanceWorkItem));
          }
        }
      } else LOG.info("Missing audit log directory for node = " + nodeName);
    }

    for (Future future : futures) future.get();

    long endTime = System.currentTimeMillis(); 
    if (verbose) {
    if (endTime > beginTime) processingTime = endTime - beginTime;
      LogInfo("Phase1 Summary:" + 
               "\nProcessing time (secs) = " + (processingTime/1000) +
               "\nConverted size = " + convertedSize +
               "\nNumRecords = " + numRecords);
      //@TODO temp code`
      LogInfo("Phase1 JsonParsing time (secs) = " + (parsingTime/1000));
    }
    processingTime = 0; convertedSize = 0; numRecords = 0;
    parsingTime = 0;
  }

  void ExpandAuditLogNodePhase1(Path auditPath, String nodeName)
    throws Exception
  {

    String  outputDir = outputPath.toString() + "/" + nodeName + "/";
    Path outputPath = new Path(outputDir);

    FileStatus[] fileStatus = fs.listStatus(auditPath);

    String curDayFSLogFile = "";
    Path curDayFSLogFileOutPath = null; // current day output log file

    String curDayDBLogFile = "";
    Path curDayDBLogFileOutPath = null; // current day output log file
 
    String curDayExpandAuditLogFile = "";
    Path curDayExpandAuditLogFileOutPath = null; // current day output log file

    Metrics metrics = new Metrics();
    long beginTime = System.currentTimeMillis(); 

    // Audit logs begin with "FSAudit" or "DBAudit"
    // Vollist files begin with "Vollist_FSAudit" or "Vollist_DBAudit"
    for (int i = 0; i < fileStatus.length; i++) {
      if (fileStatus[i].getLen() == 0) continue;
      Path ipath = fileStatus[i].getPath();
      String pathStr = ipath.toString();
      String fileName =
        pathStr.substring(pathStr.lastIndexOf('/') + 1, pathStr.length());
      if (!fileName.startsWith(FSAUDITPREFIX) &&
          !fileName.startsWith(DBAUDITPREFIX) &&
          !(expandToolLogs && fileName.startsWith(EXPANDAUDITLOGPREFIX)))
        continue;
      if (!fileName.endsWith(JSONSUFFIX)) continue;
      fileName = fileName.substring(0, fileName.lastIndexOf(JSONSUFFIX));

      //TODO:: make vollist path for expand audit path
      String vollistPathStr = 
        pathStr.substring(0, pathStr.lastIndexOf('/')) + "Vollist_" + fileName;
      if (vollistCheck && !checkVolListFile(vollistPathStr, volId)) continue;

      Path opath = new Path(outputDir + "/" + fileName + JSONSUFFIX);
      Path opathp = new Path(outputDir + "/" + fileName + ".pending.json");
      // Skip if expanded audit log file exists in output. If delete
      // true, then read the input audit log files always
      boolean readOnly = false;
      if ((fs.exists(opath) || fs.exists(opathp)) && (!deleted)) {
        if (!deleted) {
          if (verbose) LOG.info("Skipping Expanding AuditLog = " + pathStr);
          // Any pending files from a previous run will be processed in phase2
          continue;
        }
        readOnly = true; // process create audit logs
      }

      // delete any .part files that will be recreated in output directory
      // These are current days expanded files from an earlier run
      Path opathpart = new Path(outputDir + "/" + fileName + ".part.json");
      Path opathpartp = new Path(outputDir + "/" + fileName +
                              ".part.pending.json");
      if (fs.exists(opathpart)) fs.delete(opathpart, false);
      if (fs.exists(opathpartp)) fs.delete(opathpartp, false);

      Path otpath = new Path(opath.toString() + ".inPhase1");
      LOG.info("Processing Audit Log File = " + pathStr);
      FileParseResult res = ExpandAuditLogFilePhase1(ipath, otpath, readOnly,
                              metrics);

      if (res == FileParseResult.NONE) continue; // No expanded audit logs

      if (res == FileParseResult.PENDING) { // audit logs with unresolved fids
         fs.rename(otpath, opathp);
         opath = opathp;
      } else fs.rename(otpath, opath);
     
      if (fileName.startsWith("FSAudit") &&
          curDayFSLogFile.compareTo(ipath.toString()) < 0) {
        curDayFSLogFile = ipath.toString();
        curDayFSLogFileOutPath = opath;
      }

      if (fileName.startsWith("DBAudit") &&
          curDayDBLogFile.compareTo(ipath.toString()) < 0) {
        curDayDBLogFile = ipath.toString();
        curDayDBLogFileOutPath = opath;
      }

      if (fileName.startsWith("ExpandAudit") &&
          curDayExpandAuditLogFile.compareTo(ipath.toString()) < 0) {
        curDayExpandAuditLogFile = ipath.toString();
        curDayExpandAuditLogFileOutPath = opath;
      }


    }

    String strippedCurDayFSLogFile = "";
    String strippedCurDayDBLogFile = "";
    String strippedCurDayExpandAuditLogFile = "";

    if (curDayFSLogFile.length() > 0) {
      strippedCurDayFSLogFile =
        curDayFSLogFile.substring(
          curDayFSLogFile.lastIndexOf(FSAUDITPREFIX) + FSAUDITPREFIX.length(),
          curDayFSLogFile.length());
    }

    if (curDayDBLogFile.length() > 0) {
      strippedCurDayDBLogFile =
        curDayDBLogFile.substring(
          curDayDBLogFile.lastIndexOf(DBAUDITPREFIX) + DBAUDITPREFIX.length(),
          curDayDBLogFile.length());
    }

    if (curDayExpandAuditLogFile.length() > 0) {
      strippedCurDayExpandAuditLogFile =
        curDayExpandAuditLogFile.substring(
          curDayExpandAuditLogFile.lastIndexOf(EXPANDAUDITLOGPREFIX) +
          EXPANDAUDITLOGPREFIX.length(),
          curDayExpandAuditLogFile.length());
    }


    boolean renameFSFile = true; 
    boolean renameDBFile = true; 
    boolean renameExpandAuditFile = true;

    //TODO: Do we have to do something for ExpamndAudit file
    if (strippedCurDayFSLogFile.compareTo(strippedCurDayDBLogFile) > 0)
      renameDBFile = false;
    else if (strippedCurDayFSLogFile.compareTo(strippedCurDayDBLogFile) < 0) {
      //Not so sure
      renameExpandAuditFile = false;
      renameFSFile = false;
    }

    if (strippedCurDayFSLogFile.length() == 0) renameFSFile = false;
    if (strippedCurDayDBLogFile.length() == 0) renameDBFile = false;
    if (strippedCurDayExpandAuditLogFile.length() == 0) 
      renameExpandAuditFile = false;

    if (renameFSFile) RenameCurDayFile(curDayFSLogFileOutPath);
    if (renameDBFile) RenameCurDayFile(curDayDBLogFileOutPath);
    if (renameExpandAuditFile)
      RenameCurDayFile(curDayExpandAuditLogFileOutPath);
    
    long endTime = System.currentTimeMillis(); 
    if (verbose && (endTime > beginTime))
      LOG.info("Phase1 Summary for NodeName = " +  nodeName +
               "\nProcessingTime (in ms) = " + (endTime - beginTime) + 
               "\nNumber of audit logs read = " + metrics.nCount +
               "\nSize of audit logs processed = " + metrics.nSize);
    synchronized (this) {
      convertedSize = convertedSize + metrics.nSize;
      numRecords = numRecords + metrics.nCount;
    }
  }

  void RenameCurDayFile(Path opath) throws Exception
  {
    Path newopath;
    int last;

    if (!fs.exists(opath)) return;
    if (opath.toString().endsWith(".pending.json")) {
      last = opath.toString().lastIndexOf(".pending.json");
      newopath =
        new Path(opath.toString().substring(0, last)  + ".part.pending.json");
    }
    else {
      last = opath.toString().lastIndexOf(JSONSUFFIX);
      newopath =
        new Path(opath.toString().substring(0, last)  + ".part.json");
    }
    fs.rename(opath, newopath);
  }

  boolean checkVolListFile(String pathStr, int volId)  
  {
    Path volListPath = new Path(pathStr);
    try {
      if (!fs.exists(volListPath) || !fs.isFile(volListPath)) return true;
      FSDataInputStream in = fs.open(volListPath);
      BufferedReader reader = new BufferedReader(new InputStreamReader(in));
      boolean res = false;
      int fileVolId = 0;
      String line;
      long lineNumber = 0;
      while (true) {
        if ((line = reader.readLine()) == null)  break; // eof
        lineNumber = lineNumber + 1;
        try {
          fileVolId = Integer.parseInt(line);
        } catch (NumberFormatException e) {
          LOG.error("Found Invalid VolId = " + volId +
                    " in vollist file = " + volListPath +
                    " at line number = " + lineNumber);
          res = true; // worst case assumption for invalid vollist file
          break;
        }
        if (volId == fileVolId) {
          res = true;
          break;
        }
      }
      reader.close(); in.close();
      if (lineNumber == 0) {
        LOG.error("Found Invalid vollist file = " + volListPath +
                  " with zero size");
        return (true);
      }
      return res;
    } catch (IOException e) {
      LOG.error("IOException", e);
      return (true); // worst case assumption for invalid vollist file
    }
  }

  FileParseResult
  ExpandAuditLogFilePhase1(Path ipath, Path opath,  boolean readOnly,
    Metrics metrics) throws Exception
  {
    FSDataInputStream in = null;
    FSDataOutputStream out = null;
    try {
      in = fs.open(ipath);
    } catch (IOException e) {
      LOG.error("Audit Log File Open Exception", e);
      return  FileParseResult.NONE;
    }
    BufferedReader reader =
      new BufferedReader(new InputStreamReader(in), 4 * 1024 * 1024);
    BufferedWriter writer = null;
    if (!readOnly) {
      out = fs.create(opath);
      writer = new BufferedWriter(new OutputStreamWriter(out), 2 * 1024 * 1024);  
    }
    String line;
    boolean logsWritten = false;
    boolean pending = false;
    long linenum = 0;

    JSONParser jsonParser = new JSONParser();
    while (true) {
      if ((line = reader.readLine()) == null) break; //eof
      metrics.nCount = metrics.nCount + 1;
      metrics.nSize = metrics.nSize + line.length();
      linenum = linenum + 1;
      LogParseResult res  = ExpandAuditLogPhase1(writer, jsonParser, line,
                              readOnly, ipath.toString(), linenum);
      if (res == LogParseResult.FIDEXPANDED) logsWritten = true;
      else if (res == LogParseResult.FIDEXPANDFAILED) {
        logsWritten = true;
        pending = true;
      } else if (res == LogParseResult.JSONPARSEFAILED)
        LOG.error("Failure Json Decode: Log file = " + ipath.toString() +
         "\nLog = " + line);
      // else VOLIDMISMATCH / READONLY
    }
    reader.close(); in.close();
    if (!readOnly) {
      writer.close(); out.close();
    } else return FileParseResult.NONE;
    if (!logsWritten) {
      fs.delete(opath, false);
      return FileParseResult.NONE;
    }
    else if (pending) return  FileParseResult.PENDING;
    return FileParseResult.COMPLETED;
  }

  LogParseResult ExpandAuditLogPhase1(BufferedWriter writer,
    JSONParser parser, String line, boolean readOnly,
    String ipathStr, long linenum) throws Exception
  {
     LinkedHashMap ijsonObj;

     if ((line.length() == 0) || !line.contains(volIdStr))
       return LogParseResult.VOLIDMISMATCH;

     long beginTime = System.currentTimeMillis(); 
     try {
       ijsonObj = (LinkedHashMap)parser.parse(line, orderedKeyFactory);
     } catch (ParseException e) {
       LogErrorAndContinue("JsonParse Exception in Log File = " + ipathStr +
         ":" + linenum + "\nposition = " + e.getPosition());
       return LogParseResult.JSONPARSEFAILED;
     } catch (ClassCastException e) { // can happen for garbled audit logs
       LogErrorAndContinue("Unexpected Exception during json parse" +
         "\nLog file = " + ipathStr + ":" + linenum +
         "\nmessage = " + e.toString());
       return LogParseResult.JSONPARSEFAILED;
     }

     long endTime = System.currentTimeMillis(); 
     if (endTime > beginTime)
       synchronized (this) {
         parsingTime = parsingTime + endTime - beginTime;
       }

     if (!ijsonObj.containsKey(VOLUMEID)  || 
         !volIdStr.equals(ijsonObj.get(VOLUMEID).toString()))
       return LogParseResult.VOLIDMISMATCH;

     String operation = null;
     if (ijsonObj.get("operation") != null)
       operation = ijsonObj.get("operation").toString();

     if ((operation != null) && (operation.equals("CREATE") ||
         operation.equals("CREATESYM") || operation.equals("MKDIR")  ||
         operation.equals("DB_TABLECREATE")))
       HandleCreateRecord(ijsonObj, ipathStr);

     if (readOnly) return (LogParseResult.READONLY);

     if ((operation != null) &&
        (operation.equals("DELETE") || operation.equals("RMDIR")))
       HandleDeleteRecord(ijsonObj, ipathStr);

     boolean expFailed = false;
     LinkedHashMap ojsonObj = new LinkedHashMap();
     Iterator iter = ijsonObj.entrySet().iterator();
     while(iter.hasNext()){
       Map.Entry entry = (Map.Entry)iter.next();
       String key = entry.getKey().toString();
       String value = entry.getValue().toString();

       if (key.endsWith("Fid")) {
         String expandedFid = null;
         if (fidPathMap.get(value) != null)
           expandedFid = fidPathMap.get(value).toString();
         else if (deleteRecordHandling.get(value) != null) {
           expandedFid = deleteRecordHandling.get(value).toString();
         } else {
           try {
             expandedFid = null;
             if (failedAttrFid.get(value) == null) {
               expandedFid = fs.getMountPathFid(value);
             }

           } catch(IOException e) {
             LogErrorAndContinue("getMountPath failed for invalid fid = " + value +
               " in log file = " + ipathStr + "\naudit log = " + line);
             expandedFid = null;
           }

           if (expandedFid != null) {
             fidPathMap.put(value, expandedFid);
           } else {
             failedAttrFid.putIfAbsent(value, "1");
           }
         }
         if (expandedFid != null) {
           ojsonObj.put(key.replace("Fid", "Path"), expandedFid);
           ojsonObj.put(key, value); // Retain original Fid field
         } else {
           ojsonObj.put(key, value);
           expFailed = true;
         }
       } else if (volName.length() != 0 && key.equals(VOLUMEID)) {
           ojsonObj.put("VolumeName", volName);
           ojsonObj.put(VOLUMEID, entry.getValue()); // Retain original volumeId Field
       } else if (key.equals("uid")) {
         String userName = "";
         try {
           int uidInt = Integer.parseInt(value);
           if (userMap.get(uidInt) != null) userName = userMap.get(uidInt).toString();
           else {
             userName = ExecuteCommand("getent passwd "  + value);
             if (userName.indexOf(':') != -1)
               userName = userName.substring(0, userName.indexOf(':'));
             if (userName.length() > 0) userMap.putIfAbsent(uidInt, userName);
           }
         } catch (NumberFormatException e) {
           LOG.error("Found Invalid userId = " + value);
         }
         if (userName.length() > 0) {
           ojsonObj.put("user", userName);
           ojsonObj.put("uid", entry.getValue()); // Retain original uid field
         }
         else ojsonObj.put("uid", entry.getValue());
       } else if (key.equals("accessReq") || key.equals("accessResp")) {
         StringBuilder expandedPerm = new StringBuilder();
         try {
           int nfsPerm = Integer.parseInt(value, 8);
           if ((nfsPerm & 0x0001) == 0x0001) expandedPerm.append("read,");
           if ((nfsPerm & 0x0002) == 0x0002) expandedPerm.append("lookup,");
           if ((nfsPerm & 0x0004) == 0x0004) expandedPerm.append("modify,");
           if ((nfsPerm & 0x0008) == 0x0008) expandedPerm.append("extend,");
           if ((nfsPerm & 0x0010) == 0x0010) expandedPerm.append("delete,");
           if ((nfsPerm & 0x0020) == 0x0020) expandedPerm.append("execute,");
           if (expandedPerm.length() > 0)
             expandedPerm.setLength(expandedPerm.length() - 1);
         } catch (NumberFormatException e) {
           LOG.error("Found Invalid accessReq or accessResp = " + value);
         }
         if (expandedPerm.length() > 0)
           ojsonObj.put(key + "InText", expandedPerm.toString());
         ojsonObj.put(key, value); // Retain original key/value field
       } else ojsonObj.put(key, entry.getValue());
     }
     String expandedLog = JSONValue.toJSONString(ojsonObj).replace("\\", "");
     writer.write(expandedLog + "\n");
     if (expFailed) return LogParseResult.FIDEXPANDFAILED;
     return LogParseResult.FIDEXPANDED;
  }

  void HandleCreateRecord(LinkedHashMap ijsonObj, String ipathStr)
    throws Exception
  {
    String cFid = null, pFid = null, cName = null;
    if (ijsonObj.get(CHILDFID) != null)
      cFid = ijsonObj.get(CHILDFID).toString();
    // Only  either ChildFid or tableFid field can exist in audit log
    if (ijsonObj.get(TABLEFID) != null)
      cFid = ijsonObj.get(TABLEFID).toString();
    if (ijsonObj.get(PARENTFID) != null)
      pFid = ijsonObj.get(PARENTFID).toString();
    if (ijsonObj.get(CHILDNAME) != null)
      cName = ijsonObj.get(CHILDNAME).toString();
    if (ijsonObj.get(SRCNAME) != null)
      cName = ijsonObj.get(SRCNAME).toString();
    if ((cFid == null) || (pFid == null) || (cName == null)) return;
    if (!fs.isFidString(cFid)) {
      LOG.error("Invalid Child Fid = " + cFid + " in log file = " +
        ipathStr);
      return;
    }
    if (!fs.isFidString(pFid)) {
      LOG.error("Invalid parent Fid = " + pFid + " in log file = " +
        ipathStr);
      return;
    }
    createRecordHandling.ProcessCreateRecord(pFid, cFid, cName);
  }

  void HandleDeleteRecord(LinkedHashMap ijsonObj, String ipathStr)
    throws Exception
  {
    String cFid = null, pFid = null, cName = null;
    if (ijsonObj.get(CHILDFID) != null)
      cFid = ijsonObj.get(CHILDFID).toString();
    if (ijsonObj.get(PARENTFID) != null)
      pFid = ijsonObj.get(PARENTFID).toString();
    if (ijsonObj.get(CHILDNAME) != null)
      cName = ijsonObj.get(CHILDNAME).toString();
    if ((cFid == null) || (pFid == null) || (cName == null)) return;
    deleteRecordHandling.ProcessDeleteRecord(pFid, cFid, cName, ipathStr);
  }

  public void Phase2Expand() throws Exception
  {
    long beginTime = System.currentTimeMillis(); 

    createRecordHandling.ResolveAllFids();

    FileStatus[] fileStatus = fs.listStatus(outputPath);
    Path[] paths = FileUtil.stat2Paths(fileStatus);
    Collection<Future> futures = new LinkedList<Future>();

    for (Path path : paths) {
      if (!fs.isDirectory(path)) {
        LOG.error("Unexpected non directory path at = " + path.toString());
        continue;
      }
        Callable<Void> workItem = 
          new ExpandAuditLogNodePhase2ExecutorService(this, path);
        futures.add(expandAuditLogClusterPool.submit(workItem));
        FileStatus[] instances = fs.listStatus(path);
        for (FileStatus s: instances) {
          if (s.isDirectory()) {
            Path[] instancePathArr = FileUtil.stat2Paths(new FileStatus[] {s});
            Path instancePath = instancePathArr[0];
            Callable<Void> instanceWorkItem = 
              new ExpandAuditLogNodePhase2ExecutorService(this, instancePath);
            futures.add(expandAuditLogClusterPool.submit(instanceWorkItem));
          }
        }
 
    }

    for (Future future : futures) future.get();

    long endTime = System.currentTimeMillis(); 
    if (verbose) {
      if (endTime > beginTime) processingTime = endTime - beginTime;
      LOG.info("Phase2 Summary:" + 
               "\nProcesing time (secs) = " + (processingTime/1000) +
               "\nConverted size = " + convertedSize + 
               "\nNumRecords = " + numRecords);
    }
  }

  void ExpandAuditLogNodePhase2(Path nodePath) throws Exception
  {
    FileStatus[] fileStatus = fs.listStatus(nodePath);
    Path[] paths = FileUtil.stat2Paths(fileStatus);
    Path opath;

    for (int i = 0; i < fileStatus.length; i++) {
      Path ipath = fileStatus[i].getPath();

      if (!fs.isFile(ipath) || (fileStatus[i].getLen() == 0)) continue;

      if (!ipath.toString().endsWith(".pending.json")) continue;
      LOG.info("ExpandAuditLogNodePhase2: Processing pending file = " +
               ipath.toString());
      int last = ipath.toString().lastIndexOf(".pending.json");
      opath = new Path(ipath.toString().substring(0, last) + JSONSUFFIX);
      Path otpath = new Path(ipath.toString().substring(0, last) + ".inPhase2");
      boolean res = ExpandAuditLogFilePhase2(ipath, otpath);
      fs.delete(ipath, false);
      if (res) {
        fs.rename(otpath, ipath); // expanded file with .pending ext
        LogInfo("Pending fids after phase2 in file = " + ipath.toString());
      } else fs.rename(otpath, opath);
    }
  }

  boolean ExpandAuditLogFilePhase2(Path ipath, Path opath) throws Exception
  {
    FSDataInputStream in = fs.open(ipath);
    BufferedReader reader = new BufferedReader(new InputStreamReader(in));
    FSDataOutputStream out = fs.create(opath);
    BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out));  

    String line;
    boolean pending = false;
    JSONParser jsonParser = new JSONParser();
    while (true) {
      if ((line = reader.readLine()) == null) break; //eof
      if (ExpandAuditLogPhase2(writer, jsonParser, line)) pending = true;
    }
    reader.close(); in.close(); writer.close(); out.close();
    return pending;
  }

  boolean ExpandAuditLogPhase2(BufferedWriter writer,
      JSONParser parser, String line) throws Exception
  {
    boolean pending = false;

    long beginTime = System.currentTimeMillis(); 

    LinkedHashMap ijsonObj = null;
    try {
      ijsonObj = (LinkedHashMap)parser.parse(line, orderedKeyFactory);
    } catch (ParseException e) {
       LOG.info("Phase2: JsonParse Exception Line = " + line + " position = " +
                 e.getPosition());
       throw e;
    }

    long endTime = System.currentTimeMillis(); 
    if (endTime > beginTime)
      parsingTime = parsingTime + endTime - beginTime;

    LinkedHashMap ojsonObj = new LinkedHashMap();
    Iterator iter = ijsonObj.entrySet().iterator();
    while(iter.hasNext()) {
      Map.Entry entry = (Map.Entry)iter.next();
      String key = entry.getKey().toString();
      String value = entry.getValue().toString();

      // Phase1 retains the original Fid fields, hence the below check
      if (key.endsWith("Fid") &&
          !ijsonObj.containsKey(key.toString().replace("Fid","Path"))) {
        String fidExpD = deleteRecordHandling.get(value);
        String fidExpC= createRecordHandling.GetFullPathForFid(value);
        String fidExp = null;
        if (fidExpD == null) fidExp = fidExpC;
        else if (fidExpC == null) fidExp = fidExpD;
        else if (fidExpD.length() > fidExpC.length()) fidExp = fidExpD;
        else fidExp = fidExpC;

        if (fidExp != null) {
          ojsonObj.put(key.toString().replace("Fid", "Path"), fidExp);
          ojsonObj.put(key, value); // Retain original Fid field
        }
        else {
          fidExp = "UnResolved Path";
          pending = true;
          ojsonObj.put(key.toString().replace("Fid", "Path"), fidExp);
          ojsonObj.put(key, value);
          LOG.error("Path resolution failed for Fid " + value);
        }
      } else  ojsonObj.put(key, entry.getValue());
    }
    String expandedLog = JSONValue.toJSONString(ojsonObj).replace("\\", "");
    writer.write(expandedLog + "\n");
    return pending;
  }

  // Runs the shell command,and return stdout first line only
  String ExecuteCommand(String command) {

    StringBuffer output = new StringBuffer();
    Process p;
    try {
      p = Runtime.getRuntime().exec(command);
      int status = p.waitFor();
      BufferedReader reader =
        new BufferedReader(new InputStreamReader(p.getInputStream()));
      String line = "";
      if (status == 0) {
        if ((line  = reader.readLine())!= null) output.append(line);
      } else LOG.error("execute Command = " + command + " status = " + status);
    }
    catch (Exception e) {
      LOG.error("ExecuteCommand Exception", e);
    }
    return output.toString().trim();
  }

  // Executor class for phase1 expand
  class ExpandAuditLogNodePhase1ExecutorService implements Callable<Void> {

    ExpandAuditLogCluster expandAuditLogCluster;
    Path auditPath;
    String nodeName;

    ExpandAuditLogNodePhase1ExecutorService(
      ExpandAuditLogCluster expandAuditLogCluster, String nodeName,
      Path auditPath)
    {
      this.expandAuditLogCluster = expandAuditLogCluster;
      this.nodeName = nodeName;
      this.auditPath = auditPath;
    }

    public Void call() throws Exception {

      expandAuditLogCluster.ExpandAuditLogNodePhase1(auditPath, nodeName);
      return null; // executor requirement to return
    }
  }

  // Executor class for phase2 expand
  class ExpandAuditLogNodePhase2ExecutorService implements Callable<Void> {

    ExpandAuditLogCluster expandAuditLogCluster;
    Path nodePath;

    ExpandAuditLogNodePhase2ExecutorService(
      ExpandAuditLogCluster expandAuditLogCluster, Path nodePath)
    {
      this.expandAuditLogCluster = expandAuditLogCluster;
      this.nodePath = nodePath;
    }

    public Void call() throws Exception {

      expandAuditLogCluster.ExpandAuditLogNodePhase2(nodePath);
      return null; // executor requirement to return
    }
  }

} // AuditExpandCluster
