/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.audit;

import java.util.*;
import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import com.mapr.fs.MapRFileSystem;
import org.apache.log4j.Logger;

class DeleteRecordHandling {

  // resolved fid entries based on delete audit log processing
  private ConcurrentHashMap<String, String> deleteMap =
    new ConcurrentHashMap<String, String>();

  // Unresolved child fid entry
  private class UnresolvedEntry {
    String cFid;
    String name;
    UnresolvedEntry(String cFid, String name)
    {
      this.cFid = cFid;
      this.name = name;
    }
  }

  // Map of unresolved delete entries - 
  // key - parent Fid
  // value - list of unresolved child fid entries
  private ConcurrentHashMap<String, LinkedList<UnresolvedEntry>> unresolvedMap =
    new ConcurrentHashMap<String, LinkedList<UnresolvedEntry>>();

  class ResolvedFid {
    String fid;
    String path;
    ResolvedFid(String fid, String path)
    {
      this.fid = fid;
      this.path = path;
    }
  }

  private ConcurrentFIFOHashMap<String, String> fidMap;
  private MapRFileSystem fs;

  private static final Logger LOG = Logger.getLogger(DeleteRecordHandling.class);

  void LogErrorAndContinue(String string)
  {
    LOG.error(string);
    System.out.println(string);
  }

  DeleteRecordHandling(ConcurrentFIFOHashMap fidMap, MapRFileSystem fs)
  {
    this.fidMap = fidMap;
    this.fs = fs;
  }

  // can be invoked from multiple threads
  public void ProcessDeleteRecord(String pFid, String cFid, String name,
    String ipathStr) throws Exception
  {
    String expPFid;
    if (fidMap.get(pFid) != null) expPFid = fidMap.get(pFid).toString();
    else if (deleteMap.get(pFid) != null) // filled by another thread
      expPFid = deleteMap.get(pFid).toString();
    else {
      try {
        expPFid = fs.getMountPathFid(pFid);
      } catch (IOException e) {
        LogErrorAndContinue("getMountPath failed for invalid fid = " + pFid +
          " in log file = " + ipathStr);
        return;
      }
      if (expPFid != null) fidMap.put(pFid, expPFid);
    }

    if (expPFid != null) {
      String expCFid;
      if (expPFid.charAt(expPFid.length() -1) != '/')
        expCFid = expPFid + "/" + name;
      else expCFid = expPFid + name;
      // Above resolution of cFid expansion happens on one node / thread only
      deleteMap.put(cFid, expCFid);

      // Resolve all child fids for which above cFid is parent fid
      LinkedList<ResolvedFid> resolvedFids = new LinkedList<ResolvedFid>();
      resolvedFids.add(new ResolvedFid(cFid, expCFid));

      while (resolvedFids.size() != 0) {
        ResolvedFid tmpFid = resolvedFids.removeFirst();
        String fslash = "";
        if (tmpFid.path.charAt(tmpFid.path.length() - 1) != '/') fslash = "/";
        synchronized (this) {
          LinkedList<UnresolvedEntry> list = unresolvedMap.get(tmpFid.fid);
          if (list == null) continue;
          Iterator<UnresolvedEntry> itr = list.iterator();
          while(itr.hasNext()){
            UnresolvedEntry uentry = itr.next();
            expCFid = tmpFid.path + fslash + uentry.name;
            deleteMap.putIfAbsent(uentry.cFid, expCFid);
            resolvedFids.add(new ResolvedFid(uentry.cFid, expCFid));
          }
        }
        unresolvedMap.remove(tmpFid.fid);
      }
    } else {
      synchronized (this) {
        LinkedList<UnresolvedEntry> list = unresolvedMap.get(pFid);
        if (list == null) {
          list = new LinkedList<UnresolvedEntry>();
          LinkedList<UnresolvedEntry> tmp =
            unresolvedMap.putIfAbsent(pFid, list);
          if (tmp != null) list = tmp;
        }
        list.add(new UnresolvedEntry(cFid, name));
      }
    }
  }

  public String get(String fid)
  {
    return deleteMap.get(fid);
  }
}
