/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.audit;

import java.util.*;
import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import com.mapr.fs.MapRFileSystem;
import org.apache.log4j.Logger;

class CreateRecordHandling {

  // record for each 
  class CreateRecord {
    String fid;
    String name;
    String fullPath;
    CreateRecord parent;
    CreateRecord(String fid)
    {
      name = null;
      fullPath = null;
      parent = null;
      this.fid = fid;
    }
  }

  private ConcurrentHashMap<String, CreateRecord> createMap =
    new ConcurrentHashMap<String, CreateRecord>();

  private static final Logger LOG = Logger.getLogger(CreateRecordHandling.class);

  private MapRFileSystem fs;

  CreateRecordHandling(MapRFileSystem fs)
  {
    this.fs = fs;
  }

  // invoked in phase1 only, multi threaded
  public void ProcessCreateRecord(String pFid, String cFid, String name)
  {
    CreateRecord childcr = createMap.get(cFid);
    if (childcr == null) {
      childcr = new CreateRecord(cFid);
      CreateRecord tmpcr = createMap.get(cFid);
      if (tmpcr != null) childcr = tmpcr;
    }
    childcr.name = name;
    createMap.put(cFid, childcr);

    // get of pFid in one thread may clash with another thread
    // where cFid == pFid
    CreateRecord parentcr = createMap.get(pFid);
    if (parentcr == null) {
      parentcr =  new CreateRecord(pFid);
      CreateRecord tmp = createMap.putIfAbsent(pFid, parentcr);
      if (tmp != null) parentcr = tmp;
    }
    childcr.parent = parentcr;
  }

  // invoked single threaded
  public void ResolveAllFids() throws Exception
  {
    for (Map.Entry<String, CreateRecord> entry:createMap.entrySet()) {
      String fid = entry.getKey();
      CreateRecord cr = entry.getValue();
      if (cr.fullPath != null) continue;
      StringBuilder sb = new StringBuilder(32);
      GetFullPath(sb, cr, fid);
    }
  }

  private void GetFullPath(StringBuilder sb, CreateRecord cr, String fid)
    throws Exception
  {
    if (cr.fullPath != null) {
      sb.append(cr.fullPath);
      return;
    }

    if (cr.parent != null) GetFullPath(sb, cr.parent, cr.parent.fid);
    else { // top level, name will be null
      String expandedFid;
      try {
        expandedFid = fs.getMountPathFid(cr.fid);
      } catch (IOException e) {
        LOG.error("getMountPath failed for invalid fid = " + cr.fid);
        expandedFid = null;
      }
      if (expandedFid != null) sb.append(expandedFid);
      else if (fid.endsWith("16.2")) sb.append("<VolumeRootFid:" + fid + ">");
      else sb.append("<UnresolvedFid:" + fid + ">");
      return;
    }

    if ((sb.toString().length() > 0) &&
       (sb.toString().charAt(sb.toString().length() - 1) != '/'))
      sb.append("/");
    sb.append(cr.name);
    cr.fullPath = sb.toString();
  }

  // Invoked in Phase2 only
  public String GetFullPathForFid(String fid)
  {
    CreateRecord cr = createMap.get(fid);
    if (cr == null) return null; else return cr.fullPath;
  }
}
