/*
 * Decompiled with CFR 0.152.
 */
package tachyon.worker;

import com.google.common.base.Throwables;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tachyon.Constants;
import tachyon.Pair;
import tachyon.StorageDirId;
import tachyon.StorageLevelAlias;
import tachyon.UnderFileSystem;
import tachyon.Users;
import tachyon.conf.CommonConf;
import tachyon.conf.WorkerConf;
import tachyon.master.MasterClient;
import tachyon.thrift.BlockInfoException;
import tachyon.thrift.ClientFileInfo;
import tachyon.thrift.Command;
import tachyon.thrift.FailedToCheckpointException;
import tachyon.thrift.FileAlreadyExistException;
import tachyon.thrift.FileDoesNotExistException;
import tachyon.thrift.NetAddress;
import tachyon.thrift.OutOfSpaceException;
import tachyon.thrift.SuspectedFileSizeException;
import tachyon.util.CommonUtils;
import tachyon.util.ThreadFactoryUtils;
import tachyon.worker.hierarchy.StorageDir;
import tachyon.worker.hierarchy.StorageTier;

public class WorkerStorage {
    private static final Logger LOG = LoggerFactory.getLogger((String)Constants.LOGGER_TYPE);
    private final CommonConf mCommonConf;
    private volatile MasterClient mMasterClient;
    private final InetSocketAddress mMasterAddress;
    private NetAddress mWorkerAddress;
    private long mWorkerId;
    private final String mDataFolder;
    private final String mUserFolder;
    private String mUfsWorkerFolder;
    private String mUfsWorkerDataFolder;
    private String mUfsOrphansFolder;
    private UnderFileSystem mUfs;
    private Users mUsers;
    private final Object mDependencyLock = new Object();
    private final Set<Integer> mUncheckpointFiles = new HashSet<Integer>();
    private final Map<Integer, Set<Integer>> mDepIdToFiles = new HashMap<Integer, Set<Integer>>();
    private List<Integer> mPriorityDependencies = new ArrayList<Integer>();
    private final ExecutorService mCheckpointExecutor;
    private final ExecutorService mExecutorService;
    private long mCapacityBytes;
    private ArrayList<StorageTier> mStorageTiers;
    private final BlockingQueue<Long> mRemovedBlockIdList;
    private final Map<Pair<Long, Long>, StorageDir> mTempBlockLocation;
    private final Multimap<Long, Long> mUserIdToTempBlockIds;

    public WorkerStorage(InetSocketAddress masterAddress, ExecutorService executorService) {
        this.mCheckpointExecutor = Executors.newFixedThreadPool(WorkerConf.get().WORKER_CHECKPOINT_THREADS, ThreadFactoryUtils.build("checkpoint-%d"));
        this.mRemovedBlockIdList = new ArrayBlockingQueue<Long>(10000);
        this.mTempBlockLocation = Collections.synchronizedMap(new HashMap());
        this.mUserIdToTempBlockIds = Multimaps.synchronizedMultimap((Multimap)HashMultimap.create());
        this.mExecutorService = executorService;
        this.mCommonConf = CommonConf.get();
        this.mMasterAddress = masterAddress;
        this.mMasterClient = new MasterClient(this.mMasterAddress, this.mExecutorService);
        this.mDataFolder = WorkerConf.get().DATA_FOLDER;
        this.mUserFolder = CommonUtils.concat(this.mDataFolder, "users");
    }

    public void initialize(NetAddress address) {
        this.mWorkerAddress = address;
        try {
            this.initializeStorageTier();
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        this.register();
        this.mUfsWorkerFolder = CommonUtils.concat(this.mCommonConf.UNDERFS_WORKERS_FOLDER, this.mWorkerId);
        this.mUfsWorkerDataFolder = this.mUfsWorkerFolder + "/data";
        this.mUfs = UnderFileSystem.get(this.mCommonConf.UNDERFS_ADDRESS);
        this.mUsers = new Users(this.mUfsWorkerFolder);
        for (int k = 0; k < WorkerConf.get().WORKER_CHECKPOINT_THREADS; ++k) {
            this.mCheckpointExecutor.submit(new CheckpointThread(k));
        }
        try {
            this.addFoundBlocks();
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        catch (SuspectedFileSizeException e) {
            throw Throwables.propagate((Throwable)e);
        }
        catch (BlockInfoException e) {
            throw Throwables.propagate((Throwable)e);
        }
        LOG.info("Current Worker ID: {}, mWorkerAddress: {}, CapacityBytes: {}", new Object[]{this.mWorkerId, this.mWorkerAddress, this.mCapacityBytes});
    }

    void accessBlock(long blockId) {
        StorageDir foundDir = this.getStorageDirByBlockId(blockId);
        if (foundDir != null) {
            foundDir.accessBlock(blockId);
        }
    }

    public void addCheckpoint(long userId, int fileId) throws FileDoesNotExistException, SuspectedFileSizeException, FailedToCheckpointException, BlockInfoException, IOException {
        long fileSize;
        String srcPath = CommonUtils.concat(this.getUserUfsTempFolder(userId), fileId);
        String dstPath = CommonUtils.concat(this.mCommonConf.UNDERFS_DATA_FOLDER, fileId);
        try {
            if (!this.mUfs.rename(srcPath, dstPath)) {
                throw new FailedToCheckpointException("Failed to rename " + srcPath + " to " + dstPath);
            }
        }
        catch (IOException e) {
            throw new FailedToCheckpointException("Failed to rename " + srcPath + " to " + dstPath);
        }
        try {
            fileSize = this.mUfs.getFileSize(dstPath);
        }
        catch (IOException e) {
            throw new FailedToCheckpointException("Failed to getFileSize " + dstPath);
        }
        this.mMasterClient.addCheckpoint(this.mWorkerId, fileId, fileSize, dstPath);
    }

    private void addFoundBlocks() throws IOException, SuspectedFileSizeException, BlockInfoException {
        this.mUfsOrphansFolder = this.mUfsWorkerFolder + "/orphans";
        if (!this.mUfs.exists(this.mUfsOrphansFolder)) {
            this.mUfs.mkdirs(this.mUfsOrphansFolder, true);
        }
        for (StorageTier curStorageTier : this.mStorageTiers) {
            for (StorageDir curStorageDir : curStorageTier.getStorageDirs()) {
                for (Map.Entry<Long, Long> blockSize : curStorageDir.getBlockSizes()) {
                    try {
                        this.mMasterClient.worker_cacheBlock(this.mWorkerId, this.getUsedBytes(), curStorageDir.getStorageDirId(), blockSize.getKey(), blockSize.getValue());
                    }
                    catch (FileDoesNotExistException e) {
                        LOG.error("Block not exist in metadata! blockId:{}", (Object)blockSize.getKey());
                        this.swapoutOrphanBlocks(curStorageDir, blockSize.getKey());
                        this.freeBlock(blockSize.getKey());
                    }
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean asyncCheckpoint(int fileId) throws IOException {
        ClientFileInfo fileInfo = this.mMasterClient.getFileStatus(fileId, "");
        if (fileInfo.getDependencyId() != -1) {
            Object object = this.mDependencyLock;
            synchronized (object) {
                this.mUncheckpointFiles.add(fileId);
                if (!this.mDepIdToFiles.containsKey(fileInfo.getDependencyId())) {
                    this.mDepIdToFiles.put(fileInfo.getDependencyId(), new HashSet());
                }
                this.mDepIdToFiles.get(fileInfo.getDependencyId()).add(fileId);
            }
            return true;
        }
        return false;
    }

    public void cacheBlock(long userId, long blockId) throws FileDoesNotExistException, SuspectedFileSizeException, BlockInfoException, IOException {
        StorageDir storageDir = this.mTempBlockLocation.remove(new Pair<Long, Long>(userId, blockId));
        if (storageDir == null) {
            throw new FileDoesNotExistException("Block doesn't exist! blockId:" + blockId);
        }
        this.mUserIdToTempBlockIds.remove((Object)userId, (Object)blockId);
        boolean result = false;
        try {
            result = storageDir.cacheBlock(userId, blockId);
        }
        catch (IOException e) {
            throw new FileDoesNotExistException("Failed to cache block! blockId:" + blockId);
        }
        if (result) {
            long blockSize = storageDir.getBlockSize(blockId);
            this.mMasterClient.worker_cacheBlock(this.mWorkerId, this.getUsedBytes(), storageDir.getStorageDirId(), blockId, blockSize);
        }
    }

    public void cancelBlock(long userId, long blockId) {
        StorageDir storageDir = this.mTempBlockLocation.remove(new Pair<Long, Long>(userId, blockId));
        if (storageDir != null) {
            this.mUserIdToTempBlockIds.remove((Object)userId, (Object)blockId);
            try {
                storageDir.cancelBlock(userId, blockId);
            }
            catch (IOException e) {
                LOG.error("Failed to cancel block! blockId:{}", (Object)blockId);
            }
        }
    }

    public void checkStatus() {
        List<Long> removedUsers = this.mUsers.checkStatus();
        for (long userId : removedUsers) {
            Collection tempBlockIdList = this.mUserIdToTempBlockIds.removeAll((Object)userId);
            for (Long blockId : tempBlockIdList) {
                this.mTempBlockLocation.remove(new Pair<Long, Long>(userId, blockId));
            }
            for (StorageTier storageTier : this.mStorageTiers) {
                for (StorageDir storageDir : storageTier.getStorageDirs()) {
                    storageDir.cleanUserResources(userId, tempBlockIdList);
                }
            }
            this.mUsers.removeUser(userId);
        }
    }

    private void freeBlock(long blockId) {
        for (StorageTier storageTier : this.mStorageTiers) {
            for (StorageDir storageDir : storageTier.getStorageDirs()) {
                if (!storageDir.containsBlock(blockId)) continue;
                try {
                    storageDir.deleteBlock(blockId);
                }
                catch (IOException e) {
                    LOG.error("Failed to delete block file! blockId:{}", (Object)blockId);
                }
            }
        }
        this.mRemovedBlockIdList.add(blockId);
    }

    public void freeBlocks(List<Long> blockIds) {
        for (long blockId : blockIds) {
            this.freeBlock(blockId);
        }
    }

    public StorageDir getStorageDirByBlockId(long blockId) {
        StorageDir storageDir = null;
        for (StorageTier storageTier : this.mStorageTiers) {
            storageDir = storageTier.getStorageDirByBlockId(blockId);
            if (storageDir == null) continue;
            return storageDir;
        }
        return null;
    }

    public String getUfsOrphansFolder() {
        return this.mUfsOrphansFolder;
    }

    private long getUsedBytes() {
        long usedBytes = 0L;
        for (StorageTier curTier : this.mStorageTiers) {
            usedBytes += curTier.getUsedBytes();
        }
        return usedBytes;
    }

    public String getUserUfsTempFolder(long userId) {
        String ret = this.mUsers.getUserUfsTempFolder(userId);
        LOG.info("Return UserHdfsTempFolder for " + userId + " : " + ret);
        return ret;
    }

    public Command heartbeat() throws IOException {
        ArrayList<Long> removedBlockIds = new ArrayList<Long>();
        HashMap<Long, List<Long>> addedBlockIds = new HashMap<Long, List<Long>>();
        this.mRemovedBlockIdList.drainTo(removedBlockIds);
        for (StorageTier storageTier : this.mStorageTiers) {
            for (StorageDir storageDir : storageTier.getStorageDirs()) {
                addedBlockIds.put(storageDir.getStorageDirId(), storageDir.getAddedBlockIdList());
            }
        }
        return this.mMasterClient.worker_heartbeat(this.mWorkerId, this.getUsedBytes(), removedBlockIds, addedBlockIds);
    }

    public void initializeStorageTier() throws IOException {
        this.mStorageTiers = new ArrayList(WorkerConf.get().STORAGE_LEVELS);
        for (int k = 0; k < WorkerConf.get().STORAGE_LEVELS; ++k) {
            this.mStorageTiers.add(null);
        }
        StorageTier nextStorageTier = null;
        for (int level = WorkerConf.get().STORAGE_LEVELS - 1; level >= 0; --level) {
            if (WorkerConf.get().STORAGE_TIER_DIRS[level] == null) {
                throw new IOException("No directory path is set for layer " + level);
            }
            if (WorkerConf.get().STORAGE_TIER_DIR_QUOTA[level] == null) {
                throw new IOException("No directory quota is set for layer " + level);
            }
            String[] dirPaths = WorkerConf.get().STORAGE_TIER_DIRS[level].split(",");
            for (int i = 0; i < dirPaths.length; ++i) {
                dirPaths[i] = dirPaths[i].trim();
            }
            StorageLevelAlias alias = WorkerConf.get().STORAGE_LEVEL_ALIAS[level];
            String[] dirCapacityStrings = WorkerConf.get().STORAGE_TIER_DIR_QUOTA[level].split(",");
            long[] dirCapacities = new long[dirPaths.length];
            int j = 0;
            for (int i = 0; i < dirPaths.length; ++i) {
                dirCapacities[i] = CommonUtils.parseSpaceSize(dirCapacityStrings[j].trim());
                if (j >= dirCapacityStrings.length - 1) continue;
                ++j;
            }
            StorageTier curTier = new StorageTier(level, alias, dirPaths, dirCapacities, this.mDataFolder, this.mUserFolder, nextStorageTier, null);
            curTier.initialize();
            this.mCapacityBytes += curTier.getCapacityBytes();
            this.mStorageTiers.set(level, curTier);
            nextStorageTier = curTier;
        }
    }

    public StorageDir lockBlock(long blockId, long userId) {
        for (StorageTier tier : this.mStorageTiers) {
            for (StorageDir dir : tier.getStorageDirs()) {
                if (!dir.lockBlock(blockId, userId)) continue;
                return dir;
            }
        }
        LOG.warn("Failed to lock block! blockId:{}", (Object)blockId);
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean promoteBlock(long blockId) {
        long userId = -3L;
        StorageDir storageDir = this.lockBlock(blockId, -3L);
        if (storageDir == null) {
            return false;
        }
        if (StorageDirId.getStorageLevelAliasValue(storageDir.getStorageDirId()) != this.mStorageTiers.get(0).getAlias().getValue()) {
            long blockSize = storageDir.getBlockSize(blockId);
            StorageDir dstStorageDir = this.requestSpace(null, -3L, blockSize);
            if (dstStorageDir == null) {
                LOG.error("Failed to promote block! blockId:{}", (Object)blockId);
                storageDir.unlockBlock(blockId, -3L);
                return false;
            }
            boolean result = false;
            try {
                try {
                    result = storageDir.copyBlock(blockId, dstStorageDir);
                }
                finally {
                    storageDir.unlockBlock(blockId, -3L);
                }
                if (result) {
                    storageDir.deleteBlock(blockId);
                }
                return result;
            }
            catch (IOException e) {
                LOG.error("Failed to promote block! blockId:{}", (Object)blockId);
                return false;
            }
        }
        this.unlockBlock(blockId, -3L);
        return true;
    }

    public void register() {
        long id = 0L;
        HashMap<Long, List<Long>> blockIdLists = new HashMap<Long, List<Long>>();
        for (StorageTier curStorageTier : this.mStorageTiers) {
            for (StorageDir curStorageDir : curStorageTier.getStorageDirs()) {
                Set<Long> blockSet = curStorageDir.getBlockIds();
                blockIdLists.put(curStorageDir.getStorageDirId(), new ArrayList<Long>(blockSet));
            }
        }
        while (id == 0L) {
            try {
                id = this.mMasterClient.worker_register(this.mWorkerAddress, this.mCapacityBytes, this.getUsedBytes(), blockIdLists);
            }
            catch (BlockInfoException e) {
                LOG.error(e.getMessage(), (Throwable)e);
                id = 0L;
                CommonUtils.sleepMs(LOG, 1000L);
            }
            catch (IOException e) {
                LOG.error(e.getMessage(), (Throwable)e);
                id = 0L;
                CommonUtils.sleepMs(LOG, 1000L);
            }
        }
        this.mWorkerId = id;
    }

    public String requestBlockLocation(long userId, long blockId, long initialBytes) throws OutOfSpaceException, FileAlreadyExistException {
        if (this.mTempBlockLocation.containsKey(new Pair<Long, Long>(userId, blockId))) {
            throw new FileAlreadyExistException(String.format("Block file is being written! userId(%d) blockId(%d)", userId, blockId));
        }
        StorageDir storageDir = this.requestSpace(null, userId, initialBytes);
        if (storageDir == null) {
            throw new OutOfSpaceException(String.format("Failed to allocate space for block! blockId(%d) sizeBytes(%d)", blockId, initialBytes));
        }
        this.mTempBlockLocation.put(new Pair<Long, Long>(userId, blockId), storageDir);
        this.mUserIdToTempBlockIds.put((Object)userId, (Object)blockId);
        storageDir.updateTempBlockAllocatedBytes(userId, blockId, initialBytes);
        return storageDir.getUserTempFilePath(userId, blockId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private StorageDir requestSpace(StorageDir dirCandidate, long userId, long requestBytes) {
        Set<Integer> pinList;
        try {
            pinList = this.mMasterClient.worker_getPinIdList();
        }
        catch (IOException e) {
            LOG.error(e.getMessage());
            pinList = new HashSet<Integer>();
        }
        StorageDir dir = null;
        ArrayList<Long> removedBlockIds = new ArrayList<Long>();
        try {
            if (dirCandidate == null) {
                dir = this.mStorageTiers.get(0).requestSpace(userId, requestBytes, pinList, removedBlockIds);
            } else if (this.mStorageTiers.get(0).requestSpace(dirCandidate, userId, requestBytes, pinList, removedBlockIds)) {
                dir = dirCandidate;
            }
        }
        catch (IOException e) {
            LOG.error(e.getMessage());
        }
        finally {
            if (removedBlockIds.size() > 0) {
                this.mRemovedBlockIdList.addAll(removedBlockIds);
            }
        }
        return dir;
    }

    public boolean requestSpace(long userId, long blockId, long requestBytes) throws FileDoesNotExistException {
        StorageDir storageDir = this.mTempBlockLocation.get(new Pair<Long, Long>(userId, blockId));
        if (storageDir == null) {
            throw new FileDoesNotExistException("Temporary block file doesn't exist! blockId:" + blockId);
        }
        if (storageDir == this.requestSpace(storageDir, userId, requestBytes)) {
            storageDir.updateTempBlockAllocatedBytes(userId, blockId, requestBytes);
            return true;
        }
        return false;
    }

    public void resetMasterClient() {
        MasterClient tMasterClient;
        this.mMasterClient = tMasterClient = new MasterClient(this.mMasterAddress, this.mExecutorService);
    }

    public void stop() {
        this.mMasterClient.shutdown();
        this.mCheckpointExecutor.shutdownNow();
        try {
            this.mCheckpointExecutor.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void swapoutOrphanBlocks(StorageDir storageDir, long blockId) throws IOException {
        ByteBuffer buf = storageDir.getBlockData(blockId, 0L, -1);
        String ufsOrphanBlock = CommonUtils.concat(this.mUfsOrphansFolder, blockId);
        OutputStream os = this.mUfs.create(ufsOrphanBlock);
        int bulkSize = 65536;
        byte[] bulk = new byte[65536];
        try {
            for (int k = 0; k < (buf.limit() + 65536 - 1) / 65536; ++k) {
                int len = 65536 < buf.remaining() ? 65536 : buf.remaining();
                buf.get(bulk, 0, len);
                os.write(bulk, 0, len);
            }
        }
        finally {
            os.close();
            CommonUtils.cleanDirectBuffer(buf);
        }
    }

    public boolean unlockBlock(long blockId, long userId) {
        for (StorageTier tier : this.mStorageTiers) {
            for (StorageDir dir : tier.getStorageDirs()) {
                if (!dir.unlockBlock(blockId, userId)) continue;
                return true;
            }
        }
        LOG.warn("Failed to unlock block! blockId:{}", (Object)blockId);
        return false;
    }

    public void userHeartbeat(long userId) {
        this.mUsers.userHeartbeat(userId);
    }

    public class CheckpointThread
    implements Runnable {
        private final int mId;
        private UnderFileSystem mCheckpointUfs = null;

        public CheckpointThread(int id) {
            this.mId = id;
        }

        private int getFileIdBasedOnPriorityDependency() {
            if (WorkerStorage.this.mPriorityDependencies.isEmpty()) {
                return -1;
            }
            Iterator i$ = WorkerStorage.this.mPriorityDependencies.iterator();
            while (i$.hasNext()) {
                int depId = (Integer)i$.next();
                int fileId = this.getFileIdFromOneDependency(depId);
                if (fileId == -1) continue;
                return fileId;
            }
            return -1;
        }

        private int getFileIdFromOneDependency(int depId) {
            Set fileIds = (Set)WorkerStorage.this.mDepIdToFiles.get(depId);
            if (fileIds != null && !fileIds.isEmpty()) {
                int fileId = (Integer)fileIds.iterator().next();
                fileIds.remove(fileId);
                WorkerStorage.this.mUncheckpointFiles.remove(fileId);
                if (fileIds.isEmpty()) {
                    WorkerStorage.this.mDepIdToFiles.remove(depId);
                }
                return fileId;
            }
            return -1;
        }

        private int getRandomUncheckpointedFile() {
            if (WorkerStorage.this.mUncheckpointFiles.isEmpty()) {
                return -1;
            }
            Iterator i$ = WorkerStorage.this.mDepIdToFiles.keySet().iterator();
            while (i$.hasNext()) {
                int depId = (Integer)i$.next();
                int fileId = this.getFileIdFromOneDependency(depId);
                if (fileId == -1) continue;
                return fileId;
            }
            return -1;
        }

        private List<Integer> getSortedPriorityDependencyList() throws IOException {
            List<Integer> ret = WorkerStorage.this.mMasterClient.worker_getPriorityDependencyList();
            for (int i = 0; i < ret.size(); ++i) {
                for (int j = i + 1; j < ret.size(); ++j) {
                    if (ret.get(i) >= ret.get(j)) continue;
                    int k = ret.get(i);
                    ret.set(i, ret.get(j));
                    ret.set(j, k);
                }
            }
            return ret;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    int k;
                    int fileId = -1;
                    Object object = WorkerStorage.this.mDependencyLock;
                    synchronized (object) {
                        fileId = this.getFileIdBasedOnPriorityDependency();
                        if (fileId == -1) {
                            if (WorkerStorage.this.mPriorityDependencies.size() == 0) {
                                WorkerStorage.this.mPriorityDependencies = this.getSortedPriorityDependencyList();
                                if (!WorkerStorage.this.mPriorityDependencies.isEmpty()) {
                                    LOG.info("Get new mPriorityDependencies " + CommonUtils.listToString(WorkerStorage.this.mPriorityDependencies));
                                }
                            } else {
                                List<Integer> tList = this.getSortedPriorityDependencyList();
                                boolean equal = true;
                                if (WorkerStorage.this.mPriorityDependencies.size() != tList.size()) {
                                    equal = false;
                                }
                                if (equal) {
                                    for (int k2 = 0; k2 < tList.size(); ++k2) {
                                        if (tList.get(k2) == WorkerStorage.this.mPriorityDependencies.get(k2)) continue;
                                        equal = false;
                                        break;
                                    }
                                }
                                if (!equal) {
                                    WorkerStorage.this.mPriorityDependencies = tList;
                                }
                            }
                            fileId = this.getFileIdBasedOnPriorityDependency();
                        }
                        if (fileId == -1) {
                            fileId = this.getRandomUncheckpointedFile();
                        }
                    }
                    if (fileId == -1) {
                        LOG.debug("Thread {} has nothing to checkpoint. Sleep for 1 sec.", (Object)this.mId);
                        CommonUtils.sleepMs(LOG, 1000L);
                        continue;
                    }
                    String midPath = CommonUtils.concat(WorkerStorage.this.mUfsWorkerDataFolder, fileId);
                    String dstPath = CommonUtils.concat(CommonConf.get().UNDERFS_DATA_FOLDER, fileId);
                    LOG.info("Thread {} is checkpointing file {}. midPath: {} dsPath: {}", new Object[]{this.mId, fileId, midPath, dstPath});
                    if (this.mCheckpointUfs == null) {
                        this.mCheckpointUfs = UnderFileSystem.get(midPath);
                    }
                    long startCopyTimeMs = System.currentTimeMillis();
                    ClientFileInfo fileInfo = WorkerStorage.this.mMasterClient.getFileStatus(fileId, "");
                    if (!fileInfo.isComplete) {
                        LOG.error("File {} is not complete!", (Object)fileInfo);
                        continue;
                    }
                    StorageDir[] storageDirs = new StorageDir[fileInfo.blockIds.size()];
                    OutputStream os = null;
                    long fileSizeByte = 0L;
                    try {
                        for (k = 0; k < fileInfo.blockIds.size(); ++k) {
                            long blockId = fileInfo.blockIds.get(k);
                            storageDirs[k] = WorkerStorage.this.lockBlock(blockId, -2L);
                            if (storageDirs[k] != null) continue;
                            throw new IOException("Block doesn't exist! blockId:" + blockId);
                        }
                        os = this.mCheckpointUfs.create(midPath, (int)fileInfo.getBlockSizeByte());
                        for (k = 0; k < fileInfo.blockIds.size(); ++k) {
                            ByteBuffer byteBuffer = storageDirs[k].getBlockData(fileInfo.blockIds.get(k), 0L, -1);
                            byte[] buf = new byte[16384];
                            while (byteBuffer.remaining() > 0) {
                                int writeLen = byteBuffer.remaining() >= buf.length ? buf.length : byteBuffer.remaining();
                                byteBuffer.get(buf, 0, writeLen);
                                os.write(buf, 0, writeLen);
                            }
                            CommonUtils.cleanDirectBuffer(byteBuffer);
                        }
                    }
                    finally {
                        for (k = 0; k < fileInfo.blockIds.size(); ++k) {
                            if (storageDirs[k] == null) continue;
                            storageDirs[k].unlockBlock(fileInfo.blockIds.get(k), -2L);
                        }
                        if (os != null) {
                            os.close();
                        }
                    }
                    if (!this.mCheckpointUfs.rename(midPath, dstPath)) {
                        LOG.error("Failed to rename from " + midPath + " to " + dstPath);
                    }
                    WorkerStorage.this.mMasterClient.addCheckpoint(WorkerStorage.this.mWorkerId, fileId, fileSizeByte, dstPath);
                    long shouldTakeMs = (long)(1000.0 * (double)fileSizeByte / 1048576.0 / (double)WorkerConf.get().WORKER_PER_THREAD_CHECKPOINT_CAP_MB_SEC);
                    long currentTimeMs = System.currentTimeMillis();
                    if (startCopyTimeMs + shouldTakeMs <= currentTimeMs) continue;
                    long shouldSleepMs = startCopyTimeMs + shouldTakeMs - currentTimeMs;
                    LOG.info("Checkpointed last file " + fileId + " took " + (currentTimeMs - startCopyTimeMs) + " ms. Need to sleep " + shouldSleepMs + " ms.");
                    CommonUtils.sleepMs(LOG, shouldSleepMs);
                }
                catch (IOException e) {
                    LOG.error(e.getMessage(), (Throwable)e);
                }
            }
        }
    }
}

