/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LayoutVersion;
import org.apache.hadoop.hdfs.security.token.delegation.DelegationTokenSecretManager;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.InconsistentFSStateException;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.Util;
import org.apache.hadoop.hdfs.server.namenode.CheckpointSignature;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.FSImageCompression;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormat;
import org.apache.hadoop.hdfs.server.namenode.FSImageStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.protocol.CheckpointCommand;
import org.apache.hadoop.hdfs.server.protocol.NamenodeCommand;
import org.apache.hadoop.hdfs.server.protocol.NamenodeRegistration;
import org.apache.hadoop.hdfs.util.MD5FileUtils;
import org.apache.hadoop.io.MD5Hash;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class FSImage
implements Closeable {
    protected static final Log LOG = LogFactory.getLog((String)FSImage.class.getName());
    protected FSNamesystem namesystem = null;
    protected FSEditLog editLog = null;
    private boolean isUpgradeFinalized = false;
    protected NNStorage storage;
    protected long lastAppliedTxId = 0L;
    private Collection<URI> checkpointDirs;
    private Collection<URI> checkpointEditsDirs;
    private final Configuration conf;
    private final NNStorageRetentionManager archivalManager;

    public FSImage(Configuration conf) throws IOException {
        this(conf, null);
    }

    private FSImage(Configuration conf, FSNamesystem ns) throws IOException {
        this(conf, ns, FSNamesystem.getNamespaceDirs(conf), FSNamesystem.getNamespaceEditsDirs(conf));
    }

    protected FSImage(Configuration conf, FSNamesystem ns, Collection<URI> imageDirs, Collection<URI> editsDirs) throws IOException {
        this.conf = conf;
        this.setCheckpointDirectories(FSImage.getCheckpointDirs(conf, null), FSImage.getCheckpointEditsDirs(conf, null));
        this.storage = new NNStorage(conf, imageDirs, editsDirs);
        if (conf.getBoolean("dfs.namenode.name.dir.restore", false)) {
            this.storage.setRestoreFailedStorage(true);
        }
        this.editLog = new FSEditLog(this.storage);
        this.setFSNamesystem(ns);
        this.archivalManager = new NNStorageRetentionManager(conf, this.storage, this.editLog);
    }

    protected FSNamesystem getFSNamesystem() {
        return this.namesystem;
    }

    void setFSNamesystem(FSNamesystem ns) {
        this.namesystem = ns;
        if (ns != null) {
            this.storage.setUpgradeManager(ns.upgradeManager);
        }
    }

    void setCheckpointDirectories(Collection<URI> dirs, Collection<URI> editsDirs) {
        this.checkpointDirs = dirs;
        this.checkpointEditsDirs = editsDirs;
    }

    void format(String clusterId) throws IOException {
        this.storage.format(clusterId);
        this.saveFSImageInAllDirs(0L);
    }

    boolean recoverTransitionRead(HdfsServerConstants.StartupOption startOpt) throws IOException {
        assert (startOpt != HdfsServerConstants.StartupOption.FORMAT) : "NameNode formatting should be performed before reading the image";
        Collection<URI> imageDirs = this.storage.getImageDirectories();
        Collection<URI> editsDirs = this.storage.getEditsDirectories();
        if ((imageDirs.size() == 0 || editsDirs.size() == 0) && startOpt != HdfsServerConstants.StartupOption.IMPORT) {
            throw new IOException("All specified directories are not accessible or do not exist.");
        }
        if (startOpt == HdfsServerConstants.StartupOption.IMPORT && (this.checkpointDirs == null || this.checkpointDirs.isEmpty())) {
            throw new IOException("Cannot import image from a checkpoint. \"dfs.namenode.checkpoint.dir\" is not set.");
        }
        if (startOpt == HdfsServerConstants.StartupOption.IMPORT && (this.checkpointEditsDirs == null || this.checkpointEditsDirs.isEmpty())) {
            throw new IOException("Cannot import image from a checkpoint. \"dfs.namenode.checkpoint.dir\" is not set.");
        }
        HashMap<Storage.StorageDirectory, Storage.StorageState> dataDirStates = new HashMap<Storage.StorageDirectory, Storage.StorageState>();
        boolean isFormatted = this.recoverStorageDirs(startOpt, dataDirStates);
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Data dir states:\n  " + Joiner.on((String)"\n  ").withKeyValueSeparator(": ").join(dataDirStates)));
        }
        if (!isFormatted && startOpt != HdfsServerConstants.StartupOption.ROLLBACK && startOpt != HdfsServerConstants.StartupOption.IMPORT) {
            throw new IOException("NameNode is not formatted.");
        }
        int layoutVersion = this.storage.getLayoutVersion();
        if (layoutVersion < -3) {
            NNStorage.checkVersionUpgradable(this.storage.getLayoutVersion());
        }
        if (startOpt != HdfsServerConstants.StartupOption.UPGRADE && layoutVersion < -3 && layoutVersion != HdfsConstants.LAYOUT_VERSION) {
            throw new IOException("\nFile system image contains an old layout version " + this.storage.getLayoutVersion() + ".\nAn upgrade to version " + HdfsConstants.LAYOUT_VERSION + " is required.\n" + "Please restart NameNode with -upgrade option.");
        }
        this.storage.processStartupOptionsForUpgrade(startOpt, layoutVersion);
        this.storage.verifyDistributedUpgradeProgress(startOpt);
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            Storage.StorageState curState = (Storage.StorageState)((Object)dataDirStates.get(sd));
            switch (curState) {
                case NON_EXISTENT: {
                    throw new IOException((Object)((Object)Storage.StorageState.NON_EXISTENT) + " state cannot be here");
                }
                case NOT_FORMATTED: {
                    LOG.info((Object)("Storage directory " + sd.getRoot() + " is not formatted."));
                    LOG.info((Object)"Formatting ...");
                    sd.clearDirectory();
                    break;
                }
            }
        }
        switch (startOpt) {
            case UPGRADE: {
                this.doUpgrade();
                return false;
            }
            case IMPORT: {
                this.doImportCheckpoint();
                return false;
            }
            case ROLLBACK: {
                this.doRollback();
                break;
            }
        }
        return this.loadFSImage();
    }

    private boolean recoverStorageDirs(HdfsServerConstants.StartupOption startOpt, Map<Storage.StorageDirectory, Storage.StorageState> dataDirStates) throws IOException {
        boolean isFormatted = false;
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageState curState;
            Storage.StorageDirectory sd = it.next();
            try {
                curState = sd.analyzeStorage(startOpt, this.storage);
                switch (curState) {
                    case NON_EXISTENT: {
                        throw new InconsistentFSStateException(sd.getRoot(), "storage directory does not exist or is not accessible.");
                    }
                    case NOT_FORMATTED: {
                        break;
                    }
                    case NORMAL: {
                        break;
                    }
                    default: {
                        sd.doRecover(curState);
                    }
                }
                if (curState != Storage.StorageState.NOT_FORMATTED && startOpt != HdfsServerConstants.StartupOption.ROLLBACK) {
                    this.storage.readProperties(sd);
                    isFormatted = true;
                }
                if (startOpt == HdfsServerConstants.StartupOption.IMPORT && isFormatted) {
                    throw new IOException("Cannot import image from a checkpoint.  NameNode already contains an image in " + sd.getRoot());
                }
            }
            catch (IOException ioe) {
                sd.unlock();
                throw ioe;
            }
            dataDirStates.put(sd, curState);
        }
        return isFormatted;
    }

    private void doUpgrade() throws IOException {
        Storage.StorageDirectory sd;
        if (this.storage.getDistributedUpgradeState()) {
            this.loadFSImage();
            this.storage.initializeDistributedUpgrade();
            return;
        }
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd2 = it.next();
            if (!sd2.getPreviousDir().exists()) continue;
            throw new InconsistentFSStateException(sd2.getRoot(), "previous fs state should not exist during upgrade. Finalize or rollback first.");
        }
        this.loadFSImage();
        long oldCTime = this.storage.getCTime();
        this.storage.cTime = Util.now();
        int oldLV = this.storage.getLayoutVersion();
        this.storage.layoutVersion = HdfsConstants.LAYOUT_VERSION;
        List<Storage.StorageDirectory> errorSDs = Collections.synchronizedList(new ArrayList());
        Iterator<Storage.StorageDirectory> it2 = this.storage.dirIterator();
        while (it2.hasNext()) {
            sd = it2.next();
            LOG.info((Object)("Starting upgrade of image directory " + sd.getRoot() + ".\n   old LV = " + oldLV + "; old CTime = " + oldCTime + ".\n   new LV = " + this.storage.getLayoutVersion() + "; new CTime = " + this.storage.getCTime()));
            try {
                File curDir = sd.getCurrentDir();
                File prevDir = sd.getPreviousDir();
                File tmpDir = sd.getPreviousTmp();
                assert (curDir.exists()) : "Current directory must exist.";
                assert (!prevDir.exists()) : "prvious directory must not exist.";
                assert (!tmpDir.exists()) : "prvious.tmp directory must not exist.";
                assert (!this.editLog.isOpen()) : "Edits log must not be open.";
                NNStorage.rename(curDir, tmpDir);
                if (curDir.mkdir()) continue;
                throw new IOException("Cannot create directory " + curDir);
            }
            catch (Exception e) {
                LOG.error((Object)("Failed to move aside pre-upgrade storage in image directory " + sd.getRoot()), (Throwable)e);
                errorSDs.add(sd);
            }
        }
        this.storage.reportErrorsOnDirectories(errorSDs);
        errorSDs.clear();
        this.saveFSImageInAllDirs(this.editLog.getLastWrittenTxId());
        it2 = this.storage.dirIterator();
        while (it2.hasNext()) {
            sd = it2.next();
            try {
                this.storage.writeProperties(sd);
                File prevDir = sd.getPreviousDir();
                File tmpDir = sd.getPreviousTmp();
                NNStorage.rename(tmpDir, prevDir);
            }
            catch (IOException ioe) {
                LOG.error((Object)("Unable to rename temp to previous for " + sd.getRoot()), (Throwable)ioe);
                errorSDs.add(sd);
                continue;
            }
            LOG.info((Object)("Upgrade of " + sd.getRoot() + " is complete."));
        }
        this.storage.reportErrorsOnDirectories(errorSDs);
        this.isUpgradeFinalized = false;
        if (!this.storage.getRemovedStorageDirs().isEmpty()) {
            throw new IOException("Upgrade failed in " + this.storage.getRemovedStorageDirs().size() + " storage directory(ies), previously logged.");
        }
        this.storage.initializeDistributedUpgrade();
    }

    private void doRollback() throws IOException {
        File prevDir;
        Storage.StorageDirectory sd;
        boolean canRollback = false;
        FSImage prevState = new FSImage(this.conf, this.getFSNamesystem());
        prevState.getStorage().layoutVersion = HdfsConstants.LAYOUT_VERSION;
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            sd = it.next();
            prevDir = sd.getPreviousDir();
            if (!prevDir.exists()) {
                LOG.info((Object)("Storage directory " + sd.getRoot() + " does not contain previous fs state."));
                this.storage.readProperties(sd);
                continue;
            }
            prevState.getStorage().readPreviousVersionProperties(sd);
            if (prevState.getLayoutVersion() != HdfsConstants.LAYOUT_VERSION) {
                throw new IOException("Cannot rollback to storage version " + prevState.getLayoutVersion() + " using this version of the NameNode, which uses storage version " + HdfsConstants.LAYOUT_VERSION + ". " + "Please use the previous version of HDFS to perform the rollback.");
            }
            canRollback = true;
        }
        if (!canRollback) {
            throw new IOException("Cannot rollback. None of the storage directories contain previous fs state.");
        }
        it = this.storage.dirIterator();
        while (it.hasNext()) {
            sd = it.next();
            prevDir = sd.getPreviousDir();
            if (!prevDir.exists()) continue;
            LOG.info((Object)("Rolling back storage directory " + sd.getRoot() + ".\n   new LV = " + prevState.getStorage().getLayoutVersion() + "; new CTime = " + prevState.getStorage().getCTime()));
            File tmpDir = sd.getRemovedTmp();
            assert (!tmpDir.exists()) : "removed.tmp directory must not exist.";
            File curDir = sd.getCurrentDir();
            assert (curDir.exists()) : "Current directory must exist.";
            NNStorage.rename(curDir, tmpDir);
            NNStorage.rename(prevDir, curDir);
            NNStorage.deleteDir(tmpDir);
            LOG.info((Object)("Rollback of " + sd.getRoot() + " is complete."));
        }
        this.isUpgradeFinalized = true;
        this.storage.verifyDistributedUpgradeProgress(HdfsServerConstants.StartupOption.REGULAR);
    }

    private void doFinalize(Storage.StorageDirectory sd) throws IOException {
        File prevDir = sd.getPreviousDir();
        if (!prevDir.exists()) {
            LOG.info((Object)("Directory " + prevDir + " does not exist."));
            LOG.info((Object)("Finalize upgrade for " + sd.getRoot() + " is not required."));
            return;
        }
        LOG.info((Object)("Finalizing upgrade for storage directory " + sd.getRoot() + "." + (this.storage.getLayoutVersion() == 0 ? "" : "\n   cur LV = " + this.storage.getLayoutVersion() + "; cur CTime = " + this.storage.getCTime())));
        assert (sd.getCurrentDir().exists()) : "Current directory must exist.";
        File tmpDir = sd.getFinalizedTmp();
        NNStorage.rename(prevDir, tmpDir);
        NNStorage.deleteDir(tmpDir);
        this.isUpgradeFinalized = true;
        LOG.info((Object)("Finalize upgrade for " + sd.getRoot() + " is complete."));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doImportCheckpoint() throws IOException {
        FSNamesystem fsNamesys = this.getFSNamesystem();
        FSImage ckptImage = new FSImage(this.conf, fsNamesys, this.checkpointDirs, this.checkpointEditsDirs);
        FSImage realImage = fsNamesys.getFSImage();
        assert (realImage == this);
        fsNamesys.dir.fsImage = ckptImage;
        try {
            ckptImage.recoverTransitionRead(HdfsServerConstants.StartupOption.REGULAR);
        }
        finally {
            ckptImage.close();
        }
        realImage.getStorage().setStorageInfo(ckptImage.getStorage());
        realImage.getEditLog().setNextTxId(ckptImage.getEditLog().getLastWrittenTxId() + 1L);
        fsNamesys.dir.fsImage = realImage;
        realImage.getStorage().setBlockPoolID(ckptImage.getBlockPoolID());
        this.saveNamespace();
        this.getStorage().writeAll();
    }

    void finalizeUpgrade() throws IOException {
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            this.doFinalize(sd);
        }
    }

    boolean isUpgradeFinalized() {
        return this.isUpgradeFinalized;
    }

    public FSEditLog getEditLog() {
        return this.editLog;
    }

    void openEditLog() throws IOException {
        assert (this.editLog != null) : "editLog must be initialized";
        Preconditions.checkState((!this.editLog.isOpen() ? 1 : 0) != 0, (Object)"edit log should not yet be open");
        this.editLog.open();
        this.storage.writeTransactionIdFileToStorage(this.editLog.getCurSegmentTxId());
    }

    void reloadFromImageFile(File file) throws IOException {
        this.namesystem.dir.reset();
        DelegationTokenSecretManager man = this.namesystem.getDelegationTokenSecretManager();
        if (man != null) {
            man.reset();
        }
        LOG.debug((Object)("Reloading namespace from " + file));
        this.loadFSImage(file);
    }

    boolean loadFSImage() throws IOException {
        FSImageStorageInspector inspector = this.storage.readAndInspectDirs();
        this.isUpgradeFinalized = inspector.isUpgradeFinalized();
        boolean needToSave = inspector.needToSave();
        FSImageStorageInspector.LoadPlan loadPlan = inspector.createLoadPlan();
        LOG.debug((Object)("Planning to load image using following plan:\n" + loadPlan));
        needToSave |= loadPlan.doRecovery();
        Storage.StorageDirectory sdForProperties = loadPlan.getStorageDirectoryForProperties();
        this.storage.readProperties(sdForProperties);
        File imageFile = loadPlan.getImageFile();
        try {
            if (LayoutVersion.supports(LayoutVersion.Feature.TXID_BASED_LAYOUT, this.getLayoutVersion())) {
                this.loadFSImage(imageFile);
            } else if (LayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_CHECKSUM, this.getLayoutVersion())) {
                String md5 = this.storage.getDeprecatedProperty("imageMD5Digest");
                if (md5 == null) {
                    throw new InconsistentFSStateException(sdForProperties.getRoot(), "Message digest property imageMD5Digest not set for storage directory " + sdForProperties.getRoot());
                }
                this.loadFSImage(imageFile, new MD5Hash(md5));
            } else {
                this.loadFSImage(imageFile, null);
            }
        }
        catch (IOException ioe) {
            throw new IOException("Failed to load image from " + loadPlan.getImageFile(), ioe);
        }
        long numLoaded = this.loadEdits(loadPlan.getEditsFiles());
        this.editLog.setNextTxId(this.storage.getMostRecentCheckpointTxId() + numLoaded + 1L);
        return needToSave |= this.needsResaveBasedOnStaleCheckpoint(imageFile, numLoaded);
    }

    private boolean needsResaveBasedOnStaleCheckpoint(File imageFile, long numEditsLoaded) {
        long checkpointPeriod = this.conf.getLong("dfs.namenode.checkpoint.period", 3600L);
        long checkpointTxnCount = this.conf.getLong("dfs.namenode.checkpoint.txns", 40000L);
        long checkpointAge = System.currentTimeMillis() - imageFile.lastModified();
        return checkpointAge > checkpointPeriod * 1000L || numEditsLoaded > checkpointTxnCount;
    }

    protected long loadEdits(List<File> editLogs) throws IOException {
        LOG.debug((Object)("About to load edits:\n  " + Joiner.on((String)"\n  ").join(editLogs)));
        long startingTxId = this.getLastAppliedTxId() + 1L;
        FSEditLogLoader loader = new FSEditLogLoader(this.namesystem);
        int numLoaded = 0;
        for (File edits : editLogs) {
            LOG.debug((Object)("Reading " + edits + " expecting start txid #" + startingTxId));
            EditLogFileInputStream editIn = new EditLogFileInputStream(edits);
            int thisNumLoaded = loader.loadFSEdits(editIn, startingTxId);
            startingTxId += (long)thisNumLoaded;
            numLoaded += thisNumLoaded;
            this.lastAppliedTxId += (long)thisNumLoaded;
            editIn.close();
        }
        this.getFSNamesystem().dir.updateCountForINodeWithQuota();
        return numLoaded;
    }

    private void loadFSImage(File imageFile) throws IOException {
        MD5Hash expectedMD5 = MD5FileUtils.readStoredMd5ForFile(imageFile);
        if (expectedMD5 == null) {
            throw new IOException("No MD5 file found corresponding to image file " + imageFile);
        }
        this.loadFSImage(imageFile, expectedMD5);
    }

    private void loadFSImage(File curFile, MD5Hash expectedMd5) throws IOException {
        FSImageFormat.Loader loader = new FSImageFormat.Loader(this.conf, this.getFSNamesystem());
        loader.load(curFile);
        this.namesystem.setBlockPoolId(this.getBlockPoolID());
        MD5Hash readImageMd5 = loader.getLoadedImageMd5();
        if (expectedMd5 != null && !expectedMd5.equals((Object)readImageMd5)) {
            throw new IOException("Image file " + curFile + " is corrupt with MD5 checksum of " + readImageMd5 + " but expecting " + expectedMd5);
        }
        long txId = loader.getLoadedImageTxId();
        LOG.info((Object)("Loaded image for txid " + txId + " from " + curFile));
        this.lastAppliedTxId = txId;
        this.storage.setMostRecentCheckpointInfo(txId, curFile.lastModified());
    }

    void saveFSImage(Storage.StorageDirectory sd, long txid) throws IOException {
        File newFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
        File dstFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE, txid);
        FSImageFormat.Saver saver = new FSImageFormat.Saver();
        FSImageCompression compression = FSImageCompression.createCompression(this.conf);
        saver.save(newFile, txid, this.getFSNamesystem(), compression);
        MD5FileUtils.saveMD5File(dstFile, saver.getSavedDigest());
        this.storage.setMostRecentCheckpointInfo(txid, Util.now());
    }

    private void waitForThreads(List<Thread> threads) {
        for (Thread thread : threads) {
            while (thread.isAlive()) {
                try {
                    thread.join();
                }
                catch (InterruptedException iex) {
                    LOG.error((Object)("Caught exception while waiting for thread " + thread.getName() + " to finish. Retrying join"));
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void saveNamespace() throws IOException {
        assert (this.editLog != null) : "editLog must be initialized";
        this.storage.attemptRestoreRemovedStorage();
        boolean editLogWasOpen = this.editLog.isOpen();
        if (editLogWasOpen) {
            this.editLog.endCurrentLogSegment(true);
        }
        long imageTxId = this.editLog.getLastWrittenTxId();
        try {
            this.saveFSImageInAllDirs(imageTxId);
            this.storage.writeAll();
        }
        finally {
            if (editLogWasOpen) {
                this.editLog.startLogSegment(imageTxId + 1L, true);
                this.storage.writeTransactionIdFileToStorage(imageTxId + 1L);
            }
        }
    }

    protected void saveFSImageInAllDirs(long txid) throws IOException {
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            throw new IOException("No image directories available!");
        }
        List<Storage.StorageDirectory> errorSDs = Collections.synchronizedList(new ArrayList());
        ArrayList<Thread> saveThreads = new ArrayList<Thread>();
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator(NNStorage.NameNodeDirType.IMAGE);
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            FSImageSaver saver = new FSImageSaver(sd, errorSDs, txid);
            Thread saveThread = new Thread((Runnable)saver, saver.toString());
            saveThreads.add(saveThread);
            saveThread.start();
        }
        this.waitForThreads(saveThreads);
        saveThreads.clear();
        this.storage.reportErrorsOnDirectories(errorSDs);
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            throw new IOException("Failed to save in any storage directories while saving namespace.");
        }
        this.renameCheckpoint(txid);
        this.purgeOldStorage();
    }

    public void purgeOldStorage() {
        try {
            this.archivalManager.purgeOldStorage();
        }
        catch (Exception e) {
            LOG.warn((Object)"Unable to purge old storage", (Throwable)e);
        }
    }

    private void renameCheckpoint(long txid) throws IOException {
        ArrayList al = null;
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            try {
                this.renameCheckpointInDir(sd, txid);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Unable to rename checkpoint in " + sd), (Throwable)ioe);
                if (al == null) {
                    al = Lists.newArrayList();
                }
                al.add(sd);
            }
        }
        if (al != null) {
            this.storage.reportErrorsOnDirectories(al);
        }
    }

    private void renameCheckpointInDir(Storage.StorageDirectory sd, long txid) throws IOException {
        File ckpt = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
        File curFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE, txid);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("renaming  " + ckpt.getAbsolutePath() + " to " + curFile.getAbsolutePath()));
        }
        if (!(ckpt.renameTo(curFile) || curFile.delete() && ckpt.renameTo(curFile))) {
            throw new IOException("renaming  " + ckpt.getAbsolutePath() + " to " + curFile.getAbsolutePath() + " FAILED");
        }
    }

    CheckpointSignature rollEditLog() throws IOException {
        this.getEditLog().rollEditLog();
        this.storage.writeTransactionIdFileToStorage(this.getEditLog().getCurSegmentTxId());
        return new CheckpointSignature(this);
    }

    NamenodeCommand startCheckpoint(NamenodeRegistration bnReg, NamenodeRegistration nnReg) throws IOException {
        String msg = null;
        if (bnReg.getNamespaceID() != this.storage.getNamespaceID()) {
            msg = "Name node " + bnReg.getAddress() + " has incompatible namespace id: " + bnReg.getNamespaceID() + " expected: " + this.storage.getNamespaceID();
        } else if (bnReg.isRole(HdfsServerConstants.NamenodeRole.NAMENODE)) {
            msg = "Name node " + bnReg.getAddress() + " role " + (Object)((Object)bnReg.getRole()) + ": checkpoint is not allowed.";
        } else if (bnReg.getLayoutVersion() < this.storage.getLayoutVersion() || bnReg.getLayoutVersion() == this.storage.getLayoutVersion() && bnReg.getCTime() > this.storage.getCTime()) {
            msg = "Name node " + bnReg.getAddress() + " has newer image layout version: LV = " + bnReg.getLayoutVersion() + " cTime = " + bnReg.getCTime() + ". Current version: LV = " + this.storage.getLayoutVersion() + " cTime = " + this.storage.getCTime();
        }
        if (msg != null) {
            LOG.error((Object)msg);
            return new NamenodeCommand(50);
        }
        boolean needToReturnImg = true;
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            needToReturnImg = false;
        }
        CheckpointSignature sig = this.rollEditLog();
        return new CheckpointCommand(sig, needToReturnImg);
    }

    void endCheckpoint(CheckpointSignature sig) throws IOException {
        sig.validateStorageInfo(this);
    }

    synchronized void saveDigestAndRenameCheckpointImage(long txid, MD5Hash digest) throws IOException {
        this.renameCheckpoint(txid);
        ArrayList badSds = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            File imageFile = NNStorage.getImageFile(sd, txid);
            try {
                MD5FileUtils.saveMD5File(imageFile, digest);
            }
            catch (IOException ioe) {
                badSds.add(sd);
            }
        }
        this.storage.reportErrorsOnDirectories(badSds);
        if (txid > this.storage.getMostRecentCheckpointTxId()) {
            this.storage.setMostRecentCheckpointInfo(txid, Util.now());
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.editLog != null) {
            this.getEditLog().close();
        }
        this.storage.close();
    }

    static Collection<URI> getCheckpointDirs(Configuration conf, String defaultValue) {
        Collection dirNames = conf.getStringCollection("dfs.namenode.checkpoint.dir");
        if (dirNames.size() == 0 && defaultValue != null) {
            dirNames.add(defaultValue);
        }
        return Util.stringCollectionAsURIs(dirNames);
    }

    static Collection<URI> getCheckpointEditsDirs(Configuration conf, String defaultName) {
        Collection dirNames = conf.getStringCollection("dfs.namenode.checkpoint.edits.dir");
        if (dirNames.size() == 0 && defaultName != null) {
            dirNames.add(defaultName);
        }
        return Util.stringCollectionAsURIs(dirNames);
    }

    public NNStorage getStorage() {
        return this.storage;
    }

    public int getLayoutVersion() {
        return this.storage.getLayoutVersion();
    }

    public int getNamespaceID() {
        return this.storage.getNamespaceID();
    }

    public String getClusterID() {
        return this.storage.getClusterID();
    }

    public String getBlockPoolID() {
        return this.storage.getBlockPoolID();
    }

    public synchronized long getLastAppliedTxId() {
        return this.lastAppliedTxId;
    }

    private class FSImageSaver
    implements Runnable {
        private Storage.StorageDirectory sd;
        private List<Storage.StorageDirectory> errorSDs;
        private final long txid;

        FSImageSaver(Storage.StorageDirectory sd, List<Storage.StorageDirectory> errorSDs, long txid) {
            this.sd = sd;
            this.errorSDs = errorSDs;
            this.txid = txid;
        }

        @Override
        public void run() {
            try {
                FSImage.this.saveFSImage(this.sd, this.txid);
            }
            catch (Throwable t) {
                LOG.error((Object)("Unable to save image for " + this.sd.getRoot()), t);
                this.errorSDs.add(this.sd);
            }
        }

        public String toString() {
            return "FSImageSaver for " + this.sd.getRoot() + " of type " + this.sd.getStorageDirType();
        }
    }
}

