/* 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package parquet.filter2.recordlevel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import parquet.hadoop.metadata.ColumnPath;
import parquet.filter2.predicate.FilterPredicate;
import parquet.filter2.predicate.FilterPredicate.Visitor;
import parquet.filter2.predicate.Operators.And;
import parquet.filter2.predicate.Operators.Not;
import parquet.filter2.predicate.Operators.Or;
import parquet.filter2.recordlevel.IncrementallyUpdatedFilterPredicate.ValueInspector;

import static parquet.Preconditions.checkArgument;

/**
 * The implementation of this abstract class is auto-generated by
 * {@link parquet.filter2.IncrementallyUpdatedFilterPredicateGenerator}
 *
 * Constructs a {@link IncrementallyUpdatedFilterPredicate} from a {@link parquet.filter2.predicate.FilterPredicate}
 * This is how records are filtered during record assembly. The implementation is generated in order to avoid autoboxing.
 *
 * Note: the supplied predicate must not contain any instances of the not() operator as this is not
 * supported by this filter.
 *
 * the supplied predicate should first be run through {@link parquet.filter2.predicate.LogicalInverseRewriter} to rewrite it
 * in a form that doesn't make use of the not() operator.
 *
 * the supplied predicate should also have already been run through
 * {@link parquet.filter2.predicate.SchemaCompatibilityValidator}
 * to make sure it is compatible with the schema of this file.
 *
 * TODO: UserDefinedPredicates still autobox however
 */
public abstract class IncrementallyUpdatedFilterPredicateBuilderBase implements Visitor<IncrementallyUpdatedFilterPredicate> {
  private boolean built = false;
  private final Map<ColumnPath, List<ValueInspector>> valueInspectorsByColumn = new HashMap<ColumnPath, List<ValueInspector>>();

  public IncrementallyUpdatedFilterPredicateBuilderBase() { }

  public final IncrementallyUpdatedFilterPredicate build(FilterPredicate pred) {
    checkArgument(!built, "This builder has already been used");
    IncrementallyUpdatedFilterPredicate incremental = pred.accept(this);
    built = true;
    return incremental;
  }

  protected final void addValueInspector(ColumnPath columnPath, ValueInspector valueInspector) {
    List<ValueInspector> valueInspectors = valueInspectorsByColumn.get(columnPath);
    if (valueInspectors == null) {
      valueInspectors = new ArrayList<ValueInspector>();
      valueInspectorsByColumn.put(columnPath, valueInspectors);
    }
    valueInspectors.add(valueInspector);
  }

  public Map<ColumnPath, List<ValueInspector>> getValueInspectorsByColumn() {
    return valueInspectorsByColumn;
  }

  @Override
  public final IncrementallyUpdatedFilterPredicate visit(And and) {
    return new IncrementallyUpdatedFilterPredicate.And(and.getLeft().accept(this), and.getRight().accept(this));
  }

  @Override
  public final IncrementallyUpdatedFilterPredicate visit(Or or) {
    return new IncrementallyUpdatedFilterPredicate.Or(or.getLeft().accept(this), or.getRight().accept(this));
  }

  @Override
  public final IncrementallyUpdatedFilterPredicate visit(Not not) {
    throw new IllegalArgumentException(
        "This predicate contains a not! Did you forget to run this predicate through LogicalInverseRewriter? " + not);
  }

}
