/*
 * Copyright 2006 Robert Burrell Donkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 
package rat.analysis.generation;

import java.util.regex.Pattern;

import rat.analysis.Claims;
import rat.analysis.IHeaderMatcher;
import rat.analysis.RatHeaderAnalysisException;
import rat.report.RatReportFailedException;
import rat.report.claim.IClaimReporter;

public class GeneratedLicenseNotRequired implements IHeaderMatcher {

    private static final Pattern[] DEFAULT_PATTERNS = {Pattern.compile(".*generated by Cayenne.*"),
        Pattern.compile(".*Generated By:JJTree:.*"), 
        Pattern.compile(".*THIS FILE IS AUTOMATICALLY GENERATED.*"),
        Pattern.compile(".*NOTE: this file is autogenerated by XBeans.*"),
        Pattern.compile(".*This file was automatically generated by .*"),
        Pattern.compile(".*# WARNING: DO NOT EDIT OR DELETE THIS WORKSPACE FILE!.*"),
        Pattern.compile(".*# Microsoft Developer Studio Generated NMAKE File.*"),
        Pattern.compile(".*# Microsoft Developer Studio Generated Build File.*"),
        Pattern.compile(".*Generated from configure.ac by autoheader.*"),
        Pattern.compile(".*generated automatically by aclocal.*"),
        Pattern.compile(".*build.xml generated by maven from project.xml.*"),
        Pattern.compile(".*This file was generated by.*"),
        Pattern.compile(".*This file has been automatically generated..*"),
        Pattern.compile(".*Automatically generated - do not modify!.*"),
        Pattern.compile(".*Javadoc style sheet.*"),
        Pattern.compile(".*SOURCE FILE GENERATATED.*"),
        Pattern.compile(".*Generated by the Batik.*"),
        Pattern.compile(".*this file is autogenerated.*")};
    
    
    private final Pattern[] linePatterns;
    private final int numberOfPatterns;
    
    public GeneratedLicenseNotRequired() {
        this(DEFAULT_PATTERNS);
    }
    
    public GeneratedLicenseNotRequired(final Pattern[] linePatterns) {
        this.linePatterns = linePatterns;
        this.numberOfPatterns = linePatterns.length;
    }

    public boolean match(String subject, String line, IClaimReporter reporter) throws RatHeaderAnalysisException {
        boolean result = false;
        for (int i=0;i<numberOfPatterns;i++) {
            if (linePatterns[i].matcher(line).matches()) {
                result = true;
                reportOnLicense(subject, reporter);
                break;
            }
        }
        return result;
    }

    private void reportOnLicense(String subject, IClaimReporter reporter) throws RatHeaderAnalysisException {
        try {
            Claims.reportGeneratedClaims(subject, "Generated files do not required license headers", reporter);
        } catch (RatReportFailedException e) {
            throw new RatHeaderAnalysisException("Cannot write claims", e);
        }
    }

    public void reset() {
        // stateless
    }

}
