/*
 * Decompiled with CFR 0.152.
 */
package parquet.bytes;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import parquet.Log;
import parquet.Preconditions;
import parquet.bytes.ByteBufferAllocator;

public class CapacityByteArrayOutputStream
extends OutputStream {
    private static final Log LOG = Log.getLog(CapacityByteArrayOutputStream.class);
    private static final int MINIMUM_SLAB_SIZE = 65536;
    private static final int EXPONENTIAL_SLAB_SIZE_THRESHOLD = 10;
    private int slabSize;
    private List<ByteBuffer> slabs = new ArrayList<ByteBuffer>();
    private ByteBuffer currentSlab;
    private int capacity;
    private int currentSlabIndex;
    private int currentSlabPosition;
    private int size;
    private ByteBufferAllocator allocator;

    public CapacityByteArrayOutputStream(int initialSize, ByteBufferAllocator a) {
        Preconditions.checkArgument((initialSize > 0 ? 1 : 0) != 0, (String)"initialSize must be > 0");
        this.allocator = a;
        this.initSlabs(initialSize);
    }

    private ByteBuffer allocateSlab(int size) {
        ByteBuffer b = this.allocator != null ? this.allocator.allocate(this.slabSize) : ByteBuffer.wrap(new byte[this.slabSize]);
        return b;
    }

    private void initSlabs(int initialSize) {
        if (Log.DEBUG) {
            LOG.debug((Object)String.format("initial slab of size %d", initialSize));
        }
        this.slabSize = initialSize;
        for (ByteBuffer slab : this.slabs) {
            this.allocator.release(slab);
        }
        this.slabs.clear();
        this.capacity = initialSize;
        this.currentSlab = this.allocateSlab(this.slabSize);
        this.slabs.add(this.currentSlab);
        this.currentSlabIndex = 0;
        this.currentSlabPosition = 0;
        this.size = 0;
    }

    private void addSlab(int minimumSize) {
        ++this.currentSlabIndex;
        if (this.currentSlabIndex < this.slabs.size()) {
            this.currentSlab = this.slabs.get(this.currentSlabIndex);
            if (Log.DEBUG) {
                LOG.debug((Object)String.format("reusing slab of size %d", this.currentSlab.capacity()));
            }
            if (this.currentSlab.capacity() < minimumSize) {
                if (Log.DEBUG) {
                    LOG.debug((Object)String.format("slab size %,d too small for value of size %,d. replacing slab", this.currentSlab.capacity(), minimumSize));
                }
                ByteBuffer newSlab = this.allocateSlab(minimumSize);
                this.capacity += minimumSize - this.currentSlab.capacity();
                this.currentSlab = newSlab;
                this.slabs.set(this.currentSlabIndex, newSlab);
            }
        } else {
            if (this.currentSlabIndex > 10) {
                this.slabSize = this.size;
                if (Log.DEBUG) {
                    LOG.debug((Object)String.format("used %d slabs, new slab size %d", this.currentSlabIndex, this.slabSize));
                }
            }
            if (this.slabSize < minimumSize) {
                if (Log.DEBUG) {
                    LOG.debug((Object)String.format("slab size %,d too small for value of size %,d. Bumping up slab size", this.slabSize, minimumSize));
                }
                this.slabSize = minimumSize;
            }
            if (Log.DEBUG) {
                LOG.debug((Object)String.format("new slab of size %d", this.slabSize));
            }
            this.currentSlab = this.allocateSlab(this.slabSize);
            this.slabs.add(this.currentSlab);
            this.capacity += this.slabSize;
        }
        this.currentSlabPosition = 0;
    }

    @Override
    public void write(int b) {
        if (!this.currentSlab.hasRemaining()) {
            this.addSlab(1);
        }
        this.currentSlab.put((byte)b);
        ++this.currentSlabPosition;
        ++this.size;
    }

    @Override
    public void write(byte[] b, int off, int len) {
        if (off < 0 || off > b.length || len < 0 || off + len - b.length > 0) {
            throw new IndexOutOfBoundsException();
        }
        if (this.currentSlabPosition + len >= this.currentSlab.limit()) {
            int length1 = this.currentSlab.limit() - this.currentSlabPosition;
            this.currentSlab.put(b, off, length1);
            int length2 = len - length1;
            this.addSlab(length2);
            this.currentSlab.put(b, off + length1, length2);
            this.currentSlabPosition = length2;
        } else {
            this.currentSlab.put(b, off, len);
            this.currentSlabPosition += len;
        }
        this.size += len;
    }

    public void writeTo(OutputStream out) throws IOException {
        byte[] b;
        for (int i = 0; i < this.currentSlabIndex; ++i) {
            ByteBuffer slab = this.slabs.get(i);
            int bytesToRead = slab.position();
            slab.flip();
            b = new byte[bytesToRead];
            slab.get(b);
            out.write(b);
        }
        int bytesToRead = this.currentSlab.position();
        this.currentSlab.flip();
        b = new byte[bytesToRead];
        this.currentSlab.get(b);
        out.write(b);
    }

    public int getCapacity() {
        return this.capacity;
    }

    public void reset() {
        if (this.currentSlabIndex == 0 && this.currentSlabPosition < this.currentSlab.capacity() / 2 && this.currentSlab.capacity() > 65536 || this.currentSlabIndex > 10) {
            this.initSlabs(Math.max(this.size / 5, 65536));
            if (Log.DEBUG) {
                LOG.debug((Object)String.format("used %d slabs, new slab size %d", this.currentSlabIndex + 1, this.slabSize));
            }
        } else if (this.currentSlabIndex < this.slabs.size()) {
            for (int i = this.currentSlabIndex + 1; i < this.slabs.size(); ++i) {
                ByteBuffer slab = this.slabs.get(i);
                this.allocator.release(slab);
            }
            this.slabs = new ArrayList<ByteBuffer>(this.slabs.subList(0, this.currentSlabIndex + 1));
            this.capacity = 0;
            for (ByteBuffer slab : this.slabs) {
                this.capacity += slab.capacity();
                slab.clear();
            }
        }
        this.currentSlabIndex = 0;
        this.currentSlabPosition = 0;
        this.currentSlab = this.slabs.get(this.currentSlabIndex);
        this.size = 0;
    }

    public long size() {
        return this.size;
    }

    public long getCurrentIndex() {
        Preconditions.checkArgument((this.size > 0 ? 1 : 0) != 0, (String)"This is an empty stream");
        return this.size - 1;
    }

    public void setByte(long index, byte value) {
        Preconditions.checkArgument((index < (long)this.size ? 1 : 0) != 0, (String)("Index: " + index + " is >= the current size of: " + this.size));
        long seen = 0L;
        for (int i = 0; i <= this.currentSlabIndex; ++i) {
            ByteBuffer slab = this.slabs.get(i);
            if (index < seen + (long)slab.limit()) {
                slab.put((int)(index - seen), value);
                break;
            }
            seen += (long)slab.limit();
        }
    }

    public String memUsageString(String prefix) {
        return String.format("%s %s %d slabs, %,d bytes", prefix, this.getClass().getSimpleName(), this.slabs.size(), this.getCapacity());
    }

    int getSlabCount() {
        return this.slabs.size();
    }

    @Override
    public void close() {
        for (ByteBuffer slab : this.slabs) {
            this.allocator.release(slab);
        }
        try {
            super.close();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }
}

