/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.master.cleaner.DirScanPool;
import org.apache.hadoop.hbase.master.cleaner.FileCleanerDelegate;
import org.apache.hadoop.hbase.util.FutureUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableSet;
import org.apache.hbase.thirdparty.com.google.common.collect.Iterables;
import org.apache.hbase.thirdparty.com.google.common.collect.Lists;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public abstract class CleanerChore<T extends FileCleanerDelegate>
extends ScheduledChore {
    private static final Logger LOG = LoggerFactory.getLogger(CleanerChore.class);
    private static final int AVAIL_PROCESSORS = Runtime.getRuntime().availableProcessors();
    public static final String CHORE_POOL_SIZE = "hbase.cleaner.scan.dir.concurrent.size";
    static final String DEFAULT_CHORE_POOL_SIZE = "0.25";
    private final DirScanPool pool;
    protected final FileSystem fs;
    private final Path oldFileDir;
    private final Configuration conf;
    protected final Map<String, Object> params;
    private final AtomicBoolean enabled = new AtomicBoolean(true);
    protected List<T> cleanersChain;

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey, DirScanPool pool) {
        this(name, sleepPeriod, s, conf, fs, oldFileDir, confKey, pool, null);
    }

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey, DirScanPool pool, Map<String, Object> params) {
        super(name, s, sleepPeriod);
        Preconditions.checkNotNull((Object)pool, (Object)"Chore's pool can not be null");
        this.pool = pool;
        this.fs = fs;
        this.oldFileDir = oldFileDir;
        this.conf = conf;
        this.params = params;
        this.initCleanerChain(confKey);
    }

    static int calculatePoolSize(String poolSize) {
        if (poolSize.matches("[1-9][0-9]*")) {
            int size = Math.min(Integer.parseInt(poolSize), AVAIL_PROCESSORS);
            if (size == AVAIL_PROCESSORS) {
                LOG.warn("Use full core processors to scan dir, size={}", (Object)size);
            }
            return size;
        }
        if (poolSize.matches("0.[0-9]+|1.0")) {
            int computedThreads = (int)((double)AVAIL_PROCESSORS * Double.parseDouble(poolSize));
            if (computedThreads < 1) {
                LOG.debug("Computed {} threads for CleanerChore, using 1 instead", (Object)computedThreads);
                return 1;
            }
            return computedThreads;
        }
        LOG.error("Unrecognized value: " + poolSize + " for " + CHORE_POOL_SIZE + ", use default config: " + DEFAULT_CHORE_POOL_SIZE + " instead.");
        return CleanerChore.calculatePoolSize(DEFAULT_CHORE_POOL_SIZE);
    }

    protected abstract boolean validate(Path var1);

    private void initCleanerChain(String confKey) {
        this.cleanersChain = new LinkedList<T>();
        String[] logCleaners = this.conf.getStrings(confKey);
        if (logCleaners != null) {
            for (String className : logCleaners) {
                T logCleaner;
                if ((className = className.trim()).isEmpty() || (logCleaner = this.newFileCleaner(className, this.conf)) == null) continue;
                LOG.info("Initialize cleaner={}", (Object)className);
                this.cleanersChain.add(logCleaner);
            }
        }
    }

    private T newFileCleaner(String className, Configuration conf) {
        try {
            Class<FileCleanerDelegate> c = Class.forName(className).asSubclass(FileCleanerDelegate.class);
            FileCleanerDelegate cleaner = c.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            cleaner.setConf(conf);
            cleaner.init(this.params);
            return (T)cleaner;
        }
        catch (Exception e) {
            LOG.warn("Can NOT create CleanerDelegate={}", (Object)className, (Object)e);
            return null;
        }
    }

    protected void chore() {
        if (this.getEnabled()) {
            try {
                this.pool.latchCountUp();
                if (this.runCleaner()) {
                    LOG.trace("Cleaned all WALs under {}", (Object)this.oldFileDir);
                } else {
                    LOG.trace("WALs outstanding under {}", (Object)this.oldFileDir);
                }
            }
            finally {
                this.pool.latchCountDown();
            }
            this.pool.tryUpdatePoolSize((long)(0.8 * (double)this.getTimeUnit().toMillis(this.getPeriod())));
        } else {
            LOG.trace("Cleaner chore disabled! Not cleaning.");
        }
    }

    private void preRunCleaner() {
        this.cleanersChain.forEach(FileCleanerDelegate::preClean);
    }

    public boolean runCleaner() {
        this.preRunCleaner();
        try {
            CompletableFuture future = new CompletableFuture();
            this.pool.execute(() -> this.traverseAndDelete(this.oldFileDir, true, future));
            return (Boolean)future.get();
        }
        catch (Exception e) {
            LOG.info("Failed to traverse and delete the dir: {}", (Object)this.oldFileDir, (Object)e);
            return false;
        }
    }

    private void sortByConsumedSpace(List<FileStatus> dirs) {
        if (dirs == null || dirs.size() < 2) {
            return;
        }
        dirs.sort(new Comparator<FileStatus>(){
            HashMap<FileStatus, Long> directorySpaces = new HashMap();

            @Override
            public int compare(FileStatus f1, FileStatus f2) {
                long f1ConsumedSpace = this.getSpace(f1);
                long f2ConsumedSpace = this.getSpace(f2);
                return Long.compare(f2ConsumedSpace, f1ConsumedSpace);
            }

            private long getSpace(FileStatus f) {
                Long cached = this.directorySpaces.get(f);
                if (cached != null) {
                    return cached;
                }
                try {
                    long space = f.isDirectory() ? CleanerChore.this.fs.getContentSummary(f.getPath()).getSpaceConsumed() : f.getLen();
                    this.directorySpaces.put(f, space);
                    return space;
                }
                catch (IOException e) {
                    LOG.trace("Failed to get space consumed by path={}", (Object)f, (Object)e);
                    return -1L;
                }
            }
        });
    }

    private boolean checkAndDeleteFiles(List<FileStatus> files) {
        if (files == null) {
            return true;
        }
        ArrayList validFiles = Lists.newArrayListWithCapacity((int)files.size());
        ArrayList invalidFiles = Lists.newArrayList();
        for (FileStatus fileStatus : files) {
            if (this.validate(fileStatus.getPath())) {
                validFiles.add(fileStatus);
                continue;
            }
            LOG.warn("Found a wrongly formatted file: " + fileStatus.getPath() + " - will delete it.");
            invalidFiles.add(fileStatus);
        }
        Iterable<Object> deletableValidFiles = validFiles;
        for (FileCleanerDelegate cleaner : this.cleanersChain) {
            if (cleaner.isStopped() || this.getStopper().isStopped()) {
                LOG.warn("A file cleaner" + this.getName() + " is stopped, won't delete any more files in:" + this.oldFileDir);
                return false;
            }
            Iterable<FileStatus> filteredFiles = cleaner.getDeletableFiles((Iterable<FileStatus>)deletableValidFiles);
            if (LOG.isTraceEnabled()) {
                ImmutableSet filteredFileSet = ImmutableSet.copyOf(filteredFiles);
                for (FileStatus fileStatus : deletableValidFiles) {
                    if (filteredFileSet.contains((Object)fileStatus)) continue;
                    LOG.trace(fileStatus.getPath() + " is not deletable according to:" + cleaner);
                }
            }
            deletableValidFiles = filteredFiles;
        }
        Iterable iterable = Iterables.concat((Iterable)invalidFiles, (Iterable)deletableValidFiles);
        return this.deleteFiles(iterable) == files.size();
    }

    private boolean isEmptyDirDeletable(Path dir) {
        for (FileCleanerDelegate cleaner : this.cleanersChain) {
            if (cleaner.isStopped() || this.getStopper().isStopped()) {
                LOG.warn("A file cleaner {} is stopped, won't delete the empty directory {}", (Object)this.getName(), (Object)dir);
                return false;
            }
            if (cleaner.isEmptyDirDeletable(dir)) continue;
            return false;
        }
        return true;
    }

    protected int deleteFiles(Iterable<FileStatus> filesToDelete) {
        int deletedFileCount = 0;
        for (FileStatus file : filesToDelete) {
            Path filePath = file.getPath();
            LOG.trace("Removing {} from archive", (Object)filePath);
            try {
                boolean success = this.fs.delete(filePath, false);
                if (success) {
                    ++deletedFileCount;
                    continue;
                }
                LOG.warn("Attempted to delete:" + filePath + ", but couldn't. Run cleaner chain and attempt to delete on next pass.");
            }
            catch (IOException e) {
                e = e instanceof RemoteException ? ((RemoteException)((Object)e)).unwrapRemoteException() : e;
                LOG.warn("Error while deleting: " + filePath, (Throwable)e);
            }
        }
        return deletedFileCount;
    }

    public synchronized void cleanup() {
        for (FileCleanerDelegate lc : this.cleanersChain) {
            try {
                lc.stop("Exiting");
            }
            catch (Throwable t) {
                LOG.warn("Stopping", t);
            }
        }
    }

    int getChorePoolSize() {
        return this.pool.getSize();
    }

    public boolean setEnabled(boolean enabled) {
        return this.enabled.getAndSet(enabled);
    }

    public boolean getEnabled() {
        return this.enabled.get();
    }

    private void traverseAndDelete(Path dir, boolean root, CompletableFuture<Boolean> result) {
        try {
            List<FileStatus> allPaths = Arrays.asList(this.fs.listStatus(dir));
            List<FileStatus> subDirs = allPaths.stream().filter(FileStatus::isDirectory).collect(Collectors.toList());
            List files = allPaths.stream().filter(FileStatus::isFile).collect(Collectors.toList());
            boolean allFilesDeleted = files.isEmpty() || this.deleteAction(() -> this.checkAndDeleteFiles(files), "files", dir);
            ArrayList futures = new ArrayList();
            if (!subDirs.isEmpty()) {
                this.sortByConsumedSpace(subDirs);
                subDirs.forEach(subDir -> {
                    CompletableFuture subFuture = new CompletableFuture();
                    this.pool.execute(() -> this.traverseAndDelete(subDir.getPath(), false, subFuture));
                    futures.add(subFuture);
                });
            }
            FutureUtils.addListener(CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()])), (voidObj, e) -> {
                if (e != null) {
                    result.completeExceptionally((Throwable)e);
                    return;
                }
                try {
                    boolean deleted;
                    boolean allSubDirsDeleted = futures.stream().allMatch(CompletableFuture::join);
                    boolean bl = deleted = allFilesDeleted && allSubDirsDeleted && this.isEmptyDirDeletable(dir);
                    if (deleted && !root) {
                        deleted = this.deleteAction(() -> this.fs.delete(dir, false), "dir", dir);
                    }
                    result.complete(deleted);
                }
                catch (Exception ie) {
                    result.completeExceptionally(ie);
                }
            });
        }
        catch (Exception e2) {
            LOG.debug("Failed to traverse and delete the path: {}", (Object)dir, (Object)e2);
            result.completeExceptionally(e2);
        }
    }

    private boolean deleteAction(Action<Boolean> deletion, String type, Path dir) {
        boolean deleted;
        try {
            LOG.trace("Start deleting {} under {}", (Object)type, (Object)dir);
            deleted = deletion.act();
        }
        catch (PathIsNotEmptyDirectoryException exception) {
            LOG.debug("Couldn't delete '{}' yet because it isn't empty w/exception.", (Object)dir, (Object)exception);
            deleted = false;
        }
        catch (IOException ioe) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Could not delete {} under {}; will retry. If it keeps happening, quote the exception when asking on mailing list.", new Object[]{type, dir, ioe});
            } else {
                LOG.info("Could not delete {} under {} because {}; will retry. If it  keeps happening, enableTRACE-level logging and quote the exception when asking on mailing list.", new Object[]{type, dir, ioe.getMessage()});
            }
            deleted = false;
        }
        catch (Exception e) {
            LOG.info("unexpected exception: ", (Throwable)e);
            deleted = false;
        }
        LOG.trace("Finish deleting {} under {}, deleted=", new Object[]{type, dir, deleted});
        return deleted;
    }

    private static interface Action<T> {
        public T act() throws Exception;
    }
}

