/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.security.PrivilegedExceptionAction;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FutureDataInputStreamBuilder;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.thirdparty.com.google.common.cache.CacheLoader;
import org.apache.hadoop.thirdparty.com.google.common.cache.LoadingCache;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.Futures;
import org.apache.hadoop.util.RunJar;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.functional.FutureIO;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceVisibility;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.util.Times;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"YARN", "MapReduce"})
public class FSDownload
implements Callable<Path> {
    private static final Logger LOG = LoggerFactory.getLogger(FSDownload.class);
    private FileContext files;
    private final UserGroupInformation userUgi;
    private Configuration conf;
    private LocalResource resource;
    private final LoadingCache<Path, Future<FileStatus>> statCache;
    private Path destDirPath;
    private static final FsPermission cachePerms = new FsPermission(493);
    static final FsPermission PUBLIC_FILE_PERMS = new FsPermission(365);
    static final FsPermission PRIVATE_FILE_PERMS = new FsPermission(320);
    static final FsPermission PUBLIC_DIR_PERMS = new FsPermission(493);
    static final FsPermission PRIVATE_DIR_PERMS = new FsPermission(448);

    public FSDownload(FileContext files, UserGroupInformation ugi, Configuration conf, Path destDirPath, LocalResource resource) {
        this(files, ugi, conf, destDirPath, resource, null);
    }

    public FSDownload(FileContext files, UserGroupInformation ugi, Configuration conf, Path destDirPath, LocalResource resource, LoadingCache<Path, Future<FileStatus>> statCache) {
        this.conf = conf;
        this.destDirPath = destDirPath;
        this.files = files;
        this.userUgi = ugi;
        this.resource = resource;
        this.statCache = statCache;
    }

    LocalResource getResource() {
        return this.resource;
    }

    private void createDir(Path path, FsPermission perm) throws IOException {
        this.files.mkdir(path, perm, false);
        if (!perm.equals((Object)this.files.getUMask().applyUMask(perm))) {
            this.files.setPermission(path, perm);
        }
    }

    public static CacheLoader<Path, Future<FileStatus>> createStatusCacheLoader(final Configuration conf) {
        return new CacheLoader<Path, Future<FileStatus>>(){

            public Future<FileStatus> load(Path path) {
                try {
                    FileSystem fs = path.getFileSystem(conf);
                    return Futures.immediateFuture((Object)fs.getFileStatus(path));
                }
                catch (Throwable th) {
                    return Futures.immediateFailedFuture((Throwable)th);
                }
            }
        };
    }

    @InterfaceAudience.Private
    public static boolean isPublic(FileSystem fs, Path current, FileStatus sStat, LoadingCache<Path, Future<FileStatus>> statCache) throws IOException {
        current = fs.makeQualified(current);
        if (!FSDownload.checkPublicPermsForAll(fs, sStat, FsAction.READ_EXECUTE, FsAction.READ)) {
            return false;
        }
        if (Shell.WINDOWS && fs instanceof LocalFileSystem) {
            return true;
        }
        return FSDownload.ancestorsHaveExecutePermissions(fs, current.getParent(), statCache);
    }

    private static boolean checkPublicPermsForAll(FileSystem fs, FileStatus status, FsAction dir, FsAction file) throws IOException {
        FsPermission perms = status.getPermission();
        FsAction otherAction = perms.getOtherAction();
        if (status.isDirectory()) {
            if (!otherAction.implies(dir)) {
                return false;
            }
            for (FileStatus child : fs.listStatus(status.getPath())) {
                if (FSDownload.checkPublicPermsForAll(fs, child, dir, file)) continue;
                return false;
            }
            return true;
        }
        return otherAction.implies(file);
    }

    @VisibleForTesting
    static boolean ancestorsHaveExecutePermissions(FileSystem fs, Path path, LoadingCache<Path, Future<FileStatus>> statCache) throws IOException {
        for (Path current = path; current != null; current = current.getParent()) {
            if (FSDownload.checkPermissionOfOther(fs, current, FsAction.EXECUTE, statCache)) continue;
            return false;
        }
        return true;
    }

    private static boolean checkPermissionOfOther(FileSystem fs, Path path, FsAction action, LoadingCache<Path, Future<FileStatus>> statCache) throws IOException {
        FileStatus status = FSDownload.getFileStatus(fs, path, statCache);
        FsPermission perms = status.getPermission();
        FsAction otherAction = perms.getOtherAction();
        return otherAction.implies(action);
    }

    private static FileStatus getFileStatus(FileSystem fs, Path path, LoadingCache<Path, Future<FileStatus>> statCache) throws IOException {
        if (statCache == null) {
            return fs.getFileStatus(path);
        }
        try {
            return (FileStatus)((Future)statCache.get((Object)path)).get();
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException)cause;
            }
            throw new IOException(cause);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(e);
        }
    }

    private void verifyAndCopy(Path destination) throws IOException, YarnException {
        Path sCopy;
        try {
            sCopy = this.resource.getResource().toPath();
        }
        catch (URISyntaxException e) {
            throw new IOException("Invalid resource", e);
        }
        FileSystem sourceFs = sCopy.getFileSystem(this.conf);
        FileStatus sStat = sourceFs.getFileStatus(sCopy);
        if (sStat.getModificationTime() != this.resource.getTimestamp()) {
            throw new IOException("Resource " + sCopy + " changed on src filesystem - expected: \"" + Times.formatISO8601(this.resource.getTimestamp()) + "\", was: \"" + Times.formatISO8601(sStat.getModificationTime()) + "\", current time: \"" + Times.formatISO8601(Time.now()) + "\"");
        }
        if (this.resource.getVisibility() == LocalResourceVisibility.PUBLIC && !FSDownload.isPublic(sourceFs, sCopy, sStat, this.statCache)) {
            throw new IOException("Resource " + sCopy + " is not publicly accessible and as such cannot be part of the public cache.");
        }
        this.downloadAndUnpack(sCopy, sStat, destination);
    }

    private void downloadAndUnpack(Path source, FileStatus sourceStatus, Path destination) throws YarnException {
        try {
            FileSystem sourceFileSystem = source.getFileSystem(this.conf);
            FileSystem destinationFileSystem = destination.getFileSystem(this.conf);
            if (sourceStatus.isDirectory()) {
                FileUtil.copy((FileSystem)sourceFileSystem, (FileStatus)sourceStatus, (FileSystem)destinationFileSystem, (Path)destination, (boolean)false, (boolean)true, (Configuration)this.conf);
            } else {
                this.unpack(source, destination, sourceFileSystem, destinationFileSystem);
            }
        }
        catch (Exception e) {
            throw new YarnException("Download and unpack failed", (Throwable)e);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void unpack(Path source, Path destination, FileSystem sourceFileSystem, FileSystem destinationFileSystem) throws IOException, InterruptedException, ExecutionException {
        try (InputStream inputStream = (InputStream)FutureIO.awaitFuture((Future)((FutureDataInputStreamBuilder)sourceFileSystem.openFile(source).opt("fs.option.openfile.read.policy", "whole-file")).build());){
            File dst = new File(destination.toUri());
            String lowerDst = StringUtils.toLowerCase((String)dst.getName());
            switch (this.resource.getType()) {
                case ARCHIVE: {
                    if (lowerDst.endsWith(".jar")) {
                        RunJar.unJar((InputStream)inputStream, (File)dst, (Pattern)RunJar.MATCH_ANY);
                        return;
                    }
                    if (lowerDst.endsWith(".zip")) {
                        FileUtil.unZip((InputStream)inputStream, (File)dst);
                        return;
                    }
                    if (lowerDst.endsWith(".tar.gz") || lowerDst.endsWith(".tgz") || lowerDst.endsWith(".tar")) {
                        FileUtil.unTar((InputStream)inputStream, (File)dst, (boolean)lowerDst.endsWith("gz"));
                        return;
                    }
                    LOG.warn("Cannot unpack " + source);
                    try (FSDataOutputStream outputStream = destinationFileSystem.create(destination, true);){
                        IOUtils.copy((InputStream)inputStream, (OutputStream)outputStream);
                        return;
                    }
                }
                case PATTERN: {
                    if (lowerDst.endsWith(".jar")) {
                        String p = this.resource.getPattern();
                        if (!dst.exists() && !dst.mkdir()) {
                            throw new IOException("Unable to create directory: [" + dst + "]");
                        }
                        RunJar.unJarAndSave((InputStream)inputStream, (File)dst, (String)source.getName(), (Pattern)(p == null ? RunJar.MATCH_ANY : Pattern.compile(p)));
                        return;
                    }
                    if (lowerDst.endsWith(".zip")) {
                        LOG.warn("Treating [" + source + "] as an archive even though it was specified as PATTERN");
                        FileUtil.unZip((InputStream)inputStream, (File)dst);
                        return;
                    }
                    if (lowerDst.endsWith(".tar.gz") || lowerDst.endsWith(".tgz") || lowerDst.endsWith(".tar")) {
                        LOG.warn("Treating [" + source + "] as an archive even though it was specified as PATTERN");
                        FileUtil.unTar((InputStream)inputStream, (File)dst, (boolean)lowerDst.endsWith("gz"));
                        return;
                    }
                    LOG.warn("Cannot unpack " + source);
                    try (FSDataOutputStream outputStream = destinationFileSystem.create(destination, true);){
                        IOUtils.copy((InputStream)inputStream, (OutputStream)outputStream);
                        return;
                    }
                }
                default: {
                    try (FSDataOutputStream outputStream = destinationFileSystem.create(destination, true);){
                        IOUtils.copy((InputStream)inputStream, (OutputStream)outputStream);
                        return;
                    }
                }
            }
        }
    }

    @Override
    public Path call() throws Exception {
        Path sCopy;
        try {
            sCopy = this.resource.getResource().toPath();
        }
        catch (URISyntaxException e) {
            throw new IOException("Invalid resource", e);
        }
        LOG.debug("Starting to download {} {} {}", new Object[]{sCopy, this.resource.getType(), this.resource.getPattern()});
        Path destinationTmp = new Path(this.destDirPath + "_tmp");
        this.createDir(destinationTmp, cachePerms);
        final Path dFinal = this.files.makeQualified(new Path(destinationTmp, sCopy.getName()));
        try {
            if (this.userUgi == null) {
                this.verifyAndCopy(dFinal);
            } else {
                this.userUgi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Void>(){

                    @Override
                    public Void run() throws Exception {
                        FSDownload.this.verifyAndCopy(dFinal);
                        return null;
                    }
                });
            }
            this.changePermissions(dFinal.getFileSystem(this.conf), dFinal);
            this.files.rename(destinationTmp, this.destDirPath, new Options.Rename[]{Options.Rename.OVERWRITE});
            LOG.debug("File has been downloaded to {} from {}", (Object)new Path(this.destDirPath, sCopy.getName()), (Object)sCopy);
        }
        catch (Exception e) {
            try {
                this.files.delete(this.destDirPath, true);
            }
            catch (IOException iOException) {
                // empty catch block
            }
            throw e;
        }
        finally {
            try {
                this.files.delete(destinationTmp, true);
            }
            catch (FileNotFoundException fileNotFoundException) {}
            this.conf = null;
            this.resource = null;
        }
        return this.files.makeQualified(new Path(this.destDirPath, sCopy.getName()));
    }

    private void changePermissions(FileSystem fs, final Path path) throws IOException, InterruptedException {
        File f = new File(path.toUri());
        if (FileUtils.isSymlink((File)f)) {
            return;
        }
        boolean isDir = f.isDirectory();
        FsPermission perm = cachePerms;
        perm = this.resource.getVisibility() == LocalResourceVisibility.PUBLIC ? (isDir ? PUBLIC_DIR_PERMS : PUBLIC_FILE_PERMS) : (isDir ? PRIVATE_DIR_PERMS : PRIVATE_FILE_PERMS);
        LOG.debug("Changing permissions for path {} to perm {}", (Object)path, (Object)perm);
        final FsPermission fPerm = perm;
        if (null == this.userUgi) {
            this.files.setPermission(path, perm);
        } else {
            this.userUgi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Void>(){

                @Override
                public Void run() throws Exception {
                    FSDownload.this.files.setPermission(path, fPerm);
                    return null;
                }
            });
        }
        if (isDir) {
            FileStatus[] statuses;
            for (FileStatus status : statuses = fs.listStatus(path)) {
                this.changePermissions(fs, status.getPath());
            }
        }
    }
}

