/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.stateless.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.nifi.stateless.config.ExtensionClientDefinition;
import org.apache.nifi.stateless.config.SslContextDefinition;
import org.apache.nifi.stateless.config.StatelessConfigurationException;
import org.apache.nifi.stateless.engine.StatelessEngineConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PropertiesFileEngineConfigurationParser {
    private static final Logger logger = LoggerFactory.getLogger(PropertiesFileEngineConfigurationParser.class);
    private static final String PREFIX = "nifi.stateless.";
    private static final String NAR_DIRECTORY = "nifi.stateless.nar.directory";
    private static final String EXTENSIONS_DIRECTORY = "nifi.stateless.extensions.directory";
    private static final String READONLY_EXTENSIONS_DIRECTORY = "nifi.stateless.readonly.extensions.directory.";
    private static final String WORKING_DIRECTORY = "nifi.stateless.working.directory";
    private static final String CONTENT_REPO_DIRECTORY = "nifi.stateless.content.repository.directory";
    private static final String STATUS_TASK_INTERVAL = "nifi.stateless.status.task.interval";
    private static final String TRUSTSTORE_FILE = "nifi.stateless.security.truststore";
    private static final String TRUSTSTORE_TYPE = "nifi.stateless.security.truststoreType";
    private static final String TRUSTSTORE_PASSWORD = "nifi.stateless.security.truststorePasswd";
    private static final String KEYSTORE_FILE = "nifi.stateless.security.keystore";
    private static final String KEYSTORE_TYPE = "nifi.stateless.security.keystoreType";
    private static final String KEYSTORE_PASSWORD = "nifi.stateless.security.keystorePasswd";
    private static final String KEY_PASSWORD = "nifi.stateless.security.keyPasswd";
    private static final String SENSITIVE_PROPS_KEY = "nifi.stateless.sensitive.props.key";
    private static final String KRB5_FILE = "nifi.stateless.kerberos.krb5.file";
    private static final String DEFAULT_KRB5_FILENAME = "/etc/krb5.conf";
    private static final Pattern EXTENSION_CLIENT_PATTERN = Pattern.compile("\\Qnifi.stateless.extension.client.\\E(.*?)\\.(.+)");
    private static final int PROPERTIES_KEY_LENGTH = 24;

    public StatelessEngineConfiguration parseEngineConfiguration(File propertiesFile) throws IOException, StatelessConfigurationException {
        File extensionsDirectory;
        if (!propertiesFile.exists()) {
            throw new FileNotFoundException("Could not find properties file " + propertiesFile.getAbsolutePath());
        }
        Properties properties = new Properties();
        try (FileInputStream in = new FileInputStream(propertiesFile);){
            properties.load(in);
        }
        final File narDirectory = new File(this.getRequired(properties, NAR_DIRECTORY));
        if (!narDirectory.exists()) {
            throw new StatelessConfigurationException("NAR Directory " + narDirectory.getAbsolutePath() + " specified in properties file does not exist");
        }
        final File workingDirectory = new File(this.getRequired(properties, WORKING_DIRECTORY));
        if (!workingDirectory.exists() && !workingDirectory.mkdirs()) {
            throw new StatelessConfigurationException("Working Directory " + workingDirectory.getAbsolutePath() + " specified in properties file does not exist and could not be created");
        }
        String extensionsDirectoryFilename = properties.getProperty(EXTENSIONS_DIRECTORY);
        File file = extensionsDirectory = extensionsDirectoryFilename == null ? narDirectory : new File(extensionsDirectoryFilename);
        if (!extensionsDirectory.exists() && !extensionsDirectory.mkdirs()) {
            throw new StatelessConfigurationException("Extensions Directory " + narDirectory.getAbsolutePath() + " specified in properties file does not exist and could not be created");
        }
        final List<File> readOnlyExtensionsDirectories = this.getReadOnlyExtensionsDirectories(properties);
        String contentRepoDirectoryFilename = properties.getProperty(CONTENT_REPO_DIRECTORY, "");
        final File contentRepoDirectory = contentRepoDirectoryFilename.isEmpty() ? null : new File(contentRepoDirectoryFilename);
        String krb5Filename = properties.getProperty(KRB5_FILE, DEFAULT_KRB5_FILENAME);
        final File krb5File = new File(krb5Filename);
        final String sensitivePropsKey = this.getSensitivePropsKey(propertiesFile, properties);
        final SslContextDefinition sslContextDefinition = this.parseSslContextDefinition(properties);
        final List<ExtensionClientDefinition> extensionClients = this.parseExtensionClients(properties);
        final String statusTaskInterval = properties.getProperty(STATUS_TASK_INTERVAL, "1 min");
        return new StatelessEngineConfiguration(){

            @Override
            public File getWorkingDirectory() {
                return workingDirectory;
            }

            @Override
            public File getNarDirectory() {
                return narDirectory;
            }

            @Override
            public File getExtensionsDirectory() {
                return extensionsDirectory;
            }

            @Override
            public Collection<File> getReadOnlyExtensionsDirectories() {
                return readOnlyExtensionsDirectories;
            }

            @Override
            public File getKrb5File() {
                return krb5File;
            }

            @Override
            public Optional<File> getContentRepositoryDirectory() {
                return Optional.ofNullable(contentRepoDirectory);
            }

            @Override
            public SslContextDefinition getSslContext() {
                return sslContextDefinition;
            }

            @Override
            public String getSensitivePropsKey() {
                return sensitivePropsKey;
            }

            @Override
            public List<ExtensionClientDefinition> getExtensionClients() {
                return extensionClients;
            }

            @Override
            public String getStatusTaskInterval() {
                return statusTaskInterval;
            }
        };
    }

    private List<File> getReadOnlyExtensionsDirectories(Properties properties) {
        return properties.keySet().stream().map(Object::toString).filter(key -> key.startsWith(READONLY_EXTENSIONS_DIRECTORY)).map(properties::getProperty).map(File::new).collect(Collectors.toList());
    }

    private List<ExtensionClientDefinition> parseExtensionClients(Properties properties) {
        LinkedHashMap<String, ExtensionClientDefinition> extensionClientDefinitions = new LinkedHashMap<String, ExtensionClientDefinition>();
        block12: for (String propertyName : properties.stringPropertyNames()) {
            Matcher matcher = EXTENSION_CLIENT_PATTERN.matcher(propertyName);
            if (!matcher.matches()) continue;
            String clientKey = matcher.group(1);
            ExtensionClientDefinition definition = extensionClientDefinitions.computeIfAbsent(clientKey, key -> new ExtensionClientDefinition());
            String relativePropertyName = matcher.group(2);
            String propertyValue = properties.getProperty(propertyName);
            switch (relativePropertyName) {
                case "type": {
                    definition.setExtensionClientType(propertyValue);
                    continue block12;
                }
                case "baseUrl": {
                    definition.setBaseUrl(propertyValue);
                    continue block12;
                }
                case "timeout": {
                    definition.setCommsTimeout(propertyValue);
                    continue block12;
                }
                case "useSslContext": {
                    definition.setUseSslContext(Boolean.parseBoolean(propertyValue));
                    continue block12;
                }
            }
            logger.warn("Encountered invalid property: <{}>. Will ignore this property.", (Object)propertyName);
        }
        return new ArrayList<ExtensionClientDefinition>(extensionClientDefinitions.values());
    }

    private SslContextDefinition parseSslContextDefinition(Properties properties) {
        String truststoreFile = properties.getProperty(TRUSTSTORE_FILE);
        if (truststoreFile == null || truststoreFile.trim().isEmpty()) {
            return null;
        }
        SslContextDefinition sslContextDefinition = new SslContextDefinition();
        sslContextDefinition.setTruststoreFile(truststoreFile);
        sslContextDefinition.setTruststorePass(properties.getProperty(TRUSTSTORE_PASSWORD));
        sslContextDefinition.setTruststoreType(properties.getProperty(TRUSTSTORE_TYPE));
        String keystoreFile = properties.getProperty(KEYSTORE_FILE);
        if (keystoreFile != null && !keystoreFile.trim().isEmpty()) {
            sslContextDefinition.setKeystoreFile(keystoreFile);
            sslContextDefinition.setKeystoreType(properties.getProperty(KEYSTORE_TYPE));
            String keystorePass = properties.getProperty(KEYSTORE_PASSWORD);
            sslContextDefinition.setKeystorePass(keystorePass);
            String explicitKeyPass = properties.getProperty(KEY_PASSWORD);
            String keyPass = explicitKeyPass == null || explicitKeyPass.trim().isEmpty() ? keystorePass : explicitKeyPass;
            sslContextDefinition.setKeyPass(keyPass);
        }
        return sslContextDefinition;
    }

    private String getRequired(Properties properties, String key) throws StatelessConfigurationException {
        String propertyValue = properties.getProperty(key);
        if (propertyValue == null || propertyValue.trim().isEmpty()) {
            throw new StatelessConfigurationException("Properties file is missing required property " + key);
        }
        return propertyValue.trim();
    }

    private String getSensitivePropsKey(File propertiesFile, Properties properties) {
        String sensitivePropsKey = properties.getProperty(SENSITIVE_PROPS_KEY);
        if (sensitivePropsKey == null || sensitivePropsKey.isEmpty()) {
            logger.warn("Generating Random Properties Encryption Key [{}]", (Object)SENSITIVE_PROPS_KEY);
            SecureRandom secureRandom = new SecureRandom();
            byte[] sensitivePropertiesKeyBinary = new byte[24];
            secureRandom.nextBytes(sensitivePropertiesKeyBinary);
            Base64.Encoder encoder = Base64.getEncoder().withoutPadding();
            sensitivePropsKey = encoder.encodeToString(sensitivePropertiesKeyBinary);
            properties.put(SENSITIVE_PROPS_KEY, sensitivePropsKey);
            try (FileOutputStream outputStream = new FileOutputStream(propertiesFile);){
                properties.store(outputStream, StatelessEngineConfiguration.class.getSimpleName());
            }
            catch (IOException e) {
                String message = String.format("Store Configuration Properties [%s] Failed", propertiesFile);
                throw new UncheckedIOException(message, e);
            }
        }
        return sensitivePropsKey;
    }
}

