/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import groovy.time.TimeCategory;
import groovy.time.TimeDuration;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.security.Provider;
import java.security.Security;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.crypto.Cipher;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.Processor;
import org.apache.nifi.processors.standard.EncryptContent;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.security.util.KeyDerivationFunction;
import org.apache.nifi.security.util.crypto.Argon2CipherProvider;
import org.apache.nifi.security.util.crypto.Argon2SecureHasher;
import org.apache.nifi.security.util.crypto.CipherUtility;
import org.apache.nifi.security.util.crypto.KeyedEncryptor;
import org.apache.nifi.security.util.crypto.PasswordBasedEncryptor;
import org.apache.nifi.security.util.crypto.RandomIVPBECipherProvider;
import org.apache.nifi.util.MockFlowFile;
import org.apache.nifi.util.MockProcessContext;
import org.apache.nifi.util.StringUtils;
import org.apache.nifi.util.TestRunner;
import org.apache.nifi.util.TestRunners;
import org.bouncycastle.bcpg.BCPGInputStream;
import org.bouncycastle.bcpg.SymmetricKeyEncSessionPacket;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPUtil;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TestEncryptContent {
    private static final List<EncryptionMethod> SUPPORTED_KEYED_ENCRYPTION_METHODS = Arrays.stream(EncryptionMethod.values()).filter(method -> method.isKeyedCipher() && method != EncryptionMethod.AES_CBC_NO_PADDING).collect(Collectors.toList());

    private static AllowableValue[] getPGPCipherList() {
        try {
            Method method = EncryptContent.class.getDeclaredMethod("buildPGPSymmetricCipherAllowableValues", new Class[0]);
            method.setAccessible(true);
            return (AllowableValue[])method.invoke(null, new Object[0]);
        }
        catch (Exception e) {
            Assertions.fail((String)"Cannot access buildPGPSymmetricCipherAllowableValues");
            return null;
        }
    }

    @BeforeEach
    public void setUp() {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }

    @Test
    public void testRoundTrip() throws IOException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.PASSWORD, "short");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NIFI_LEGACY.name());
        testRunner.setProperty(EncryptContent.ALLOW_WEAK_CRYPTO, "allowed");
        for (EncryptionMethod encryptionMethod : EncryptionMethod.values()) {
            if (encryptionMethod.isUnlimitedStrength() || encryptionMethod.isKeyedCipher()) continue;
            testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
            testRunner.setProperty(EncryptContent.MODE, "Encrypt");
            testRunner.enqueue(Paths.get("src/test/resources/hello.txt", new String[0]));
            testRunner.clearTransferState();
            testRunner.run();
            testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
            MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
            testRunner.assertQueueEmpty();
            testRunner.setProperty(EncryptContent.MODE, "Decrypt");
            testRunner.enqueue(new FlowFile[]{flowFile});
            testRunner.clearTransferState();
            testRunner.run();
            testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
            flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
            flowFile.assertContentEquals(new File("src/test/resources/hello.txt"));
        }
    }

    @Test
    public void testKeyedCiphersRoundTrip() throws IOException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        String RAW_KEY_HEX = StringUtils.repeat((String)"ab", (int)16);
        testRunner.setProperty(EncryptContent.RAW_KEY_HEX, RAW_KEY_HEX);
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
        for (EncryptionMethod encryptionMethod : SUPPORTED_KEYED_ENCRYPTION_METHODS) {
            testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
            testRunner.setProperty(EncryptContent.MODE, "Encrypt");
            testRunner.enqueue(Paths.get("src/test/resources/hello.txt", new String[0]));
            testRunner.clearTransferState();
            testRunner.run();
            testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
            MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
            testRunner.assertQueueEmpty();
            testRunner.setProperty(EncryptContent.MODE, "Decrypt");
            testRunner.enqueue(new FlowFile[]{flowFile});
            testRunner.clearTransferState();
            testRunner.run();
            testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
            flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
            flowFile.assertContentEquals(new File("src/test/resources/hello.txt"));
        }
    }

    @Test
    public void testPGPCiphersRoundTrip() {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.PASSWORD, "passwordpassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
        ArrayList<String> pgpAlgorithms = new ArrayList<String>();
        pgpAlgorithms.add("PGP");
        pgpAlgorithms.add("PGP_ASCII_ARMOR");
        for (String algorithm : pgpAlgorithms) {
            testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, algorithm);
            for (AllowableValue cipher : Objects.requireNonNull(TestEncryptContent.getPGPCipherList())) {
                testRunner.setProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER, cipher.getValue());
                testRunner.setProperty(EncryptContent.MODE, "Encrypt");
                testRunner.enqueue("A cool plaintext!");
                testRunner.clearTransferState();
                testRunner.run();
                testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
                MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
                testRunner.assertQueueEmpty();
                testRunner.setProperty(EncryptContent.MODE, "Decrypt");
                testRunner.setProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER, "1");
                testRunner.enqueue(new FlowFile[]{flowFile});
                testRunner.clearTransferState();
                testRunner.run();
                testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
                flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
                flowFile.assertContentEquals("A cool plaintext!");
            }
        }
    }

    @Test
    public void testPGPCiphers() throws Exception {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.PASSWORD, "passwordpassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
        ArrayList<String> pgpAlgorithms = new ArrayList<String>();
        pgpAlgorithms.add("PGP");
        pgpAlgorithms.add("PGP_ASCII_ARMOR");
        for (String algorithm : pgpAlgorithms) {
            testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, algorithm);
            for (AllowableValue cipher : Objects.requireNonNull(TestEncryptContent.getPGPCipherList())) {
                testRunner.setProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER, cipher.getValue());
                testRunner.setProperty(EncryptContent.MODE, "Encrypt");
                testRunner.enqueue("A cool plaintext!");
                testRunner.clearTransferState();
                testRunner.run();
                testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
                MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
                testRunner.assertQueueEmpty();
                ByteArrayInputStream ciphertext = new ByteArrayInputStream(flowFile.toByteArray());
                BCPGInputStream pgpin = new BCPGInputStream(PGPUtil.getDecoderStream((InputStream)ciphertext));
                assert (pgpin.nextPacketTag() == 3);
                assert (((SymmetricKeyEncSessionPacket)pgpin.readPacket()).getEncAlgorithm() == Integer.valueOf(cipher.getValue()).intValue());
                pgpin.close();
            }
        }
    }

    @Test
    public void testShouldDetermineMaxKeySizeForAlgorithms() {
        String AES_ALGORITHM = EncryptionMethod.MD5_256AES.getAlgorithm();
        String DES_ALGORITHM = EncryptionMethod.MD5_DES.getAlgorithm();
        int AES_MAX_LENGTH = Integer.MAX_VALUE;
        int DES_MAX_LENGTH = Integer.MAX_VALUE;
        int determinedAESMaxLength = PasswordBasedEncryptor.getMaxAllowedKeyLength((String)AES_ALGORITHM);
        int determinedTDESMaxLength = PasswordBasedEncryptor.getMaxAllowedKeyLength((String)DES_ALGORITHM);
        Assertions.assertEquals((int)Integer.MAX_VALUE, (int)determinedAESMaxLength);
        Assertions.assertEquals((int)Integer.MAX_VALUE, (int)determinedTDESMaxLength);
    }

    @Test
    public void testShouldDecryptOpenSSLRawSalted() throws IOException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        String password = "thisIsABadPassword";
        EncryptionMethod method = EncryptionMethod.MD5_256AES;
        KeyDerivationFunction kdf = KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY;
        testRunner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, kdf.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, method.name());
        testRunner.setProperty(EncryptContent.MODE, "Decrypt");
        testRunner.enqueue(Paths.get("src/test/resources/TestEncryptContent/salted_raw.enc", new String[0]));
        testRunner.clearTransferState();
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        testRunner.assertQueueEmpty();
        MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        flowFile.assertContentEquals(new File("src/test/resources/TestEncryptContent/plain.txt"));
    }

    @Test
    public void testShouldDecryptOpenSSLRawUnsalted() throws IOException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        String password = "thisIsABadPassword";
        EncryptionMethod method = EncryptionMethod.MD5_256AES;
        KeyDerivationFunction kdf = KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY;
        testRunner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, kdf.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, method.name());
        testRunner.setProperty(EncryptContent.MODE, "Decrypt");
        testRunner.enqueue(Paths.get("src/test/resources/TestEncryptContent/unsalted_raw.enc", new String[0]));
        testRunner.clearTransferState();
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        testRunner.assertQueueEmpty();
        MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        flowFile.assertContentEquals(new File("src/test/resources/TestEncryptContent/plain.txt"));
    }

    @Test
    public void testDecryptSmallerThanSaltSize() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.PASSWORD, "Hello, World!");
        runner.setProperty(EncryptContent.MODE, "Decrypt");
        runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NIFI_LEGACY.name());
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.MD5_128AES.name());
        runner.enqueue(new byte[4]);
        runner.run();
        runner.assertAllFlowFilesTransferred(EncryptContent.REL_FAILURE, 1);
    }

    @Test
    public void testPGPDecrypt() throws IOException {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.MODE, "Decrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP_ASCII_ARMOR.name());
        runner.setProperty(EncryptContent.PASSWORD, "Hello, World!");
        runner.enqueue(Paths.get("src/test/resources/TestEncryptContent/text.txt.asc", new String[0]));
        runner.run();
        runner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        MockFlowFile flowFile = (MockFlowFile)runner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        flowFile.assertContentEquals(Paths.get("src/test/resources/TestEncryptContent/text.txt", new String[0]));
    }

    @Test
    public void testShouldValidatePGPPublicKeyringRequiresUserId() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        runner.setProperty(EncryptContent.PUBLIC_KEYRING, "src/test/resources/TestEncryptContent/pubring.gpg");
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        ValidationResult vr = (ValidationResult)results.toArray()[0];
        String expectedResult = " encryption without a " + EncryptContent.PASSWORD.getDisplayName() + " requires both " + EncryptContent.PUBLIC_KEYRING.getDisplayName() + " and " + EncryptContent.PUBLIC_KEY_USERID.getDisplayName();
        String message = "'" + vr.toString() + "' contains '" + expectedResult + "'";
        Assertions.assertTrue((boolean)vr.toString().contains(expectedResult), (String)message);
    }

    @Test
    public void testShouldValidatePGPPublicKeyringExists() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        runner.setProperty(EncryptContent.PUBLIC_KEYRING, "src/test/resources/TestEncryptContent/pubring.gpg.missing");
        runner.setProperty(EncryptContent.PUBLIC_KEY_USERID, "USERID");
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        ValidationResult vr = (ValidationResult)results.toArray()[0];
        String expectedResult = "java.io.FileNotFoundException";
        String message = "'" + vr.toString() + "' contains '" + expectedResult + "'";
        Assertions.assertTrue((boolean)vr.toString().contains(expectedResult), (String)message);
    }

    @Test
    public void testShouldValidatePGPPublicKeyringIsProperFormat() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        runner.setProperty(EncryptContent.PUBLIC_KEYRING, "src/test/resources/TestEncryptContent/text.txt");
        runner.setProperty(EncryptContent.PUBLIC_KEY_USERID, "USERID");
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        ValidationResult vr = (ValidationResult)results.toArray()[0];
        String expectedResult = " java.io.IOException: invalid header encountered";
        String message = "'" + vr.toString() + "' contains '" + expectedResult + "'";
        Assertions.assertTrue((boolean)vr.toString().contains(expectedResult), (String)message);
    }

    @Test
    public void testShouldValidatePGPPublicKeyringContainsUserId() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        runner.setProperty(EncryptContent.PUBLIC_KEYRING, "src/test/resources/TestEncryptContent/pubring.gpg");
        runner.setProperty(EncryptContent.PUBLIC_KEY_USERID, "USERID");
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        ValidationResult vr = (ValidationResult)results.toArray()[0];
        String expectedResult = "PGPException: Could not find a public key with the given userId";
        String message = "'" + vr.toString() + "' contains '" + expectedResult + "'";
        Assertions.assertTrue((boolean)vr.toString().contains(expectedResult), (String)message);
    }

    @Test
    public void testShouldExtractPGPPublicKeyFromKeyring() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        runner.setProperty(EncryptContent.PUBLIC_KEYRING, "src/test/resources/TestEncryptContent/pubring.gpg");
        runner.setProperty(EncryptContent.PUBLIC_KEY_USERID, "NiFi PGP Test Key (Short test key for NiFi PGP unit tests) <alopresto.apache+test@gmail.com>");
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)0, (int)results.size());
    }

    @Test
    public void testValidation() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        String RAW_KEY_ERROR = "'raw-key-hex' is invalid because Raw Key (hexadecimal) is required when using algorithm AES/GCM/NoPadding and KDF KeyDerivationFunction[KDF Name=None,Description=The cipher is given a raw key conforming to the algorithm specifications]. See Admin Guide.";
        HashSet<String> EXPECTED_ERRORS = new HashSet<String>();
        EXPECTED_ERRORS.add("'raw-key-hex' is invalid because Raw Key (hexadecimal) is required when using algorithm AES/GCM/NoPadding and KDF KeyDerivationFunction[KDF Name=None,Description=The cipher is given a raw key conforming to the algorithm specifications]. See Admin Guide.");
        Assertions.assertEquals((int)EXPECTED_ERRORS.size(), (int)results.size(), (String)results.toString());
        for (ValidationResult vr : results) {
            Assertions.assertTrue((boolean)EXPECTED_ERRORS.contains(vr.toString()));
        }
        runner.enqueue(new byte[0]);
        EncryptionMethod encryptionMethod = EncryptionMethod.MD5_128AES;
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NIFI_LEGACY.name());
        runner.setProperty(EncryptContent.PASSWORD, "ThisIsAPasswordThatIsLongerThanSixteenCharacters");
        pc = (MockProcessContext)runner.getProcessContext();
        results = pc.validate();
        Assertions.assertEquals((int)0, (int)results.size(), (String)results.toString());
        runner.removeProperty(EncryptContent.PASSWORD);
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        runner.setProperty(EncryptContent.PUBLIC_KEYRING, "src/test/resources/TestEncryptContent/text.txt");
        runner.enqueue(new byte[0]);
        pc = (MockProcessContext)runner.getProcessContext();
        results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        for (ValidationResult vr : results) {
            Assertions.assertTrue((boolean)vr.toString().contains(" encryption without a " + EncryptContent.PASSWORD.getDisplayName() + " requires both " + EncryptContent.PUBLIC_KEYRING.getDisplayName() + " and " + EncryptContent.PUBLIC_KEY_USERID.getDisplayName()));
        }
        runner.removeProperty(EncryptContent.PUBLIC_KEYRING);
        runner.removeProperty(EncryptContent.PUBLIC_KEY_USERID);
        runner.setProperty(EncryptContent.MODE, "Decrypt");
        runner.setProperty(EncryptContent.PRIVATE_KEYRING, "src/test/resources/TestEncryptContent/secring.gpg");
        runner.enqueue(new byte[0]);
        pc = (MockProcessContext)runner.getProcessContext();
        results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        for (ValidationResult vr : results) {
            Assertions.assertTrue((boolean)vr.toString().contains(" decryption without a " + EncryptContent.PASSWORD.getDisplayName() + " requires both " + EncryptContent.PRIVATE_KEYRING.getDisplayName() + " and " + EncryptContent.PRIVATE_KEYRING_PASSPHRASE.getDisplayName()));
        }
        runner.setProperty(EncryptContent.PRIVATE_KEYRING_PASSPHRASE, "PASSWORD");
        runner.enqueue(new byte[0]);
        pc = (MockProcessContext)runner.getProcessContext();
        results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        for (ValidationResult vr : results) {
            Assertions.assertTrue((boolean)vr.toString().contains(" could not be opened with the provided " + EncryptContent.PRIVATE_KEYRING_PASSPHRASE.getDisplayName()));
        }
        runner.removeProperty(EncryptContent.PRIVATE_KEYRING_PASSPHRASE);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, "PGP");
        runner.setProperty(EncryptContent.PASSWORD, "PASSWORD");
        runner.setProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER, "256");
        runner.assertNotValid();
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, "PGP");
        runner.setProperty(EncryptContent.PASSWORD, "PASSWORD");
        runner.setProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER, "5");
        runner.assertNotValid();
        runner.setProperty(EncryptContent.MODE, "Decrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, "PGP");
        runner.setProperty(EncryptContent.PASSWORD, "PASSWORD");
        runner.removeProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER);
        runner.assertValid();
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, "PGP");
        runner.setProperty(EncryptContent.PASSWORD, "PASSWORD");
        runner.removeProperty(EncryptContent.PGP_SYMMETRIC_ENCRYPTION_CIPHER);
        runner.assertValid();
    }

    @Test
    void testShouldValidateMaxKeySizeForAlgorithmsOnUnlimitedStrengthJVM() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC;
        int MAX_KEY_LENGTH = Math.min(PasswordBasedEncryptor.getMaxAllowedKeyLength((String)encryptionMethod.getAlgorithm()), 256);
        String TOO_LONG_KEY_HEX = StringUtils.repeat((String)"ab", (int)(MAX_KEY_LENGTH / 8 + 1));
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
        runner.setProperty(EncryptContent.RAW_KEY_HEX, TOO_LONG_KEY_HEX);
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        ValidationResult vr = (ValidationResult)results.iterator().next();
        String expectedResult = "'raw-key-hex' is invalid because Key must be valid length [128, 192, 256]";
        String message = "'" + vr.toString() + "' contains '" + expectedResult + "'";
        Assertions.assertTrue((boolean)vr.toString().contains(expectedResult), (String)message);
    }

    @Test
    void testShouldValidateKeyFormatAndSizeForAlgorithms() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC;
        int INVALID_KEY_LENGTH = 120;
        String INVALID_KEY_HEX = StringUtils.repeat((String)"ab", (int)15);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
        runner.setProperty(EncryptContent.RAW_KEY_HEX, INVALID_KEY_HEX);
        runner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)1, (int)results.size());
        ValidationResult keyLengthInvalidVR = (ValidationResult)results.iterator().next();
        String expectedResult = "'raw-key-hex' is invalid because Key must be valid length [128, 192, 256]";
        String message = "'" + keyLengthInvalidVR.toString() + "' contains '" + expectedResult + "'";
        Assertions.assertTrue((boolean)keyLengthInvalidVR.toString().contains(expectedResult), (String)message);
    }

    @Test
    void testShouldValidateKDFWhenKeyedCipherSelected() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        int VALID_KEY_LENGTH = 128;
        String VALID_KEY_HEX = StringUtils.repeat((String)"ab", (int)16);
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        for (EncryptionMethod encryptionMethod : SUPPORTED_KEYED_ENCRYPTION_METHODS) {
            Collection results;
            MockProcessContext pc;
            runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
            List<KeyDerivationFunction> invalidKDFs = Arrays.asList(KeyDerivationFunction.NIFI_LEGACY, KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY);
            for (KeyDerivationFunction invalidKDF : invalidKDFs) {
                runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, invalidKDF.name());
                runner.setProperty(EncryptContent.RAW_KEY_HEX, VALID_KEY_HEX);
                runner.removeProperty(EncryptContent.PASSWORD);
                runner.enqueue(new byte[0]);
                pc = (MockProcessContext)runner.getProcessContext();
                results = pc.validate();
                Assertions.assertEquals((int)1, (int)results.size());
                ValidationResult keyLengthInvalidVR = (ValidationResult)results.iterator().next();
                String expectedResult = String.format("'key-derivation-function' is invalid because Key Derivation Function is required to be BCRYPT, SCRYPT, PBKDF2, ARGON2, NONE when using algorithm %s", encryptionMethod.getAlgorithm());
                String message = "'" + keyLengthInvalidVR.toString() + "' contains '" + expectedResult + "'";
                Assertions.assertTrue((boolean)keyLengthInvalidVR.toString().contains(expectedResult), (String)message);
            }
            runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
            runner.setProperty(EncryptContent.RAW_KEY_HEX, VALID_KEY_HEX);
            runner.removeProperty(EncryptContent.PASSWORD);
            runner.enqueue(new byte[0]);
            pc = (MockProcessContext)runner.getProcessContext();
            results = pc.validate();
            Assertions.assertTrue((boolean)results.isEmpty());
            List<KeyDerivationFunction> validKDFs = Arrays.asList(KeyDerivationFunction.BCRYPT, KeyDerivationFunction.SCRYPT, KeyDerivationFunction.PBKDF2, KeyDerivationFunction.ARGON2);
            for (KeyDerivationFunction validKDF : validKDFs) {
                runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, validKDF.name());
                runner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
                runner.removeProperty(EncryptContent.RAW_KEY_HEX);
                runner.enqueue(new byte[0]);
                pc = (MockProcessContext)runner.getProcessContext();
                results = pc.validate();
                Assertions.assertTrue((boolean)results.isEmpty());
            }
        }
    }

    @Test
    void testShouldValidateKeyMaterialSourceWhenKeyedCipherSelected() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        int VALID_KEY_LENGTH = 128;
        String VALID_KEY_HEX = StringUtils.repeat((String)"ab", (int)16);
        String VALID_PASSWORD = "thisIsABadPassword";
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        KeyDerivationFunction none = KeyDerivationFunction.NONE;
        for (EncryptionMethod kem : SUPPORTED_KEYED_ENCRYPTION_METHODS) {
            runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, kem.name());
            runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, none.name());
            runner.setProperty(EncryptContent.RAW_KEY_HEX, VALID_KEY_HEX);
            runner.removeProperty(EncryptContent.PASSWORD);
            runner.enqueue(new byte[0]);
            MockProcessContext pc = (MockProcessContext)runner.getProcessContext();
            Collection results = pc.validate();
            Assertions.assertTrue((boolean)results.isEmpty());
            List validKDFs = Arrays.stream(KeyDerivationFunction.values()).filter(it -> it.isStrongKDF()).collect(Collectors.toList());
            for (KeyDerivationFunction kdf : validKDFs) {
                runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, kem.name());
                runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, kdf.name());
                runner.removeProperty(EncryptContent.RAW_KEY_HEX);
                runner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
                runner.enqueue(new byte[0]);
                pc = (MockProcessContext)runner.getProcessContext();
                results = pc.validate();
                Assertions.assertTrue((boolean)results.isEmpty());
            }
        }
    }

    @Test
    void testShouldValidateKDFWhenPBECipherSelected() {
        TestRunner runner = TestRunners.newTestRunner(EncryptContent.class);
        String PASSWORD = "short";
        List encryptionMethods = Arrays.stream(EncryptionMethod.values()).filter(it -> it.getAlgorithm().startsWith("PBE")).collect(Collectors.toList());
        runner.setProperty(EncryptContent.MODE, "Encrypt");
        runner.setProperty(EncryptContent.PASSWORD, "short");
        runner.setProperty(EncryptContent.ALLOW_WEAK_CRYPTO, "allowed");
        for (EncryptionMethod encryptionMethod : encryptionMethods) {
            Collection results;
            MockProcessContext pc;
            runner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
            List<KeyDerivationFunction> invalidKDFs = Arrays.asList(KeyDerivationFunction.NONE, KeyDerivationFunction.BCRYPT, KeyDerivationFunction.SCRYPT, KeyDerivationFunction.PBKDF2, KeyDerivationFunction.ARGON2);
            for (KeyDerivationFunction invalidKDF : invalidKDFs) {
                runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, invalidKDF.name());
                runner.enqueue(new byte[0]);
                pc = (MockProcessContext)runner.getProcessContext();
                results = pc.validate();
                Assertions.assertEquals((int)1, (int)results.size());
                ValidationResult keyLengthInvalidVR = (ValidationResult)results.iterator().next();
                String expectedResult = String.format("'Key Derivation Function' is invalid because Key Derivation Function is required to be NIFI_LEGACY, OPENSSL_EVP_BYTES_TO_KEY when using algorithm %s", encryptionMethod.getAlgorithm());
                String message = "'" + keyLengthInvalidVR.toString() + "' contains '" + expectedResult + "'";
                Assertions.assertTrue((boolean)keyLengthInvalidVR.toString().contains(expectedResult), (String)message);
            }
            List<KeyDerivationFunction> validKDFs = Arrays.asList(KeyDerivationFunction.NIFI_LEGACY, KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY);
            for (KeyDerivationFunction validKDF : validKDFs) {
                runner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, validKDF.name());
                runner.enqueue(new byte[0]);
                pc = (MockProcessContext)runner.getProcessContext();
                results = pc.validate();
                Assertions.assertEquals((int)0, (int)results.size());
            }
        }
    }

    @Test
    void testDecryptAesCbcNoPadding() throws DecoderException, IOException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        String RAW_KEY_HEX = StringUtils.repeat((String)"ab", (int)16);
        testRunner.setProperty(EncryptContent.RAW_KEY_HEX, RAW_KEY_HEX);
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NONE.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.AES_CBC_NO_PADDING.name());
        testRunner.setProperty(EncryptContent.MODE, "Decrypt");
        String content = "ExactBlockSizeRequiredForProcess";
        byte[] bytes = "ExactBlockSizeRequiredForProcess".getBytes(StandardCharsets.UTF_8);
        ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        KeyedEncryptor encryptor = new KeyedEncryptor(EncryptionMethod.AES_CBC_NO_PADDING, Hex.decodeHex((String)RAW_KEY_HEX));
        encryptor.getEncryptionCallback().process((InputStream)inputStream, (OutputStream)outputStream);
        outputStream.close();
        byte[] encrypted = outputStream.toByteArray();
        testRunner.enqueue(encrypted);
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        flowFile.assertContentEquals("ExactBlockSizeRequiredForProcess");
    }

    @Test
    void testArgon2EncryptionShouldWriteAttributesWithEncryptionMetadata() throws ParseException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        KeyDerivationFunction kdf = KeyDerivationFunction.ARGON2;
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC;
        testRunner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, kdf.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        testRunner.setProperty(EncryptContent.MODE, "Encrypt");
        String PLAINTEXT = "This is a plaintext message. ";
        testRunner.enqueue(PLAINTEXT);
        testRunner.clearTransferState();
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        testRunner.assertQueueEmpty();
        byte[] flowfileContentBytes = flowFile.getData();
        int ivDelimiterStart = CipherUtility.findSequence((byte[])flowfileContentBytes, (byte[])RandomIVPBECipherProvider.IV_DELIMITER);
        byte[] EXPECTED_KDF_SALT_BYTES = TestEncryptContent.extractFullSaltFromCipherBytes(flowfileContentBytes);
        String EXPECTED_KDF_SALT = new String(EXPECTED_KDF_SALT_BYTES);
        String EXPECTED_SALT_HEX = TestEncryptContent.extractRawSaltHexFromFullSalt(EXPECTED_KDF_SALT_BYTES, kdf);
        String EXPECTED_IV_HEX = Hex.encodeHexString((byte[])Arrays.copyOfRange(flowfileContentBytes, ivDelimiterStart - 16, ivDelimiterStart));
        TimeDuration diff = TestEncryptContent.calculateTimestampDifference(new Date(), flowFile.getAttribute("encryptcontent.timestamp"));
        Assertions.assertTrue((diff.toMilliseconds() < 1000L ? 1 : 0) != 0);
        Assertions.assertEquals((Object)encryptionMethod.name(), (Object)flowFile.getAttribute("encryptcontent.algorithm"));
        Assertions.assertEquals((Object)kdf.name(), (Object)flowFile.getAttribute("encryptcontent.kdf"));
        Assertions.assertEquals((Object)"encrypted", (Object)flowFile.getAttribute("encryptcontent.action"));
        Assertions.assertEquals((Object)EXPECTED_SALT_HEX, (Object)flowFile.getAttribute("encryptcontent.salt"));
        Assertions.assertEquals((Object)"16", (Object)flowFile.getAttribute("encryptcontent.salt_length"));
        Assertions.assertEquals((Object)EXPECTED_KDF_SALT, (Object)flowFile.getAttribute("encryptcontent.kdf_salt"));
        int kdfSaltLength = Integer.valueOf(flowFile.getAttribute("encryptcontent.kdf_salt_length"));
        Assertions.assertTrue((kdfSaltLength >= 29 && kdfSaltLength <= 54 ? 1 : 0) != 0);
        Assertions.assertEquals((Object)EXPECTED_IV_HEX, (Object)flowFile.getAttribute("encryptcontent.iv"));
        Assertions.assertEquals((Object)"16", (Object)flowFile.getAttribute("encryptcontent.iv_length"));
        Assertions.assertEquals((Object)String.valueOf(PLAINTEXT.length()), (Object)flowFile.getAttribute("encryptcontent.plaintext_length"));
        Assertions.assertEquals((Object)String.valueOf(flowfileContentBytes.length), (Object)flowFile.getAttribute("encryptcontent.cipher_text_length"));
    }

    @Test
    void testKeyedEncryptionShouldWriteAttributesWithEncryptionMetadata() throws ParseException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        KeyDerivationFunction kdf = KeyDerivationFunction.NONE;
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC;
        testRunner.setProperty(EncryptContent.RAW_KEY_HEX, "0123456789ABCDEFFEDCBA9876543210");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, kdf.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        testRunner.setProperty(EncryptContent.MODE, "Encrypt");
        String PLAINTEXT = "This is a plaintext message. ";
        testRunner.enqueue(PLAINTEXT);
        testRunner.clearTransferState();
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        testRunner.assertQueueEmpty();
        byte[] flowfileContentBytes = flowFile.getData();
        int ivDelimiterStart = CipherUtility.findSequence((byte[])flowfileContentBytes, (byte[])RandomIVPBECipherProvider.IV_DELIMITER);
        Assertions.assertEquals((int)16, (int)ivDelimiterStart);
        TimeDuration diff = TestEncryptContent.calculateTimestampDifference(new Date(), flowFile.getAttribute("encryptcontent.timestamp"));
        Assertions.assertTrue((diff.toMilliseconds() < 1000L ? 1 : 0) != 0);
        String EXPECTED_IV_HEX = Hex.encodeHexString((byte[])Arrays.copyOfRange(flowfileContentBytes, 0, ivDelimiterStart));
        int EXPECTED_CIPHER_TEXT_LENGTH = CipherUtility.calculateCipherTextLength((int)PLAINTEXT.length(), (int)0);
        Assertions.assertEquals((Object)encryptionMethod.name(), (Object)flowFile.getAttribute("encryptcontent.algorithm"));
        Assertions.assertEquals((Object)kdf.name(), (Object)flowFile.getAttribute("encryptcontent.kdf"));
        Assertions.assertEquals((Object)"encrypted", (Object)flowFile.getAttribute("encryptcontent.action"));
        Assertions.assertEquals((Object)EXPECTED_IV_HEX, (Object)flowFile.getAttribute("encryptcontent.iv"));
        Assertions.assertEquals((Object)"16", (Object)flowFile.getAttribute("encryptcontent.iv_length"));
        Assertions.assertEquals((Object)String.valueOf(PLAINTEXT.length()), (Object)flowFile.getAttribute("encryptcontent.plaintext_length"));
        Assertions.assertEquals((Object)String.valueOf(EXPECTED_CIPHER_TEXT_LENGTH), (Object)flowFile.getAttribute("encryptcontent.cipher_text_length"));
    }

    @Test
    void testDifferentCompatibleConfigurations() throws Exception {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        KeyDerivationFunction argon2 = KeyDerivationFunction.ARGON2;
        EncryptionMethod aesCbcEM = EncryptionMethod.AES_CBC;
        int keyLength = CipherUtility.parseKeyLengthFromAlgorithm((String)aesCbcEM.getAlgorithm());
        String PASSWORD = "thisIsABadPassword";
        testRunner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, argon2.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, aesCbcEM.name());
        testRunner.setProperty(EncryptContent.MODE, "Encrypt");
        String PLAINTEXT = "This is a plaintext message. ";
        testRunner.enqueue(PLAINTEXT);
        testRunner.clearTransferState();
        testRunner.run();
        MockFlowFile encryptedFlowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        byte[] fullCipherBytes = encryptedFlowFile.getData();
        String argon2Salt = encryptedFlowFile.getAttribute("encryptcontent.kdf_salt");
        Argon2SecureHasher a2sh = new Argon2SecureHasher(Integer.valueOf(keyLength / 8));
        byte[] fullSaltBytes = argon2Salt.getBytes(StandardCharsets.UTF_8);
        byte[] rawSaltBytes = Hex.decodeHex((String)encryptedFlowFile.getAttribute("encryptcontent.salt"));
        byte[] keyBytes = a2sh.hashRaw("thisIsABadPassword".getBytes(StandardCharsets.UTF_8), rawSaltBytes);
        String keyHex = Hex.encodeHexString((byte[])keyBytes);
        byte[] ivBytes = Hex.decodeHex((String)encryptedFlowFile.getAttribute("encryptcontent.iv"));
        Argon2CipherProvider a2cp = new Argon2CipherProvider();
        Cipher sanityCipher = a2cp.getCipher(aesCbcEM, "thisIsABadPassword", fullSaltBytes, ivBytes, CipherUtility.parseKeyLengthFromAlgorithm((String)aesCbcEM.getAlgorithm()), false);
        byte[] cipherTextBytes = Arrays.copyOfRange(fullCipherBytes, fullCipherBytes.length - 32, fullCipherBytes.length);
        byte[] recoveredBytes = sanityCipher.doFinal(cipherTextBytes);
        KeyDerivationFunction kdf = KeyDerivationFunction.NONE;
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC;
        testRunner.setProperty(EncryptContent.RAW_KEY_HEX, keyHex);
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, kdf.name());
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        testRunner.setProperty(EncryptContent.MODE, "Decrypt");
        testRunner.removeProperty(EncryptContent.PASSWORD);
        testRunner.enqueue(fullCipherBytes);
        testRunner.clearTransferState();
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        MockFlowFile decryptedFlowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        testRunner.assertQueueEmpty();
        byte[] flowfileContentBytes = decryptedFlowFile.getData();
        Assertions.assertArrayEquals((byte[])recoveredBytes, (byte[])flowfileContentBytes);
    }

    @Test
    void testShouldCheckLengthOfPasswordWhenNotAllowed() {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NIFI_LEGACY.name());
        List encryptionMethods = Arrays.stream(EncryptionMethod.values()).filter(it -> it.getAlgorithm().startsWith("PBE")).collect(Collectors.toList());
        testRunner.setProperty(EncryptContent.ALLOW_WEAK_CRYPTO, "not-allowed");
        for (EncryptionMethod encryptionMethod : encryptionMethods) {
            int shortPasswordLength = Math.min(PasswordBasedEncryptor.getMinimumSafePasswordLength() - 1, CipherUtility.getMaximumPasswordLengthForAlgorithmOnLimitedStrengthCrypto((EncryptionMethod)encryptionMethod) - 1);
            String shortPassword = StringUtils.repeat((String)"x", (int)shortPasswordLength);
            if (encryptionMethod.isUnlimitedStrength() || encryptionMethod.isKeyedCipher()) continue;
            testRunner.setProperty(EncryptContent.PASSWORD, shortPassword);
            testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
            testRunner.setProperty(EncryptContent.MODE, "Encrypt");
            testRunner.clearTransferState();
            testRunner.enqueue(new byte[0]);
            MockProcessContext pc = (MockProcessContext)testRunner.getProcessContext();
            Collection results = pc.validate();
            Assertions.assertEquals((int)1, (int)results.size());
            ValidationResult passwordLengthVR = (ValidationResult)results.iterator().next();
            String expectedResult = String.format("'Password' is invalid because Password length less than %s characters is potentially unsafe. See Admin Guide.", PasswordBasedEncryptor.getMinimumSafePasswordLength());
            String message = "'" + passwordLengthVR.toString() + "' contains '" + expectedResult + "'";
            Assertions.assertTrue((boolean)passwordLengthVR.toString().contains(expectedResult), (String)message);
        }
    }

    @Test
    void testShouldNotCheckLengthOfPasswordWhenAllowed() {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.NIFI_LEGACY.name());
        List encryptionMethods = Arrays.stream(EncryptionMethod.values()).filter(it -> it.getAlgorithm().startsWith("PBE")).collect(Collectors.toList());
        testRunner.setProperty(EncryptContent.ALLOW_WEAK_CRYPTO, "allowed");
        for (EncryptionMethod encryptionMethod : encryptionMethods) {
            int shortPasswordLength = Math.min(PasswordBasedEncryptor.getMinimumSafePasswordLength() - 1, CipherUtility.getMaximumPasswordLengthForAlgorithmOnLimitedStrengthCrypto((EncryptionMethod)encryptionMethod) - 1);
            String shortPassword = StringUtils.repeat((String)"x", (int)shortPasswordLength);
            if (encryptionMethod.isUnlimitedStrength() || encryptionMethod.isKeyedCipher()) continue;
            testRunner.setProperty(EncryptContent.PASSWORD, shortPassword);
            testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
            testRunner.setProperty(EncryptContent.MODE, "Encrypt");
            testRunner.clearTransferState();
            testRunner.enqueue(new byte[0]);
            MockProcessContext pc = (MockProcessContext)testRunner.getProcessContext();
            Collection results = pc.validate();
            Assertions.assertEquals((int)0, (int)results.size(), (String)results.toString());
        }
    }

    @Test
    void testPGPPasswordShouldSupportExpressionLanguage() {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.MODE, "Decrypt");
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, EncryptionMethod.PGP.name());
        testRunner.setProperty(EncryptContent.PRIVATE_KEYRING, "src/test/resources/TestEncryptContent/secring.gpg");
        String passphraseWithoutEL = "thisIsABadPassword";
        testRunner.setProperty(EncryptContent.PRIVATE_KEYRING_PASSPHRASE, "thisIsABadPassword");
        testRunner.clearTransferState();
        testRunner.enqueue(new byte[0]);
        MockProcessContext pc = (MockProcessContext)testRunner.getProcessContext();
        Collection results = pc.validate();
        Assertions.assertEquals((int)0, (int)results.size(), (String)results.toString());
        String passphraseWithEL = "${literal('thisIsABadPassword')}";
        testRunner.setProperty(EncryptContent.PRIVATE_KEYRING_PASSPHRASE, "${literal('thisIsABadPassword')}");
        testRunner.clearTransferState();
        testRunner.enqueue(new byte[0]);
        results = pc.validate();
        Assertions.assertEquals((int)0, (int)results.size(), (String)results.toString());
    }

    @Test
    void testArgon2ShouldIncludeFullSalt() throws IOException {
        TestRunner testRunner = TestRunners.newTestRunner((Processor)new EncryptContent());
        testRunner.setProperty(EncryptContent.PASSWORD, "thisIsABadPassword");
        testRunner.setProperty(EncryptContent.KEY_DERIVATION_FUNCTION, KeyDerivationFunction.ARGON2.name());
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC;
        testRunner.setProperty(EncryptContent.ENCRYPTION_ALGORITHM, encryptionMethod.name());
        testRunner.setProperty(EncryptContent.MODE, "Encrypt");
        testRunner.enqueue(Paths.get("src/test/resources/hello.txt", new String[0]));
        testRunner.clearTransferState();
        testRunner.run();
        testRunner.assertAllFlowFilesTransferred(EncryptContent.REL_SUCCESS, 1);
        MockFlowFile flowFile = (MockFlowFile)testRunner.getFlowFilesForRelationship(EncryptContent.REL_SUCCESS).get(0);
        testRunner.assertQueueEmpty();
        String flowFileContent = flowFile.getContent();
        String fullSalt = flowFileContent.substring(0, flowFileContent.indexOf(new String(RandomIVPBECipherProvider.SALT_DELIMITER, StandardCharsets.UTF_8)));
        boolean isValidFormattedSalt = Argon2CipherProvider.isArgon2FormattedSalt((String)fullSalt);
        Assertions.assertTrue((boolean)isValidFormattedSalt);
        boolean fullSaltIsValidLength = fullSalt.getBytes().length >= 49 && fullSalt.getBytes().length <= 57;
        Assertions.assertTrue((boolean)fullSaltIsValidLength);
    }

    private static byte[] extractFullSaltFromCipherBytes(byte[] cipherBytes) {
        int saltDelimiterStart = CipherUtility.findSequence((byte[])cipherBytes, (byte[])RandomIVPBECipherProvider.SALT_DELIMITER);
        return Arrays.copyOfRange(cipherBytes, 0, saltDelimiterStart);
    }

    private static String extractRawSaltHexFromFullSalt(byte[] fullSaltBytes, KeyDerivationFunction kdf) {
        byte[] rawSaltBytes = CipherUtility.extractRawSalt((byte[])fullSaltBytes, (KeyDerivationFunction)kdf);
        String rawSaltHex = Hex.encodeHexString((byte[])rawSaltBytes);
        return rawSaltHex;
    }

    private static TimeDuration calculateTimestampDifference(Date date, String timestamp) throws ParseException {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS Z");
        Date parsedTimestamp = formatter.parse(timestamp);
        return TimeCategory.minus((Date)date, (Date)parsedTimestamp);
    }
}

