/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.snmp.processors;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.Validator;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.snmp.dto.SNMPSingleResponse;
import org.apache.nifi.snmp.dto.SNMPTreeResponse;
import org.apache.nifi.snmp.exception.SNMPWalkException;
import org.apache.nifi.snmp.operations.GetSNMPHandler;
import org.apache.nifi.snmp.processors.AbstractSNMPProcessor;
import org.apache.nifi.snmp.processors.properties.BasicProperties;
import org.apache.nifi.snmp.processors.properties.V3SecurityProperties;
import org.apache.nifi.snmp.validators.OIDValidator;

@Tags(value={"snmp", "get", "oid", "walk"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_ALLOWED)
@CapabilityDescription(value="Retrieves information from SNMP Agent with SNMP Get request and outputs a FlowFile with information in attributes and without any content")
@WritesAttributes(value={@WritesAttribute(attribute="snmp$<OID>", description="Response variable binding: OID (e.g. 1.3.6.1.4.1.343) and its value."), @WritesAttribute(attribute="snmp$errorIndex", description="Denotes the variable binding in which the error occured."), @WritesAttribute(attribute="snmp$errorStatus", description="The snmp4j error status of the PDU."), @WritesAttribute(attribute="snmp$errorStatusText", description="The description of error status."), @WritesAttribute(attribute="snmp$nonRepeaters", description="The number of non repeater variable bindings in a GETBULK PDU (currently not supported)."), @WritesAttribute(attribute="snmp$requestID", description="The request ID associated with the PDU."), @WritesAttribute(attribute="snmp$type", description="The snmp4j numeric representation of the type of the PDU."), @WritesAttribute(attribute="snmp$typeString", description="The name of the PDU type."), @WritesAttribute(attribute="snmp$textualOid", description="This attribute will exist if and only if the strategy is GET and will be equal to the value given in Textual Oid property.")})
public class GetSNMP
extends AbstractSNMPProcessor {
    public static final AllowableValue GET = new AllowableValue("GET", "GET", "A manager-to-agent request to retrieve the value of a variable. A response with the current value returned.");
    public static final AllowableValue WALK = new AllowableValue("WALK", "WALK", "A manager-to-agent request to retrieve the value of multiple variables. Snmp WALK also traverses all subnodes under the specified OID.");
    public static final PropertyDescriptor OID = new PropertyDescriptor.Builder().name("snmp-oid").displayName("OID").description("Each OID (object identifier) identifies a variable that can be read or set via SNMP. This value is not taken into account for an input flowfile and will be omitted. Can be set to emptystring when the OIDs are provided through flowfile.").required(true).addValidator((Validator)new OIDValidator()).build();
    public static final PropertyDescriptor SNMP_STRATEGY = new PropertyDescriptor.Builder().name("snmp-strategy").displayName("SNMP Strategy").description("SNMP strategy to use (SNMP Get or SNMP Walk)").required(true).allowableValues(new AllowableValue[]{GET, WALK}).defaultValue(GET.getValue()).build();
    public static final PropertyDescriptor TEXTUAL_OID = new PropertyDescriptor.Builder().name("snmp-textual-oid").displayName("Textual OID").description("The textual form of the numeric OID to request. This property is user defined, not processed and appended to the outgoing flowfile.").required(false).addValidator(StandardValidators.NON_BLANK_VALIDATOR).defaultValue(null).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("All FlowFiles that are received from the SNMP agent are routed to this relationship.").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("All FlowFiles that cannot received from the SNMP agent are routed to this relationship.").build();
    protected static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = Collections.unmodifiableList(Arrays.asList(AGENT_HOST, AGENT_PORT, BasicProperties.SNMP_VERSION, BasicProperties.SNMP_COMMUNITY, V3SecurityProperties.SNMP_SECURITY_LEVEL, V3SecurityProperties.SNMP_SECURITY_NAME, V3SecurityProperties.SNMP_AUTH_PROTOCOL, V3SecurityProperties.SNMP_AUTH_PASSWORD, V3SecurityProperties.SNMP_PRIVACY_PROTOCOL, V3SecurityProperties.SNMP_PRIVACY_PASSWORD, BasicProperties.SNMP_RETRIES, BasicProperties.SNMP_TIMEOUT, OID, TEXTUAL_OID, SNMP_STRATEGY));
    private static final Set<Relationship> RELATIONSHIPS = Collections.unmodifiableSet(new HashSet<Relationship>(Arrays.asList(REL_SUCCESS, REL_FAILURE)));
    private volatile GetSNMPHandler snmpHandler;

    @OnScheduled
    public void init(ProcessContext context) {
        this.initSnmpManager(context);
        this.snmpHandler = new GetSNMPHandler(this.snmpResourceHandler);
    }

    public void onTrigger(ProcessContext context, ProcessSession processSession) {
        SNMPStrategy snmpStrategy = SNMPStrategy.valueOf(context.getProperty(SNMP_STRATEGY).getValue());
        String oid = context.getProperty(OID).getValue();
        FlowFile flowfile = processSession.get();
        if (SNMPStrategy.GET == snmpStrategy) {
            this.performSnmpGet(context, processSession, oid, flowfile);
        } else if (SNMPStrategy.WALK == snmpStrategy) {
            this.performSnmpWalk(context, processSession, oid, flowfile);
        }
    }

    void performSnmpWalk(ProcessContext context, ProcessSession processSession, String oid, FlowFile flowFile) {
        try {
            if (flowFile != null) {
                this.performSnmpWalkWithFlowFile(processSession, flowFile);
            } else {
                this.performSnmpWalkWithoutFlowFile(processSession, oid);
            }
        }
        catch (SNMPWalkException e) {
            this.getLogger().error(e.getMessage());
            context.yield();
        }
    }

    private void performSnmpWalkWithFlowFile(ProcessSession processSession, FlowFile flowFile) {
        Optional<SNMPTreeResponse> optionalResponse = this.snmpHandler.walk(flowFile.getAttributes());
        if (optionalResponse.isPresent()) {
            SNMPTreeResponse response = optionalResponse.get();
            response.logErrors(this.getLogger());
            processSession.putAllAttributes(flowFile, response.getAttributes());
            processSession.getProvenanceReporter().modifyAttributes(flowFile, response.getTargetAddress() + "/walk");
            processSession.transfer(flowFile, response.isError() ? REL_FAILURE : REL_SUCCESS);
        } else {
            this.getLogger().warn("No SNMP specific attributes found in flowfile.");
            processSession.getProvenanceReporter().receive(flowFile, "/walk");
            processSession.transfer(flowFile, REL_FAILURE);
        }
    }

    private void performSnmpWalkWithoutFlowFile(ProcessSession processSession, String oid) {
        SNMPTreeResponse response = this.snmpHandler.walk(oid);
        response.logErrors(this.getLogger());
        FlowFile outgoingFlowFile = processSession.create();
        processSession.putAllAttributes(outgoingFlowFile, response.getAttributes());
        processSession.getProvenanceReporter().create(outgoingFlowFile, response.getTargetAddress() + "/walk");
        processSession.transfer(outgoingFlowFile, REL_SUCCESS);
    }

    void performSnmpGet(ProcessContext context, ProcessSession processSession, String oid, FlowFile flowFile) {
        String textualOidKey = "snmp$textualOid";
        Map<String, String> textualOidMap = Collections.singletonMap("snmp$textualOid", context.getProperty(TEXTUAL_OID).getValue());
        try {
            if (flowFile != null) {
                this.performSnmpGetWithFlowFile(context, processSession, flowFile, textualOidMap);
            } else {
                this.performSnmpGetWithoutFlowFile(context, processSession, oid, textualOidMap);
            }
        }
        catch (IOException e) {
            this.getLogger().error("Failed to send request to the agent. Check if the agent supports the used version.", (Throwable)e);
            context.yield();
        }
    }

    private void performSnmpGetWithoutFlowFile(ProcessContext context, ProcessSession processSession, String oid, Map<String, String> textualOidMap) throws IOException {
        SNMPSingleResponse response = this.snmpHandler.get(oid);
        FlowFile outgoingFlowFile = processSession.create();
        processSession.putAllAttributes(outgoingFlowFile, textualOidMap);
        processSession.getProvenanceReporter().receive(outgoingFlowFile, response.getTargetAddress() + "/get");
        this.handleResponse(context, processSession, outgoingFlowFile, response, REL_SUCCESS, REL_FAILURE, "/get");
    }

    private void performSnmpGetWithFlowFile(ProcessContext context, ProcessSession processSession, FlowFile flowFile, Map<String, String> textualOidMap) throws IOException {
        Optional<SNMPSingleResponse> optionalResponse = this.snmpHandler.get(flowFile.getAttributes());
        if (optionalResponse.isPresent()) {
            SNMPSingleResponse response = optionalResponse.get();
            processSession.putAllAttributes(flowFile, textualOidMap);
            this.handleResponse(context, processSession, flowFile, response, REL_SUCCESS, REL_FAILURE, "/get");
        } else {
            this.getLogger().warn("No SNMP specific attributes found in flowfile.");
            processSession.transfer(flowFile, REL_FAILURE);
            context.yield();
        }
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    @Override
    protected String getTargetHost(ProcessContext processContext) {
        return processContext.getProperty(AGENT_HOST).getValue();
    }

    @Override
    protected String getTargetPort(ProcessContext processContext) {
        return processContext.getProperty(AGENT_PORT).getValue();
    }

    private static enum SNMPStrategy {
        GET,
        WALK;

    }
}

