/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.poi;

import com.github.pjfanning.xlsx.StreamingReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.flowfile.attributes.FragmentAttributes;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.poi.ss.usermodel.CellCopyPolicy;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

@SideEffectFree
@SupportsBatching
@Tags(value={"split", "text"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@CapabilityDescription(value="Splits a multi sheet Microsoft Excel spreadsheet into multiple Microsoft Excel spreadsheets where each sheet from the original file is converted to an individual spreadsheet in its own flow file.  This processor is currently only capable of processing .xlsx (XSSF 2007 OOXML file format) Excel documents and not older .xls (HSSF '97(-2007) file format) documents. NOTE: All original cell styles are dropped when splitting into multiple Microsoft Excel spreadsheets.")
@WritesAttributes(value={@WritesAttribute(attribute="fragment.identifier", description="All split Excel FlowFiles produced from the same parent Excel FlowFile will have the same randomly generated UUID added for this attribute"), @WritesAttribute(attribute="fragment.index", description="A one-up number that indicates the ordering of the split Excel FlowFiles that were created from a single parent Excel FlowFile"), @WritesAttribute(attribute="fragment.count", description="The number of split Excel FlowFiles generated from the parent Excel FlowFile"), @WritesAttribute(attribute="segment.original.filename", description="The filename of the parent Excel FlowFile"), @WritesAttribute(attribute="sheetname", description="The name of the Excel sheet from the original spreadsheet."), @WritesAttribute(attribute="total.rows", description="The number of rows in the Excel sheet from the original spreadsheet.")})
public class SplitExcel
extends AbstractProcessor {
    public static final String SHEET_NAME = "sheetname";
    public static final String TOTAL_ROWS = "total.rows";
    public static final PropertyDescriptor PROTECTION_TYPE = new PropertyDescriptor.Builder().name("Protection Type").description("Specifies whether an Excel spreadsheet is protected by a password or not.").required(true).allowableValues(ProtectionType.class).defaultValue(ProtectionType.UNPROTECTED.getValue()).build();
    public static final PropertyDescriptor PASSWORD = new PropertyDescriptor.Builder().name("Password").description("The password for a password protected Excel spreadsheet").required(true).sensitive(true).addValidator(StandardValidators.NON_BLANK_VALIDATOR).dependsOn(PROTECTION_TYPE, (DescribedValue)ProtectionType.PASSWORD, new DescribedValue[0]).build();
    public static final Relationship REL_ORIGINAL = new Relationship.Builder().name("original").description("The original FlowFile that was split into segments. If the FlowFile fails processing, nothing will be sent to this relationship").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("If a FlowFile cannot be transformed from the configured input format to the configured output format, the unchanged FlowFile will be routed to this relationship.").build();
    public static final Relationship REL_SPLIT = new Relationship.Builder().name("split").description("The individual Excel 'segments' of the original Excel FlowFile will be routed to this relationship.").build();
    private static final List<PropertyDescriptor> DESCRIPTORS;
    private static final Set<Relationship> RELATIONSHIPS;
    private static final CellCopyPolicy CELL_COPY_POLICY;

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    public final List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return DESCRIPTORS;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        FlowFile originalFlowFile = session.get();
        if (originalFlowFile == null) {
            return;
        }
        String password = context.getProperty(PASSWORD).getValue();
        ArrayList workbookSplits = new ArrayList();
        try {
            session.read(originalFlowFile, in -> {
                Workbook originalWorkbook = StreamingReader.builder().rowCacheSize(100).bufferSize(4096).password(password).setReadHyperlinks(true).setReadSharedFormulas(true).open(in);
                int index = 0;
                for (Sheet originalSheet : originalWorkbook) {
                    String originalSheetName = originalSheet.getSheetName();
                    try (XSSFWorkbook newWorkbook = new XSSFWorkbook();){
                        XSSFSheet newSheet = newWorkbook.createSheet(originalSheetName);
                        ArrayList<Row> originalRows = new ArrayList<Row>();
                        for (Row originalRow : originalSheet) {
                            originalRows.add(originalRow);
                        }
                        if (!originalRows.isEmpty()) {
                            newSheet.copyRows(originalRows, originalSheet.getFirstRowNum(), CELL_COPY_POLICY);
                        }
                        FlowFile newFlowFile = session.create(originalFlowFile);
                        try (OutputStream out = session.write(newFlowFile);){
                            newWorkbook.write(out);
                            workbookSplits.add(new WorkbookSplit(index, newFlowFile, originalSheetName, originalRows.size()));
                        }
                    }
                    ++index;
                }
            });
        }
        catch (RuntimeException e) {
            this.getLogger().error("Failed to split {}", new Object[]{originalFlowFile, e});
            session.remove((Collection)workbookSplits.stream().map(WorkbookSplit::content).collect(Collectors.toList()));
            workbookSplits.clear();
            session.transfer(originalFlowFile, REL_FAILURE);
            return;
        }
        String fragmentId = UUID.randomUUID().toString();
        String originalFileName = originalFlowFile.getAttribute(CoreAttributes.FILENAME.key());
        int extensionIndex = originalFileName.lastIndexOf(".");
        String originalFileNameWithoutExtension = originalFileName;
        String originalFileNameExtension = "";
        if (extensionIndex > -1) {
            originalFileNameWithoutExtension = originalFileName.substring(0, extensionIndex);
            originalFileNameExtension = originalFileName.substring(extensionIndex);
        }
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put(FragmentAttributes.FRAGMENT_COUNT.key(), String.valueOf(workbookSplits.size()));
        attributes.put(FragmentAttributes.FRAGMENT_ID.key(), fragmentId);
        attributes.put(FragmentAttributes.SEGMENT_ORIGINAL_FILENAME.key(), originalFileName);
        for (WorkbookSplit split : workbookSplits) {
            attributes.put(CoreAttributes.FILENAME.key(), String.format("%s-%s%s", originalFileNameWithoutExtension, split.index(), originalFileNameExtension));
            attributes.put(FragmentAttributes.FRAGMENT_INDEX.key(), Integer.toString(split.index()));
            attributes.put(SHEET_NAME, split.sheetName());
            attributes.put(TOTAL_ROWS, Integer.toString(split.numRows()));
            session.putAllAttributes(split.content(), attributes);
        }
        session.transfer(originalFlowFile, REL_ORIGINAL);
        List flowFileSplits = workbookSplits.stream().map(WorkbookSplit::content).collect(Collectors.toList());
        session.transfer(flowFileSplits, REL_SPLIT);
    }

    static {
        CELL_COPY_POLICY = new CellCopyPolicy.Builder().cellFormula(true).cellStyle(false).cellValue(true).condenseRows(false).copyHyperlink(true).mergeHyperlink(false).mergedRegions(true).rowHeight(true).build();
        DESCRIPTORS = Collections.unmodifiableList(Arrays.asList(PROTECTION_TYPE, PASSWORD));
        RELATIONSHIPS = Collections.unmodifiableSet(new HashSet<Relationship>(Arrays.asList(REL_ORIGINAL, REL_FAILURE, REL_SPLIT)));
    }

    private static class WorkbookSplit {
        private final int index;
        private final FlowFile content;
        private final String sheetName;
        private final int numRows;

        public WorkbookSplit(int index, FlowFile content, String sheetName, int numRows) {
            this.index = index;
            this.content = content;
            this.sheetName = sheetName;
            this.numRows = numRows;
        }

        public int index() {
            return this.index;
        }

        public FlowFile content() {
            return this.content;
        }

        public String sheetName() {
            return this.sheetName;
        }

        public int numRows() {
            return this.numRows;
        }
    }

    public static enum ProtectionType implements DescribedValue
    {
        UNPROTECTED("Unprotected", "An Excel spreadsheet not protected by a password"),
        PASSWORD("Password Protected", "An Excel spreadsheet protected by a password");

        private final String displayName;
        private final String description;

        private ProtectionType(String displayName, String description) {
            this.displayName = displayName;
            this.description = description;
        }

        public String getValue() {
            return this.name();
        }

        public String getDisplayName() {
            return this.displayName;
        }

        public String getDescription() {
            return this.description;
        }
    }
}

