/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.server;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.StringEscapeUtils;
import org.apache.http.conn.util.InetAddressUtils;
import org.apache.nifi.util.NiFiProperties;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.ScopedHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HostHeaderHandler
extends ScopedHandler {
    private static final Logger logger = LoggerFactory.getLogger(HostHeaderHandler.class);
    private final String serverName;
    private final int serverPort;
    private final List<String> validHosts;

    public HostHeaderHandler(String serverName) {
        this(serverName, 0);
    }

    public HostHeaderHandler(String serverName, int serverPort) {
        this.serverName = Objects.requireNonNull(serverName);
        this.serverPort = serverPort;
        this.validHosts = HostHeaderHandler.generateDefaultHostnames(null);
        this.validHosts.add(serverName.toLowerCase());
        this.validHosts.add(serverName.toLowerCase() + ":" + serverPort);
        this.validHosts.add("localhost");
        this.validHosts.add("localhost:" + serverPort);
        this.validHosts.add("");
        try {
            this.validHosts.add(InetAddress.getLocalHost().getHostName().toLowerCase());
            this.validHosts.add(InetAddress.getLocalHost().getHostName().toLowerCase() + ":" + serverPort);
        }
        catch (Exception e) {
            logger.warn("Failed to determine local hostname.", (Throwable)e);
        }
        logger.info("Created " + this.toString());
    }

    public HostHeaderHandler(NiFiProperties niFiProperties) {
        this.serverName = Objects.requireNonNull(this.determineServerHostname(niFiProperties));
        this.serverPort = this.determineServerPort(niFiProperties);
        List<String> hosts = HostHeaderHandler.generateDefaultHostnames(niFiProperties);
        hosts.add(this.serverName.toLowerCase());
        hosts.add(this.serverName.toLowerCase() + ":" + this.serverPort);
        hosts.addAll(this.parseCustomHostnames(niFiProperties));
        hosts.add("");
        this.validHosts = HostHeaderHandler.uniqueList(hosts);
        logger.info("Determined {} valid hostnames and IP addresses for incoming headers: {}", (Object)this.validHosts.size(), (Object)StringUtils.join(this.validHosts, (String)", "));
        logger.debug("Created {}", (Object)this);
    }

    List<String> parseCustomHostnames(NiFiProperties niFiProperties) {
        List customHostnames = niFiProperties.getAllowedHostsAsList();
        List portlessHostnames = customHostnames.stream().map(hostname -> {
            if (HostHeaderHandler.isIPv6Address(hostname)) {
                return hostname;
            }
            return StringUtils.substringBeforeLast((String)hostname, (String)":");
        }).collect(Collectors.toList());
        customHostnames.addAll(portlessHostnames);
        if (logger.isDebugEnabled()) {
            logger.debug("Parsed {} custom hostnames from nifi.web.proxy.host: {}", (Object)customHostnames.size(), (Object)StringUtils.join((Iterable)customHostnames, (String)", "));
        }
        return HostHeaderHandler.uniqueList(customHostnames);
    }

    private static List<String> uniqueList(List<String> duplicateList) {
        return new ArrayList<String>(new LinkedHashSet<String>(duplicateList));
    }

    static boolean isIPv6Address(String address) {
        boolean isNormalIPv6 = InetAddressUtils.isIPv6Address((String)address);
        String everythingAfterLastColon = StringUtils.substringAfterLast((String)address, (String)":");
        boolean isIPv4 = InetAddressUtils.isIPv4Address((String)everythingAfterLastColon);
        boolean isIPv4Mapped = InetAddressUtils.isIPv4MappedIPv64Address((String)everythingAfterLastColon);
        boolean isCompressable = address.contains("0:0") && !address.contains("::");
        return isNormalIPv6 || isIPv4;
    }

    private int determineServerPort(NiFiProperties props) {
        return props.getSslPort() != null ? props.getSslPort() : props.getPort();
    }

    private String determineServerHostname(NiFiProperties props) {
        if (props.getSslPort() != null) {
            return props.getProperty("nifi.web.https.host", "localhost");
        }
        return props.getProperty("nifi.web.http.host", "localhost");
    }

    public void doScope(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        logger.debug("HostHeaderHandler#doScope on " + request.getRequestURI());
        this.nextScope(target, baseRequest, request, response);
    }

    boolean hostHeaderIsValid(String hostHeader) {
        return this.validHosts.contains(hostHeader.toLowerCase().trim());
    }

    public String toString() {
        return "HostHeaderHandler for " + this.serverName + ":" + this.serverPort;
    }

    public void doHandle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        String hostHeader = request.getHeader("Host");
        logger.debug("Received request [" + request.getRequestURI() + "] with host header: " + hostHeader);
        if (!this.hostHeaderIsValid(hostHeader)) {
            logger.warn("Request host header [" + hostHeader + "] different from web hostname [" + this.serverName + "(:" + this.serverPort + ")]. Overriding to [" + this.serverName + ":" + this.serverPort + request.getRequestURI() + "]");
            response.setContentType("text/html; charset=utf-8");
            response.setStatus(200);
            PrintWriter out = response.getWriter();
            out.println("<h1>System Error</h1>");
            out.println("<h2>The request contained an invalid host header [<code>" + StringEscapeUtils.escapeHtml4((String)hostHeader) + "</code>] in the request [<code>" + StringEscapeUtils.escapeHtml4((String)request.getRequestURI()) + "</code>]. Check for request manipulation or third-party intercept.</h2>");
            out.println("<h3>Valid host headers are [<code>empty</code>] or: <br/><code>");
            out.println(this.printValidHosts());
            out.println("</code></h3>");
            baseRequest.setHandled(true);
        }
    }

    String printValidHosts() {
        StringBuilder sb = new StringBuilder("<ul>");
        for (String vh : this.validHosts) {
            if (!StringUtils.isNotBlank((CharSequence)vh)) continue;
            sb.append("<li>").append(StringEscapeUtils.escapeHtml4((String)vh)).append("</li>\n");
        }
        return sb.append("</ul>\n").toString();
    }

    public static List<String> generateDefaultHostnames(NiFiProperties niFiProperties) {
        ArrayList<String> validHosts = new ArrayList<String>();
        int serverPort = 0;
        if (niFiProperties == null) {
            logger.warn("NiFiProperties not configured; returning minimal default hostnames");
        } else {
            try {
                serverPort = niFiProperties.getConfiguredHttpOrHttpsPort();
            }
            catch (RuntimeException e) {
                logger.warn("Cannot fully generate list of default hostnames because the server port is not configured in nifi.properties. Defaulting to port 0 for host header evaluation");
            }
            try {
                int lambdaPort = serverPort;
                List<String> customIPs = HostHeaderHandler.extractIPsFromNetworkInterfaces(niFiProperties);
                customIPs.forEach(ip -> {
                    validHosts.add((String)ip);
                    validHosts.add(ip + ":" + lambdaPort);
                });
            }
            catch (Exception e) {
                logger.warn("Failed to determine custom network interfaces.", (Throwable)e);
            }
        }
        validHosts.add("127.0.0.1");
        validHosts.add("127.0.0.1:" + serverPort);
        validHosts.add("localhost");
        validHosts.add("localhost:" + serverPort);
        validHosts.add("[::1]");
        validHosts.add("[::1]:" + serverPort);
        try {
            validHosts.add(InetAddress.getLoopbackAddress().getHostAddress().toLowerCase());
            validHosts.add(InetAddress.getLoopbackAddress().getHostAddress().toLowerCase() + ":" + serverPort);
            validHosts.add(InetAddress.getLocalHost().getHostName().toLowerCase());
            validHosts.add(InetAddress.getLocalHost().getHostName().toLowerCase() + ":" + serverPort);
            validHosts.add(InetAddress.getLocalHost().getHostAddress().toLowerCase());
            validHosts.add(InetAddress.getLocalHost().getHostAddress().toLowerCase() + ":" + serverPort);
        }
        catch (Exception e) {
            logger.warn("Failed to determine local hostname.", (Throwable)e);
        }
        List<String> uniqueHosts = HostHeaderHandler.uniqueList(validHosts);
        if (logger.isDebugEnabled()) {
            logger.debug("Determined {} valid default hostnames and IP addresses for incoming headers: {}", (Object)uniqueHosts.size(), (Object)StringUtils.join(uniqueHosts, (String)", "));
        }
        return uniqueHosts;
    }

    static List<String> extractIPsFromNetworkInterfaces(NiFiProperties niFiProperties) {
        Map networkInterfaces;
        Map map = networkInterfaces = niFiProperties.isHTTPSConfigured() ? niFiProperties.getHttpsNetworkInterfaces() : niFiProperties.getHttpNetworkInterfaces();
        if (HostHeaderHandler.isNotDefined(networkInterfaces)) {
            return Collections.emptyList();
        }
        ArrayList<String> allIPAddresses = new ArrayList<String>();
        for (Map.Entry entry : networkInterfaces.entrySet()) {
            String networkInterfaceName = (String)entry.getValue();
            try {
                NetworkInterface ni = NetworkInterface.getByName(networkInterfaceName);
                if (ni == null) {
                    logger.warn("Cannot resolve network interface named " + networkInterfaceName);
                    continue;
                }
                List ipAddresses = Collections.list(ni.getInetAddresses()).stream().map(inetAddress -> inetAddress.getHostAddress().toLowerCase()).collect(Collectors.toList());
                logger.debug("Resolved the following IP addresses for network interface {}: {}", (Object)networkInterfaceName, (Object)StringUtils.join(ipAddresses, (String)", "));
                allIPAddresses.addAll(ipAddresses);
            }
            catch (SocketException e) {
                logger.warn("Cannot resolve network interface named " + networkInterfaceName);
            }
        }
        return HostHeaderHandler.uniqueList(allIPAddresses);
    }

    static boolean isNotDefined(Map<String, String> networkInterfaces) {
        return networkInterfaces == null || networkInterfaces.isEmpty() || networkInterfaces.values().stream().filter(value -> StringUtils.isNotBlank((CharSequence)value)).collect(Collectors.toList()).isEmpty();
    }
}

