/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.parameter.gcp;

import com.google.cloud.secretmanager.v1.AccessSecretVersionResponse;
import com.google.cloud.secretmanager.v1.ProjectName;
import com.google.cloud.secretmanager.v1.Secret;
import com.google.cloud.secretmanager.v1.SecretManagerServiceClient;
import com.google.cloud.secretmanager.v1.SecretManagerServiceSettings;
import com.google.cloud.secretmanager.v1.SecretVersionName;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.ConfigVerificationResult;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.gcp.credentials.service.GCPCredentialsService;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.parameter.AbstractParameterProvider;
import org.apache.nifi.parameter.Parameter;
import org.apache.nifi.parameter.ParameterDescriptor;
import org.apache.nifi.parameter.ParameterGroup;
import org.apache.nifi.parameter.VerifiableParameterProvider;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Tags(value={"gcp", "secret", "manager"})
@CapabilityDescription(value="Fetches parameters from GCP Secret Manager.  Each secret becomes a Parameter, which can be mapped to a Parameter Group by adding a GCP label named 'group-name'.")
public class GcpSecretManagerParameterProvider
extends AbstractParameterProvider
implements VerifiableParameterProvider {
    private static final Logger logger = LoggerFactory.getLogger(GcpSecretManagerParameterProvider.class);
    public static final PropertyDescriptor GROUP_NAME_PATTERN = new PropertyDescriptor.Builder().name("group-name-pattern").displayName("Group Name Pattern").description("A Regular Expression matching on the 'group-name' label value that identifies Secrets whose parameters should be fetched. Any secrets without a 'group-name' label value that matches this Regex will not be fetched.").addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).required(true).defaultValue(".*").build();
    public static final PropertyDescriptor PROJECT_ID = new PropertyDescriptor.Builder().name("gcp-project-id").displayName("Project ID").description("Google Cloud Project ID").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).required(true).build();
    public static final PropertyDescriptor GCP_CREDENTIALS_PROVIDER_SERVICE = new PropertyDescriptor.Builder().name("gcp-credentials-provider-service").displayName("GCP Credentials Provider Service").description("The Controller Service used to obtain Google Cloud Platform credentials.").required(true).identifiesControllerService(GCPCredentialsService.class).build();
    private static final String GROUP_NAME_LABEL = "group-name";
    private static final String SECRETS_PATH = "secrets/";
    private static final List<PropertyDescriptor> PROPERTIES = Collections.unmodifiableList(Arrays.asList(GROUP_NAME_PATTERN, PROJECT_ID, GCP_CREDENTIALS_PROVIDER_SERVICE));

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTIES;
    }

    public List<ParameterGroup> fetchParameters(ConfigurationContext context) throws IOException {
        HashMap<String, ParameterGroup> providedParameterGroups = new HashMap<String, ParameterGroup>();
        SecretManagerServiceClient secretsManager = this.configureClient(context);
        ProjectName projectName = ProjectName.of((String)context.getProperty(PROJECT_ID).getValue());
        SecretManagerServiceClient.ListSecretsPagedResponse pagedResponse = secretsManager.listSecrets(projectName);
        SecretManagerServiceClient.ListSecretsPage page = (SecretManagerServiceClient.ListSecretsPage)pagedResponse.getPage();
        do {
            for (Secret secret : page.getValues()) {
                String contextName = secret.getLabelsOrDefault(GROUP_NAME_LABEL, null);
                if (contextName == null) {
                    this.getLogger().debug("Secret [{}] does not have the {} label, and will be skipped", new Object[]{secret.getName(), GROUP_NAME_LABEL});
                    continue;
                }
                String secretName = StringUtils.substringAfter((String)secret.getName(), (String)SECRETS_PATH);
                this.fetchSecret(secretsManager, context, secretName, contextName, providedParameterGroups);
            }
            if (!page.hasNextPage()) continue;
            page = (SecretManagerServiceClient.ListSecretsPage)page.getNextPage();
        } while (page.hasNextPage());
        return new ArrayList<ParameterGroup>(providedParameterGroups.values());
    }

    public List<ConfigVerificationResult> verify(ConfigurationContext context, ComponentLog verificationLogger) {
        ArrayList<ConfigVerificationResult> results = new ArrayList<ConfigVerificationResult>();
        try {
            List<ParameterGroup> parameterGroups = this.fetchParameters(context);
            int count = 0;
            for (ParameterGroup group : parameterGroups) {
                count += group.getParameters().size();
            }
            results.add(new ConfigVerificationResult.Builder().outcome(ConfigVerificationResult.Outcome.SUCCESSFUL).verificationStepName("Fetch Parameters").explanation(String.format("Fetched secret keys [%d] as parameters within groups [%d]", count, parameterGroups.size())).build());
        }
        catch (Exception e) {
            verificationLogger.error("Failed to fetch parameters", (Throwable)e);
            results.add(new ConfigVerificationResult.Builder().outcome(ConfigVerificationResult.Outcome.FAILED).verificationStepName("Fetch Parameters").explanation("Failed to fetch parameters: " + e.getMessage()).build());
        }
        return results;
    }

    private void fetchSecret(SecretManagerServiceClient secretsManager, ConfigurationContext context, String secretName, String groupName, Map<String, ParameterGroup> providedParameterGroups) {
        Pattern groupNamePattern = Pattern.compile(context.getProperty(GROUP_NAME_PATTERN).getValue());
        String projectId = context.getProperty(PROJECT_ID).getValue();
        if (!groupNamePattern.matcher(groupName).matches()) {
            logger.debug("Secret [{}] label [{}] does not match the group name pattern {}", new Object[]{secretName, groupName, groupNamePattern});
            return;
        }
        SecretVersionName secretVersionName = SecretVersionName.of((String)projectId, (String)secretName, (String)"latest");
        AccessSecretVersionResponse response = secretsManager.accessSecretVersion(secretVersionName);
        String parameterValue = response.getPayload().getData().toStringUtf8();
        Parameter parameter = this.createParameter(secretName, parameterValue);
        if (parameter != null) {
            ParameterGroup group = providedParameterGroups.computeIfAbsent(groupName, key -> new ParameterGroup(groupName, new ArrayList()));
            ArrayList<Parameter> updatedParameters = new ArrayList<Parameter>(group.getParameters());
            updatedParameters.add(parameter);
            providedParameterGroups.put(groupName, new ParameterGroup(groupName, updatedParameters));
        }
    }

    private Parameter createParameter(String parameterName, String parameterValue) {
        ParameterDescriptor parameterDescriptor = new ParameterDescriptor.Builder().name(parameterName).build();
        return new Parameter(parameterDescriptor, parameterValue, null, Boolean.valueOf(true));
    }

    SecretManagerServiceClient configureClient(ConfigurationContext context) throws IOException {
        GCPCredentialsService credentialsService = (GCPCredentialsService)context.getProperty(GCP_CREDENTIALS_PROVIDER_SERVICE).asControllerService(GCPCredentialsService.class);
        SecretManagerServiceClient client = SecretManagerServiceClient.create((SecretManagerServiceSettings)((SecretManagerServiceSettings.Builder)SecretManagerServiceSettings.newBuilder().setCredentialsProvider(() -> credentialsService.getGoogleCredentials())).build());
        return client;
    }
}

