/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hive.ql.parse;

import junit.framework.Assert;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.Context;
import org.apache.hadoop.hive.ql.ErrorMsg;
import org.apache.hadoop.hive.ql.exec.Task;
import org.apache.hadoop.hive.ql.io.orc.OrcInputFormat;
import org.apache.hadoop.hive.ql.io.orc.OrcOutputFormat;
import org.apache.hadoop.hive.ql.metadata.Hive;
import org.apache.hadoop.hive.ql.metadata.Table;
import org.apache.hadoop.hive.ql.plan.AlterTableSimpleDesc;
import org.apache.hadoop.hive.ql.plan.DDLWork;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.junit.BeforeClass;
import org.junit.Test;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Tests for parsing and semantic analysis of ALTER TABLE ... compact.
 */
public class TestQBCompact {
  static HiveConf conf;

  @BeforeClass
  public static void init() throws Exception {
    conf = new HiveConf();
    conf.set("fs.default.name", "file:///");
    SessionState.start(conf);

    // Create a table so we can work against it
    Hive h = Hive.get(conf);
    List<String> cols = new ArrayList<String>();
    cols.add("a");
    List<String> partCols = new ArrayList<String>();
    partCols.add("ds");
    h.createTable("foo", cols, partCols, OrcInputFormat.class, OrcOutputFormat.class);
    Table t = h.getTable("foo");
    Map<String, String> partSpec = new HashMap<String, String>();
    partSpec.put("ds", "today");
    h.createPartition(t, partSpec);
  }

  private AlterTableSimpleDesc parseAndAnalyze(String query) throws Exception {
    ParseDriver hd = new ParseDriver();
    ASTNode head = (ASTNode)hd.parse(query).getChild(0);
    BaseSemanticAnalyzer a = SemanticAnalyzerFactory.get(conf, head);
    a.analyze(head, new Context(conf));
    List<Task<? extends Serializable>> roots = a.getRootTasks();
    Assert.assertEquals(1, roots.size());
    return ((DDLWork)roots.get(0).getWork()).getAlterTblSimpleDesc();
  }


  @Test
  public void testNonPartitionedTable() throws Exception {
    boolean sawException = false;
    AlterTableSimpleDesc desc = parseAndAnalyze("alter table foo compact 'major'");
    Assert.assertEquals("major", desc.getCompactionType());
    Assert.assertEquals("default.foo", desc.getTableName());
  }

  @Test
  public void testBogusLevel() throws Exception {
    boolean sawException = false;
    try {
      parseAndAnalyze("alter table foo partition(ds = 'today') compact 'bogus'");
    } catch (SemanticException e) {
      sawException = true;
      Assert.assertEquals(ErrorMsg.INVALID_COMPACTION_TYPE.getMsg(), e.getMessage());
    }
    Assert.assertTrue(sawException);
  }

  @Test
  public void testMajor() throws Exception {
    AlterTableSimpleDesc desc =
        parseAndAnalyze("alter table foo partition(ds = 'today') compact 'major'");
    Assert.assertEquals("major", desc.getCompactionType());
    Assert.assertEquals("default.foo", desc.getTableName());
    HashMap<String, String> parts = desc.getPartSpec();
    Assert.assertEquals(1, parts.size());
    Assert.assertEquals("today", parts.get("ds"));
  }

  @Test
  public void testMinor() throws Exception {
    AlterTableSimpleDesc desc =
        parseAndAnalyze("alter table foo partition(ds = 'today') compact 'minor'");
    Assert.assertEquals("minor", desc.getCompactionType());
    Assert.assertEquals("default.foo", desc.getTableName());
    HashMap<String, String> parts = desc.getPartSpec();
    Assert.assertEquals(1, parts.size());
    Assert.assertEquals("today", parts.get("ds"));
  }

  @Test
  public void showCompactions() throws Exception {
    parseAndAnalyze("show compactions");
  }

  @Test
  public void showTxns() throws Exception {
    parseAndAnalyze("show transactions");
  }
}
