/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.metadata;

import com.google.common.collect.Sets;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.common.StringInternUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.ql.log.PerfLogger;
import org.apache.hadoop.hive.ql.metadata.CheckResult;
import org.apache.hadoop.hive.ql.metadata.Hive;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.metadata.Partition;
import org.apache.hadoop.hive.ql.metadata.PartitionIterable;
import org.apache.hadoop.hive.ql.metadata.Table;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveMetaStoreChecker {
    public static final Logger LOG = LoggerFactory.getLogger(HiveMetaStoreChecker.class);
    public static final String CLASS_NAME = HiveMetaStoreChecker.class.getName();
    private final Hive hive;
    private final HiveConf conf;

    public HiveMetaStoreChecker(Hive hive) {
        this.hive = hive;
        this.conf = hive.getConf();
    }

    public void checkMetastore(String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws HiveException, IOException {
        if (dbName == null || "".equalsIgnoreCase(dbName)) {
            dbName = "default";
        }
        try {
            if (tableName == null || "".equals(tableName)) {
                List<String> tables = this.hive.getTablesForDb(dbName, ".*");
                for (String currentTableName : tables) {
                    this.checkTable(dbName, currentTableName, null, result);
                }
                this.findUnknownTables(dbName, tables, result);
            } else if (partitions == null || partitions.isEmpty()) {
                this.checkTable(dbName, tableName, null, result);
            } else {
                this.checkTable(dbName, tableName, partitions, result);
            }
            LOG.info("Number of partitionsNotInMs=" + result.getPartitionsNotInMs() + ", partitionsNotOnFs=" + result.getPartitionsNotOnFs() + ", tablesNotInMs=" + result.getTablesNotInMs() + ", tablesNotOnFs=" + result.getTablesNotOnFs());
        }
        catch (MetaException e) {
            throw new HiveException(e);
        }
        catch (TException e) {
            throw new HiveException(e);
        }
    }

    void findUnknownTables(String dbName, List<String> tables, CheckResult result) throws IOException, MetaException, TException, HiveException {
        HashSet<Path> dbPaths = new HashSet<Path>();
        HashSet<String> tableNames = new HashSet<String>(tables);
        for (String tableName : tables) {
            Table table = this.hive.getTable(dbName, tableName);
            String isExternal = table.getParameters().get("EXTERNAL");
            if (isExternal != null && "TRUE".equalsIgnoreCase(isExternal)) continue;
            dbPaths.add(table.getPath().getParent());
        }
        for (Path dbPath : dbPaths) {
            FileStatus[] statuses;
            FileSystem fs = dbPath.getFileSystem((Configuration)this.conf);
            for (FileStatus status : statuses = fs.listStatus(dbPath, FileUtils.HIDDEN_FILES_PATH_FILTER)) {
                if (!status.isDir() || tableNames.contains(status.getPath().getName())) continue;
                result.getTablesNotInMs().add(status.getPath().getName());
            }
        }
    }

    void checkTable(String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws MetaException, IOException, HiveException {
        PartitionIterable parts;
        Table table = null;
        try {
            table = this.hive.getTable(dbName, tableName);
        }
        catch (HiveException e) {
            result.getTablesNotInMs().add(tableName);
            return;
        }
        boolean findUnknownPartitions = true;
        if (table.isPartitioned()) {
            if (partitions == null || partitions.isEmpty()) {
                String mode = HiveConf.getVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVEMAPREDMODE, (String)null);
                if ("strict".equalsIgnoreCase(mode)) {
                    parts = new PartitionIterable(this.hive, table, null, this.conf.getIntVar(HiveConf.ConfVars.METASTORE_BATCH_RETRIEVE_MAX));
                } else {
                    ArrayList<Partition> loadedPartitions = new ArrayList<Partition>();
                    PerfLogger perfLogger = SessionState.getPerfLogger();
                    perfLogger.PerfLogBegin(CLASS_NAME, "partition-retrieving");
                    loadedPartitions.addAll(this.hive.getAllPartitionsOf(table));
                    perfLogger.PerfLogEnd(CLASS_NAME, "partition-retrieving");
                    parts = new PartitionIterable(loadedPartitions);
                }
            } else {
                findUnknownPartitions = false;
                ArrayList<Partition> loadedPartitions = new ArrayList<Partition>();
                for (Map<String, String> map : partitions) {
                    Partition part = this.hive.getPartition(table, map, false);
                    if (part == null) {
                        CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                        pr.setTableName(tableName);
                        pr.setPartitionName(Warehouse.makePartPath(map));
                        result.getPartitionsNotInMs().add(pr);
                        continue;
                    }
                    loadedPartitions.add(part);
                }
                parts = new PartitionIterable(loadedPartitions);
            }
        } else {
            parts = new PartitionIterable(Collections.emptyList());
        }
        this.checkTable(table, parts, findUnknownPartitions, result);
    }

    void checkTable(Table table, PartitionIterable parts, boolean findUnknownPartitions, CheckResult result) throws IOException, HiveException {
        Path tablePath = table.getPath();
        FileSystem fs = tablePath.getFileSystem((Configuration)this.conf);
        if (!fs.exists(tablePath)) {
            result.getTablesNotOnFs().add(table.getTableName());
            return;
        }
        HashSet<Path> partPaths = new HashSet<Path>();
        for (Partition partition : parts) {
            if (partition == null) continue;
            Path partPath = partition.getDataLocation();
            fs = partPath.getFileSystem((Configuration)this.conf);
            if (!fs.exists(partPath)) {
                CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                pr.setPartitionName(partition.getName());
                pr.setTableName(partition.getTable().getTableName());
                result.getPartitionsNotOnFs().add(pr);
            }
            for (int i = 0; i < partition.getSpec().size(); ++i) {
                Path qualifiedPath = partPath.makeQualified(fs);
                StringInternUtils.internUriStringsInPath((Path)qualifiedPath);
                partPaths.add(qualifiedPath);
                partPath = partPath.getParent();
            }
        }
        if (findUnknownPartitions) {
            this.findUnknownPartitions(table, partPaths, result);
        }
    }

    void findUnknownPartitions(Table table, Set<Path> partPaths, CheckResult result) throws IOException, HiveException {
        Path tablePath = table.getPath();
        HashSet<Path> allPartDirs = new HashSet<Path>();
        this.checkPartitionDirs(tablePath, allPartDirs, Collections.unmodifiableList(table.getPartColNames()));
        allPartDirs.remove(tablePath);
        allPartDirs.removeAll(partPaths);
        HashSet partColNames = Sets.newHashSet();
        for (FieldSchema fSchema : table.getPartCols()) {
            partColNames.add(fSchema.getName());
        }
        for (Path partPath : allPartDirs) {
            FileSystem fs = partPath.getFileSystem((Configuration)this.conf);
            String partitionName = HiveMetaStoreChecker.getPartitionName(fs.makeQualified(tablePath), partPath, partColNames);
            LOG.debug("PartitionName: " + partitionName);
            if (partitionName == null) continue;
            CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
            pr.setPartitionName(partitionName);
            pr.setTableName(table.getTableName());
            result.getPartitionsNotInMs().add(pr);
        }
        LOG.debug("Number of partitions not in metastore : " + result.getPartitionsNotInMs().size());
    }

    static String getPartitionName(Path tablePath, Path partitionPath, Set<String> partCols) {
        Object result = null;
        LOG.debug("tablePath:" + tablePath + ", partCols: " + partCols);
        for (Path currPath = partitionPath; currPath != null && !tablePath.equals((Object)currPath); currPath = currPath.getParent()) {
            String[] parts = currPath.getName().split("=");
            if (parts != null && parts.length > 0) {
                if (parts.length != 2) {
                    LOG.warn(currPath.getName() + " is not a valid partition name");
                    return result;
                }
                String partitionName = parts[0];
                if (partCols.contains(partitionName)) {
                    result = result == null ? currPath.getName() : currPath.getName() + "/" + (String)result;
                }
            }
            LOG.debug("currPath=" + currPath);
        }
        return result;
    }

    private void checkPartitionDirs(Path basePath, Set<Path> allDirs, List<String> partColNames) throws IOException, HiveException {
        Object executor;
        int poolSize = this.conf.getInt(HiveConf.ConfVars.METASTORE_FS_HANDLER_THREADS_COUNT.varname, 15);
        if (poolSize <= 1) {
            LOG.debug("Using single-threaded version of MSCK-GetPaths");
            executor = MoreExecutors.newDirectExecutorService();
        } else {
            LOG.debug("Using multi-threaded version of MSCK-GetPaths with number of threads " + poolSize);
            ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("MSCK-GetPaths-%d").build();
            executor = (ThreadPoolExecutor)Executors.newFixedThreadPool(poolSize, threadFactory);
        }
        this.checkPartitionDirs((ExecutorService)executor, basePath, allDirs, basePath.getFileSystem((Configuration)this.conf), partColNames);
        executor.shutdown();
    }

    private void checkPartitionDirs(ExecutorService executor, Path basePath, Set<Path> result, FileSystem fs, List<String> partColNames) throws HiveException {
        try {
            LinkedList<Future<Path>> futures = new LinkedList<Future<Path>>();
            ConcurrentLinkedQueue<PathDepthInfo> nextLevel = new ConcurrentLinkedQueue<PathDepthInfo>();
            nextLevel.add(new PathDepthInfo(basePath, 0));
            while (!nextLevel.isEmpty()) {
                ConcurrentLinkedQueue<PathDepthInfo> tempQueue = new ConcurrentLinkedQueue<PathDepthInfo>();
                while (!nextLevel.isEmpty()) {
                    futures.add(executor.submit(new PathDepthInfoCallable((PathDepthInfo)nextLevel.poll(), partColNames, fs, tempQueue)));
                }
                while (!futures.isEmpty()) {
                    Path p = (Path)((Future)futures.poll()).get();
                    if (p == null) continue;
                    result.add(p);
                }
                nextLevel = tempQueue;
            }
        }
        catch (InterruptedException | ExecutionException e) {
            LOG.error(e.getMessage());
            executor.shutdownNow();
            throw new HiveException(e.getCause());
        }
    }

    private static class PathDepthInfo {
        private final Path p;
        private final int depth;

        PathDepthInfo(Path p, int depth) {
            this.p = p;
            this.depth = depth;
        }
    }

    private final class PathDepthInfoCallable
    implements Callable<Path> {
        private final List<String> partColNames;
        private final FileSystem fs;
        private final ConcurrentLinkedQueue<PathDepthInfo> pendingPaths;
        private final boolean throwException;
        private final PathDepthInfo pd;

        private PathDepthInfoCallable(PathDepthInfo pd, List<String> partColNames, FileSystem fs, ConcurrentLinkedQueue<PathDepthInfo> basePaths) {
            this.partColNames = partColNames;
            this.pd = pd;
            this.fs = fs;
            this.pendingPaths = basePaths;
            this.throwException = "throw".equals(HiveConf.getVar((Configuration)HiveMetaStoreChecker.this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_MSCK_PATH_VALIDATION));
        }

        @Override
        public Path call() throws Exception {
            return this.processPathDepthInfo(this.pd);
        }

        private Path processPathDepthInfo(PathDepthInfo pd) throws IOException, HiveException, InterruptedException {
            Path currentPath = pd.p;
            int currentDepth = pd.depth;
            FileStatus[] fileStatuses = this.fs.listStatus(currentPath, FileUtils.HIDDEN_FILES_PATH_FILTER);
            if (fileStatuses.length == 0 && currentDepth > 0 && currentDepth < this.partColNames.size()) {
                this.logOrThrowExceptionWithMsg("MSCK is missing partition columns under " + currentPath.toString());
            } else {
                for (FileStatus fileStatus : fileStatuses) {
                    if (!fileStatus.isDirectory() && currentDepth < this.partColNames.size()) {
                        this.logOrThrowExceptionWithMsg("MSCK finds a file rather than a directory when it searches for " + fileStatus.getPath().toString());
                        continue;
                    }
                    if (!fileStatus.isDirectory() || currentDepth >= this.partColNames.size()) continue;
                    Path nextPath = fileStatus.getPath();
                    String[] parts = nextPath.getName().split("=");
                    if (parts.length != 2) {
                        this.logOrThrowExceptionWithMsg("Invalid partition name " + nextPath);
                        continue;
                    }
                    if (!parts[0].equalsIgnoreCase(this.partColNames.get(currentDepth))) {
                        this.logOrThrowExceptionWithMsg("Unexpected partition key " + parts[0] + " found at " + nextPath);
                        continue;
                    }
                    this.pendingPaths.add(new PathDepthInfo(nextPath, currentDepth + 1));
                }
                if (currentDepth == this.partColNames.size()) {
                    return currentPath;
                }
            }
            return null;
        }

        private void logOrThrowExceptionWithMsg(String msg) throws HiveException {
            if (this.throwException) {
                throw new HiveException(msg);
            }
            LOG.warn(msg);
        }
    }
}

