/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.txn.compactor;

import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.Comparator;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.common.ValidTxnList;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.api.CompactionRequest;
import org.apache.hadoop.hive.metastore.api.CompactionResponse;
import org.apache.hadoop.hive.metastore.api.CompactionType;
import org.apache.hadoop.hive.metastore.api.GetOpenTxnsInfoResponse;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.ShowCompactRequest;
import org.apache.hadoop.hive.metastore.api.ShowCompactResponse;
import org.apache.hadoop.hive.metastore.api.ShowCompactResponseElement;
import org.apache.hadoop.hive.metastore.api.StorageDescriptor;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.metastore.txn.CompactionInfo;
import org.apache.hadoop.hive.metastore.txn.TxnStore;
import org.apache.hadoop.hive.metastore.txn.TxnUtils;
import org.apache.hadoop.hive.ql.io.AcidUtils;
import org.apache.hadoop.hive.ql.txn.compactor.CompactorThread;
import org.apache.hadoop.hive.ql.txn.compactor.Worker;
import org.apache.hadoop.hive.shims.HadoopShims;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Initiator
extends CompactorThread {
    private static final String CLASS_NAME = Initiator.class.getName();
    private static final Logger LOG = LoggerFactory.getLogger((String)CLASS_NAME);
    private static final String COMPACTORTHRESHOLD_PREFIX = "compactorthreshold.";
    private long checkInterval;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            this.recoverFailedCompactions(false);
            int abortedThreshold = HiveConf.getIntVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_COMPACTOR_ABORTEDTXN_THRESHOLD);
            do {
                long startedAt = -1L;
                TxnStore.MutexAPI.LockHandle handle = null;
                try {
                    handle = this.txnHandler.getMutexAPI().acquireLock(TxnStore.MUTEX_KEY.Initiator.name());
                    startedAt = System.currentTimeMillis();
                    ShowCompactResponse currentCompactions = this.txnHandler.showCompact(new ShowCompactRequest());
                    ValidTxnList txns = TxnUtils.createValidCompactTxnList((GetOpenTxnsInfoResponse)this.txnHandler.getOpenTxnsInfo());
                    Set potentials = this.txnHandler.findPotentialCompactions(abortedThreshold);
                    LOG.debug("Found " + potentials.size() + " potential compactions, checking to see if we should compact any of them");
                    for (CompactionInfo ci : potentials) {
                        LOG.info("Checking to see if we should compact " + ci.getFullPartitionName());
                        try {
                            Table t = this.resolveTable(ci);
                            if (t == null) {
                                LOG.info("Can't find table " + ci.getFullTableName() + ", assuming it's a temp table or has been dropped and moving on.");
                                continue;
                            }
                            if (this.noAutoCompactSet(t)) {
                                LOG.info("Table " + this.tableName(t) + " marked " + "no_auto_compaction" + "=true so we will not compact it.");
                                continue;
                            }
                            if (t.getPartitionKeys() != null && t.getPartitionKeys().size() > 0 && ci.partName == null) {
                                LOG.debug("Skipping entry for " + ci.getFullTableName() + " as it is from dynamic partitioning");
                                continue;
                            }
                            if (this.foundCurrentOrFailedCompactions(currentCompactions, ci)) {
                                LOG.debug("Found currently initiated or working compaction for " + ci.getFullPartitionName() + " so we will not initiate another compaction");
                                continue;
                            }
                            Partition p = this.resolvePartition(ci);
                            if (p == null && ci.partName != null) {
                                LOG.info("Can't find partition " + ci.getFullPartitionName() + ", assuming it has been dropped and moving on.");
                                continue;
                            }
                            StorageDescriptor sd = this.resolveStorageDescriptor(t, p);
                            String runAs = this.findUserToRunAs(sd.getLocation(), t);
                            CompactionType compactionNeeded = this.checkForCompaction(ci, txns, sd, t.getParameters(), runAs);
                            if (compactionNeeded == null) continue;
                            this.requestCompaction(ci, runAs, compactionNeeded);
                        }
                        catch (Throwable t) {
                            LOG.error("Caught exception while trying to determine if we should compact " + ci + ".  Marking failed to avoid repeated failures, " + StringUtils.stringifyException((Throwable)t));
                            this.txnHandler.markFailed(ci);
                        }
                    }
                    this.recoverFailedCompactions(true);
                    this.txnHandler.cleanEmptyAbortedTxns();
                }
                catch (Throwable t) {
                    LOG.error("Initiator loop caught unexpected exception this time through the loop: " + StringUtils.stringifyException((Throwable)t));
                }
                finally {
                    if (handle != null) {
                        handle.releaseLocks();
                    }
                }
                long elapsedTime = System.currentTimeMillis() - startedAt;
                if (elapsedTime >= this.checkInterval || this.stop.get()) continue;
                Thread.sleep(this.checkInterval - elapsedTime);
            } while (!this.stop.get());
        }
        catch (Throwable t) {
            LOG.error("Caught an exception in the main loop of compactor initiator, exiting " + StringUtils.stringifyException((Throwable)t));
        }
    }

    @Override
    public void init(AtomicBoolean stop, AtomicBoolean looped) throws MetaException {
        super.init(stop, looped);
        this.checkInterval = this.conf.getTimeVar(HiveConf.ConfVars.HIVE_COMPACTOR_CHECK_INTERVAL, TimeUnit.MILLISECONDS);
    }

    private void recoverFailedCompactions(boolean remoteOnly) throws MetaException {
        if (!remoteOnly) {
            this.txnHandler.revokeFromLocalWorkers(Worker.hostname());
        }
        this.txnHandler.revokeTimedoutWorkers(HiveConf.getTimeVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_COMPACTOR_WORKER_TIMEOUT, (TimeUnit)TimeUnit.MILLISECONDS));
    }

    private boolean foundCurrentOrFailedCompactions(ShowCompactResponse compactions, CompactionInfo ci) throws MetaException {
        boolean needsRetry;
        if (compactions.getCompacts() == null) {
            return false;
        }
        List filteredElements = compactions.getCompacts().stream().filter(e -> e.getDbname().equals(ci.dbname) && e.getTablename().equals(ci.tableName) && (e.getPartitionname() == null && ci.partName == null || e.getPartitionname().equals(ci.partName))).collect(Collectors.toList());
        if (filteredElements.stream().anyMatch(e -> "working".equals(e.getState()) || "initiated".equals(e.getState()))) {
            LOG.info("Found currently initiated or working compaction for " + ci.getFullPartitionName() + " so we will not initiate another compaction");
            return true;
        }
        int failedThreshold = this.conf.getIntVar(HiveConf.ConfVars.COMPACTOR_INITIATOR_FAILED_THRESHOLD);
        LongSummaryStatistics failedStats = filteredElements.stream().filter(e -> "succeeded".equals(e.getState()) || "failed".equals(e.getState())).sorted(Comparator.comparingLong(ShowCompactResponseElement::getId).reversed()).limit(failedThreshold).filter(e -> "failed".equals(e.getState())).collect(Collectors.summarizingLong(ShowCompactResponseElement::getStart));
        long retryTime = this.conf.getTimeVar(HiveConf.ConfVars.COMPACTOR_INITIATOR_FAILED_RETRY_TIME, TimeUnit.MILLISECONDS);
        boolean bl = needsRetry = retryTime > 0L && failedStats.getMax() + retryTime < System.currentTimeMillis();
        if (failedStats.getCount() == (long)failedThreshold && !needsRetry) {
            LOG.warn("Will not initiate compaction for " + ci.getFullPartitionName() + " since last " + HiveConf.ConfVars.COMPACTOR_INITIATOR_FAILED_THRESHOLD + " attempts to compact it failed.");
            this.txnHandler.markFailed(ci);
            return true;
        }
        return false;
    }

    private CompactionType checkForCompaction(final CompactionInfo ci, final ValidTxnList txns, final StorageDescriptor sd, final Map<String, String> tblproperties, String runAs) throws IOException, InterruptedException {
        if (ci.tooManyAborts) {
            LOG.debug("Found too many aborted transactions for " + ci.getFullPartitionName() + ", initiating major compaction");
            return CompactionType.MAJOR;
        }
        if (this.runJobAsSelf(runAs)) {
            return this.determineCompactionType(ci, txns, sd, tblproperties);
        }
        LOG.info("Going to initiate as user " + runAs);
        UserGroupInformation ugi = UserGroupInformation.createProxyUser((String)runAs, (UserGroupInformation)UserGroupInformation.getLoginUser());
        CompactionType compactionType = (CompactionType)ugi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<CompactionType>(){

            @Override
            public CompactionType run() throws Exception {
                return Initiator.this.determineCompactionType(ci, txns, sd, tblproperties);
            }
        });
        try {
            FileSystem.closeAllForUGI((UserGroupInformation)ugi);
        }
        catch (IOException exception) {
            LOG.error("Could not clean up file-system handles for UGI: " + ugi + " for " + ci.getFullPartitionName(), (Throwable)exception);
        }
        return compactionType;
    }

    private CompactionType determineCompactionType(CompactionInfo ci, ValidTxnList txns, StorageDescriptor sd, Map<String, String> tblproperties) throws IOException, InterruptedException {
        boolean enough;
        boolean noBase = false;
        Path location = new Path(sd.getLocation());
        FileSystem fs = location.getFileSystem((Configuration)this.conf);
        AcidUtils.Directory dir = AcidUtils.getAcidState(location, (Configuration)this.conf, txns, false, false);
        Path base = dir.getBaseDirectory();
        long baseSize = 0L;
        FileStatus stat = null;
        if (base != null) {
            stat = fs.getFileStatus(base);
            if (!stat.isDir()) {
                LOG.error("Was assuming base " + base.toString() + " is directory, but it's a file!");
                return null;
            }
            baseSize = this.sumDirSize(fs, base);
        }
        List<HadoopShims.HdfsFileStatusWithId> originals = dir.getOriginalFiles();
        for (HadoopShims.HdfsFileStatusWithId origStat : originals) {
            baseSize += origStat.getFileStatus().getLen();
        }
        long deltaSize = 0L;
        List<AcidUtils.ParsedDelta> deltas = dir.getCurrentDirectories();
        for (AcidUtils.ParsedDelta delta : deltas) {
            stat = fs.getFileStatus(delta.getPath());
            if (!stat.isDir()) {
                LOG.error("Was assuming delta " + delta.getPath().toString() + " is a directory, but it's a file!");
                return null;
            }
            deltaSize += this.sumDirSize(fs, delta.getPath());
        }
        if (baseSize == 0L && deltaSize > 0L) {
            noBase = true;
        } else {
            boolean bigEnough;
            String deltaPctProp = tblproperties.get(COMPACTORTHRESHOLD_PREFIX + HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_PCT_THRESHOLD);
            float deltaPctThreshold = deltaPctProp == null ? HiveConf.getFloatVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_PCT_THRESHOLD) : Float.parseFloat(deltaPctProp);
            boolean bl = bigEnough = (float)deltaSize / (float)baseSize > deltaPctThreshold;
            if (LOG.isDebugEnabled()) {
                StringBuilder msg = new StringBuilder("delta size: ");
                msg.append(deltaSize);
                msg.append(" base size: ");
                msg.append(baseSize);
                msg.append(" threshold: ");
                msg.append(deltaPctThreshold);
                msg.append(" will major compact: ");
                msg.append(bigEnough);
                LOG.debug(msg.toString());
            }
            if (bigEnough) {
                return CompactionType.MAJOR;
            }
        }
        String deltaNumProp = tblproperties.get(COMPACTORTHRESHOLD_PREFIX + HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_NUM_THRESHOLD);
        int deltaNumThreshold = deltaNumProp == null ? HiveConf.getIntVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_NUM_THRESHOLD) : Integer.parseInt(deltaNumProp);
        boolean bl = enough = deltas.size() > deltaNumThreshold;
        if (enough) {
            LOG.debug("Found " + deltas.size() + " delta files, threshold is " + deltaNumThreshold + (enough ? "" : "not") + " and no base, requesting " + (noBase ? "major" : "minor") + " compaction");
            return noBase ? CompactionType.MAJOR : CompactionType.MINOR;
        }
        return null;
    }

    private long sumDirSize(FileSystem fs, Path dir) throws IOException {
        long size = 0L;
        FileStatus[] buckets = fs.listStatus(dir, FileUtils.HIDDEN_FILES_PATH_FILTER);
        for (int i = 0; i < buckets.length; ++i) {
            size += buckets[i].getLen();
        }
        return size;
    }

    private void requestCompaction(CompactionInfo ci, String runAs, CompactionType type) throws MetaException {
        CompactionRequest rqst = new CompactionRequest(ci.dbname, ci.tableName, type);
        if (ci.partName != null) {
            rqst.setPartitionname(ci.partName);
        }
        rqst.setRunas(runAs);
        LOG.info("Requesting compaction: " + rqst);
        CompactionResponse resp = this.txnHandler.compact(rqst);
        if (resp.isAccepted()) {
            ci.id = resp.getId();
        }
    }

    private boolean noAutoCompactSet(Table t) {
        String noAutoCompact = (String)t.getParameters().get("no_auto_compaction");
        if (noAutoCompact == null) {
            noAutoCompact = (String)t.getParameters().get("no_auto_compaction".toUpperCase());
        }
        return noAutoCompact != null && noAutoCompact.equalsIgnoreCase("true");
    }
}

