/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.analysis.differentiation;

import org.apache.commons.math3.TestUtils;
import org.apache.commons.math3.analysis.QuinticFunction;
import org.apache.commons.math3.analysis.UnivariateFunction;
import org.apache.commons.math3.analysis.UnivariateMatrixFunction;
import org.apache.commons.math3.analysis.UnivariateVectorFunction;
import org.apache.commons.math3.analysis.differentiation.DerivativeStructure;
import org.apache.commons.math3.analysis.differentiation.FiniteDifferencesDifferentiator;
import org.apache.commons.math3.analysis.differentiation.UnivariateDifferentiableFunction;
import org.apache.commons.math3.analysis.differentiation.UnivariateDifferentiableMatrixFunction;
import org.apache.commons.math3.analysis.differentiation.UnivariateDifferentiableVectorFunction;
import org.apache.commons.math3.analysis.function.Gaussian;
import org.apache.commons.math3.analysis.function.Sin;
import org.apache.commons.math3.exception.MathInternalError;
import org.apache.commons.math3.exception.NotPositiveException;
import org.apache.commons.math3.exception.NumberIsTooLargeException;
import org.apache.commons.math3.exception.NumberIsTooSmallException;
import org.apache.commons.math3.util.FastMath;
import org.junit.Assert;
import org.junit.Test;

public class FiniteDifferencesDifferentiatorTest {
    @Test(expected=NumberIsTooSmallException.class)
    public void testWrongNumberOfPoints() {
        new FiniteDifferencesDifferentiator(1, 1.0);
    }

    @Test(expected=NotPositiveException.class)
    public void testWrongStepSize() {
        new FiniteDifferencesDifferentiator(3, 0.0);
    }

    @Test
    public void testSerialization() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(3, 0.001);
        FiniteDifferencesDifferentiator recovered = (FiniteDifferencesDifferentiator)TestUtils.serializeAndRecover(differentiator);
        Assert.assertEquals((long)differentiator.getNbPoints(), (long)recovered.getNbPoints());
        Assert.assertEquals((double)differentiator.getStepSize(), (double)recovered.getStepSize(), (double)1.0E-15);
    }

    @Test
    public void testConstant() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(5, 0.01);
        UnivariateDifferentiableFunction f = differentiator.differentiate(new UnivariateFunction(){

            public double value(double x) {
                return 42.0;
            }
        });
        for (double x = -10.0; x < 10.0; x += 0.1) {
            DerivativeStructure y = f.value(new DerivativeStructure(1, 2, 0, x));
            Assert.assertEquals((double)42.0, (double)y.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)0.0, (double)y.getPartialDerivative(new int[]{1}), (double)1.0E-15);
            Assert.assertEquals((double)0.0, (double)y.getPartialDerivative(new int[]{2}), (double)1.0E-15);
        }
    }

    @Test
    public void testLinear() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(5, 0.01);
        UnivariateDifferentiableFunction f = differentiator.differentiate(new UnivariateFunction(){

            public double value(double x) {
                return 2.0 - 3.0 * x;
            }
        });
        for (double x = -10.0; x < 10.0; x += 0.1) {
            DerivativeStructure y = f.value(new DerivativeStructure(1, 2, 0, x));
            Assert.assertEquals((String)("" + (2.0 - 3.0 * x - y.getValue())), (double)(2.0 - 3.0 * x), (double)y.getValue(), (double)2.0E-15);
            Assert.assertEquals((double)-3.0, (double)y.getPartialDerivative(new int[]{1}), (double)4.0E-13);
            Assert.assertEquals((double)0.0, (double)y.getPartialDerivative(new int[]{2}), (double)9.0E-11);
        }
    }

    @Test
    public void testGaussian() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(9, 0.02);
        Gaussian gaussian = new Gaussian(1.0, 2.0);
        UnivariateDifferentiableFunction f = differentiator.differentiate((UnivariateFunction)gaussian);
        double[] expectedError = new double[]{6.939E-18, 1.284E-15, 2.477E-13, 1.168E-11, 2.84E-9, 7.971E-8};
        double[] maxError = new double[expectedError.length];
        for (double x = -10.0; x < 10.0; x += 0.1) {
            DerivativeStructure dsX = new DerivativeStructure(1, maxError.length - 1, 0, x);
            DerivativeStructure yRef = gaussian.value(dsX);
            DerivativeStructure y = f.value(dsX);
            Assert.assertEquals((double)f.value(dsX.getValue()), (double)f.value(dsX).getValue(), (double)1.0E-15);
            for (int order = 0; order <= yRef.getOrder(); ++order) {
                maxError[order] = FastMath.max((double)maxError[order], (double)FastMath.abs((double)(yRef.getPartialDerivative(new int[]{order}) - y.getPartialDerivative(new int[]{order}))));
            }
        }
        for (int i = 0; i < maxError.length; ++i) {
            Assert.assertEquals((double)expectedError[i], (double)maxError[i], (double)(0.01 * expectedError[i]));
        }
    }

    @Test
    public void testStepSizeUnstability() {
        QuinticFunction quintic = new QuinticFunction();
        UnivariateDifferentiableFunction goodStep = new FiniteDifferencesDifferentiator(7, 0.25).differentiate((UnivariateFunction)quintic);
        UnivariateDifferentiableFunction badStep = new FiniteDifferencesDifferentiator(7, 1.0E-6).differentiate((UnivariateFunction)quintic);
        double[] maxErrorGood = new double[7];
        double[] maxErrorBad = new double[7];
        for (double x = -10.0; x < 10.0; x += 0.1) {
            DerivativeStructure dsX = new DerivativeStructure(1, 6, 0, x);
            DerivativeStructure yRef = quintic.value(dsX);
            DerivativeStructure yGood = goodStep.value(dsX);
            DerivativeStructure yBad = badStep.value(dsX);
            for (int order = 0; order <= 6; ++order) {
                maxErrorGood[order] = FastMath.max((double)maxErrorGood[order], (double)FastMath.abs((double)(yRef.getPartialDerivative(new int[]{order}) - yGood.getPartialDerivative(new int[]{order}))));
                maxErrorBad[order] = FastMath.max((double)maxErrorBad[order], (double)FastMath.abs((double)(yRef.getPartialDerivative(new int[]{order}) - yBad.getPartialDerivative(new int[]{order}))));
            }
        }
        double[] expectedGood = new double[]{7.276E-12, 7.276E-11, 9.968E-10, 3.092E-9, 5.432E-8, 8.196E-8, 1.818E-6};
        double[] expectedBad = new double[]{2.91E-11, 2.087E-5, 147.7, 3.82E7, 6.354E14, 6.548E19, 1.543E27};
        for (int i = 0; i < maxErrorGood.length; ++i) {
            Assert.assertEquals((double)expectedGood[i], (double)maxErrorGood[i], (double)(0.01 * expectedGood[i]));
            Assert.assertEquals((double)expectedBad[i], (double)maxErrorBad[i], (double)(0.01 * expectedBad[i]));
        }
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testWrongOrder() {
        UnivariateDifferentiableFunction f = new FiniteDifferencesDifferentiator(3, 0.01).differentiate(new UnivariateFunction(){

            public double value(double x) {
                throw new MathInternalError();
            }
        });
        f.value(new DerivativeStructure(1, 3, 0, 1.0));
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testWrongOrderVector() {
        UnivariateDifferentiableVectorFunction f = new FiniteDifferencesDifferentiator(3, 0.01).differentiate(new UnivariateVectorFunction(){

            public double[] value(double x) {
                throw new MathInternalError();
            }
        });
        f.value(new DerivativeStructure(1, 3, 0, 1.0));
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testWrongOrderMatrix() {
        UnivariateDifferentiableMatrixFunction f = new FiniteDifferencesDifferentiator(3, 0.01).differentiate(new UnivariateMatrixFunction(){

            public double[][] value(double x) {
                throw new MathInternalError();
            }
        });
        f.value(new DerivativeStructure(1, 3, 0, 1.0));
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testTooLargeStep() {
        new FiniteDifferencesDifferentiator(3, 2.5, 0.0, 1.0);
    }

    @Test
    public void testBounds() {
        double slope = 2.5;
        UnivariateFunction f = new UnivariateFunction(){

            public double value(double x) {
                if (x < 0.0) {
                    throw new NumberIsTooSmallException((Number)x, (Number)0, true);
                }
                if (x > 1.0) {
                    throw new NumberIsTooLargeException((Number)x, (Number)1, true);
                }
                return 2.5 * x;
            }
        };
        UnivariateDifferentiableFunction missingBounds = new FiniteDifferencesDifferentiator(3, 0.1).differentiate(f);
        UnivariateDifferentiableFunction properlyBounded = new FiniteDifferencesDifferentiator(3, 0.1, 0.0, 1.0).differentiate(f);
        DerivativeStructure tLow = new DerivativeStructure(1, 1, 0, 0.05);
        DerivativeStructure tHigh = new DerivativeStructure(1, 1, 0, 0.95);
        try {
            missingBounds.value(tLow);
            Assert.fail((String)"an exception should have been thrown");
        }
        catch (NumberIsTooSmallException nse) {
            Assert.assertEquals((double)-0.05, (double)nse.getArgument().doubleValue(), (double)1.0E-10);
        }
        catch (Exception e) {
            Assert.fail((String)("wrong exception caught: " + e.getClass().getName()));
        }
        try {
            missingBounds.value(tHigh);
            Assert.fail((String)"an exception should have been thrown");
        }
        catch (NumberIsTooLargeException nle) {
            Assert.assertEquals((double)1.05, (double)nle.getArgument().doubleValue(), (double)1.0E-10);
        }
        catch (Exception e) {
            Assert.fail((String)("wrong exception caught: " + e.getClass().getName()));
        }
        Assert.assertEquals((double)2.5, (double)properlyBounded.value(tLow).getPartialDerivative(new int[]{1}), (double)1.0E-10);
        Assert.assertEquals((double)2.5, (double)properlyBounded.value(tHigh).getPartialDerivative(new int[]{1}), (double)1.0E-10);
    }

    @Test
    public void testBoundedSqrt() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(9, 0.03125, 0.0, Double.POSITIVE_INFINITY);
        UnivariateDifferentiableFunction sqrt = differentiator.differentiate(new UnivariateFunction(){

            public double value(double x) {
                return FastMath.sqrt((double)x);
            }
        });
        DerivativeStructure t001 = new DerivativeStructure(1, 1, 0, 0.01);
        Assert.assertEquals((double)(0.5 / FastMath.sqrt((double)t001.getValue())), (double)sqrt.value(t001).getPartialDerivative(new int[]{1}), (double)1.6);
        DerivativeStructure t01 = new DerivativeStructure(1, 1, 0, 0.1);
        Assert.assertEquals((double)(0.5 / FastMath.sqrt((double)t01.getValue())), (double)sqrt.value(t01).getPartialDerivative(new int[]{1}), (double)0.007);
        DerivativeStructure t03 = new DerivativeStructure(1, 1, 0, 0.3);
        Assert.assertEquals((double)(0.5 / FastMath.sqrt((double)t03.getValue())), (double)sqrt.value(t03).getPartialDerivative(new int[]{1}), (double)2.1E-7);
    }

    @Test
    public void testVectorFunction() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(7, 0.01);
        UnivariateDifferentiableVectorFunction f = differentiator.differentiate(new UnivariateVectorFunction(){

            public double[] value(double x) {
                return new double[]{FastMath.cos((double)x), FastMath.sin((double)x)};
            }
        });
        for (double x = -10.0; x < 10.0; x += 0.1) {
            DerivativeStructure dsX = new DerivativeStructure(1, 2, 0, x);
            DerivativeStructure[] y = f.value(dsX);
            double cos = FastMath.cos((double)x);
            double sin = FastMath.sin((double)x);
            double[] f1 = f.value(dsX.getValue());
            DerivativeStructure[] f2 = f.value(dsX);
            Assert.assertEquals((long)f1.length, (long)f2.length);
            for (int i = 0; i < f1.length; ++i) {
                Assert.assertEquals((double)f1[i], (double)f2[i].getValue(), (double)1.0E-15);
            }
            Assert.assertEquals((double)cos, (double)y[0].getValue(), (double)7.0E-16);
            Assert.assertEquals((double)sin, (double)y[1].getValue(), (double)7.0E-16);
            Assert.assertEquals((double)(-sin), (double)y[0].getPartialDerivative(new int[]{1}), (double)6.0E-14);
            Assert.assertEquals((double)cos, (double)y[1].getPartialDerivative(new int[]{1}), (double)6.0E-14);
            Assert.assertEquals((double)(-cos), (double)y[0].getPartialDerivative(new int[]{2}), (double)2.0E-11);
            Assert.assertEquals((double)(-sin), (double)y[1].getPartialDerivative(new int[]{2}), (double)2.0E-11);
        }
    }

    @Test
    public void testMatrixFunction() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(7, 0.01);
        UnivariateDifferentiableMatrixFunction f = differentiator.differentiate(new UnivariateMatrixFunction(){

            public double[][] value(double x) {
                return new double[][]{{FastMath.cos((double)x), FastMath.sin((double)x)}, {FastMath.cosh((double)x), FastMath.sinh((double)x)}};
            }
        });
        for (double x = -1.0; x < 1.0; x += 0.02) {
            DerivativeStructure dsX = new DerivativeStructure(1, 2, 0, x);
            DerivativeStructure[][] y = f.value(dsX);
            double cos = FastMath.cos((double)x);
            double sin = FastMath.sin((double)x);
            double cosh = FastMath.cosh((double)x);
            double sinh = FastMath.sinh((double)x);
            double[][] f1 = f.value(dsX.getValue());
            DerivativeStructure[][] f2 = f.value(dsX);
            Assert.assertEquals((long)f1.length, (long)f2.length);
            for (int i = 0; i < f1.length; ++i) {
                Assert.assertEquals((long)f1[i].length, (long)f2[i].length);
                for (int j = 0; j < f1[i].length; ++j) {
                    Assert.assertEquals((double)f1[i][j], (double)f2[i][j].getValue(), (double)1.0E-15);
                }
            }
            Assert.assertEquals((double)cos, (double)y[0][0].getValue(), (double)7.0E-18);
            Assert.assertEquals((double)sin, (double)y[0][1].getValue(), (double)6.0E-17);
            Assert.assertEquals((double)cosh, (double)y[1][0].getValue(), (double)3.0E-16);
            Assert.assertEquals((double)sinh, (double)y[1][1].getValue(), (double)3.0E-16);
            Assert.assertEquals((double)(-sin), (double)y[0][0].getPartialDerivative(new int[]{1}), (double)2.0E-14);
            Assert.assertEquals((double)cos, (double)y[0][1].getPartialDerivative(new int[]{1}), (double)2.0E-14);
            Assert.assertEquals((double)sinh, (double)y[1][0].getPartialDerivative(new int[]{1}), (double)3.0E-14);
            Assert.assertEquals((double)cosh, (double)y[1][1].getPartialDerivative(new int[]{1}), (double)3.0E-14);
            Assert.assertEquals((double)(-cos), (double)y[0][0].getPartialDerivative(new int[]{2}), (double)3.0E-12);
            Assert.assertEquals((double)(-sin), (double)y[0][1].getPartialDerivative(new int[]{2}), (double)3.0E-12);
            Assert.assertEquals((double)cosh, (double)y[1][0].getPartialDerivative(new int[]{2}), (double)6.0E-12);
            Assert.assertEquals((double)sinh, (double)y[1][1].getPartialDerivative(new int[]{2}), (double)6.0E-12);
        }
    }

    @Test
    public void testSeveralFreeParameters() {
        FiniteDifferencesDifferentiator differentiator = new FiniteDifferencesDifferentiator(5, 0.001);
        Sin sine = new Sin();
        UnivariateDifferentiableFunction f = differentiator.differentiate((UnivariateFunction)sine);
        double[] expectedError = new double[]{6.696E-16, 1.371E-12, 2.007E-8, 1.754E-5};
        double[] maxError = new double[expectedError.length];
        for (double x = -2.0; x < 2.0; x += 0.1) {
            for (double y = -2.0; y < 2.0; y += 0.1) {
                DerivativeStructure dsX = new DerivativeStructure(2, maxError.length - 1, 0, x);
                DerivativeStructure dsY = new DerivativeStructure(2, maxError.length - 1, 1, y);
                DerivativeStructure dsT = dsX.multiply(3).subtract(dsY.multiply(2));
                DerivativeStructure sRef = sine.value(dsT);
                DerivativeStructure s = f.value(dsT);
                for (int xOrder = 0; xOrder <= sRef.getOrder(); ++xOrder) {
                    for (int yOrder = 0; yOrder <= sRef.getOrder(); ++yOrder) {
                        if (xOrder + yOrder > sRef.getOrder()) continue;
                        maxError[xOrder + yOrder] = FastMath.max((double)maxError[xOrder + yOrder], (double)FastMath.abs((double)(sRef.getPartialDerivative(new int[]{xOrder, yOrder}) - s.getPartialDerivative(new int[]{xOrder, yOrder}))));
                    }
                }
            }
        }
        for (int i = 0; i < maxError.length; ++i) {
            Assert.assertEquals((double)expectedError[i], (double)maxError[i], (double)(0.01 * expectedError[i]));
        }
    }
}

