/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.sharedcachemanager;

import java.io.IOException;
import java.util.concurrent.locks.Lock;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.server.sharedcache.SharedCacheUtil;
import org.apache.hadoop.yarn.server.sharedcachemanager.metrics.CleanerMetrics;
import org.apache.hadoop.yarn.server.sharedcachemanager.store.SCMStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class CleanerTask
implements Runnable {
    private static final String RENAMED_SUFFIX = "-renamed";
    private static final Logger LOG = LoggerFactory.getLogger(CleanerTask.class);
    private final String location;
    private final long sleepTime;
    private final int nestedLevel;
    private final Path root;
    private final FileSystem fs;
    private final SCMStore store;
    private final CleanerMetrics metrics;
    private final Lock cleanerTaskLock;

    public static CleanerTask create(Configuration conf, SCMStore store, CleanerMetrics metrics, Lock cleanerTaskLock) {
        try {
            String location = conf.get("yarn.sharedcache.root-dir", "/sharedcache");
            long sleepTime = conf.getLong("yarn.sharedcache.cleaner.resource-sleep-ms", 0L);
            int nestedLevel = SharedCacheUtil.getCacheDepth((Configuration)conf);
            FileSystem fs = FileSystem.get((Configuration)conf);
            return new CleanerTask(location, sleepTime, nestedLevel, fs, store, metrics, cleanerTaskLock);
        }
        catch (IOException e) {
            LOG.error("Unable to obtain the filesystem for the cleaner service", (Throwable)e);
            throw new ExceptionInInitializerError(e);
        }
    }

    CleanerTask(String location, long sleepTime, int nestedLevel, FileSystem fs, SCMStore store, CleanerMetrics metrics, Lock cleanerTaskLock) {
        this.location = location;
        this.sleepTime = sleepTime;
        this.nestedLevel = nestedLevel;
        this.root = new Path(location);
        this.fs = fs;
        this.store = store;
        this.metrics = metrics;
        this.cleanerTaskLock = cleanerTaskLock;
    }

    @Override
    public void run() {
        if (!this.cleanerTaskLock.tryLock()) {
            LOG.warn("A cleaner task is already running. This scheduled cleaner task will do nothing.");
            return;
        }
        try {
            if (!this.fs.exists(this.root)) {
                LOG.error("The shared cache root " + this.location + " was not found. The cleaner task will do nothing.");
                return;
            }
            this.process();
        }
        catch (Throwable e) {
            LOG.error("Unexpected exception while initializing the cleaner task. This task will do nothing,", e);
        }
        finally {
            this.cleanerTaskLock.unlock();
        }
    }

    void process() {
        this.metrics.reportCleaningStart();
        try {
            String pattern = SharedCacheUtil.getCacheEntryGlobPattern((int)this.nestedLevel);
            FileStatus[] resources = this.fs.globStatus(new Path(this.root, pattern));
            int numResources = resources == null ? 0 : resources.length;
            LOG.info("Processing " + numResources + " resources in the shared cache");
            long beginMs = System.currentTimeMillis();
            if (resources != null) {
                for (FileStatus resource : resources) {
                    if (Thread.currentThread().isInterrupted()) {
                        LOG.warn("The cleaner task was interrupted. Aborting.");
                        break;
                    }
                    if (resource.isDirectory()) {
                        this.processSingleResource(resource);
                    } else {
                        LOG.warn("Invalid file at path " + resource.getPath().toString() + " when a directory was expected");
                    }
                    if (this.sleepTime <= 0L) continue;
                    Thread.sleep(this.sleepTime);
                }
            }
            long endMs = System.currentTimeMillis();
            long durationMs = endMs - beginMs;
            LOG.info("Processed " + numResources + " resource(s) in " + durationMs + " ms.");
        }
        catch (IOException e1) {
            LOG.error("Unable to complete the cleaner task", (Throwable)e1);
        }
        catch (InterruptedException e2) {
            Thread.currentThread().interrupt();
        }
    }

    Path getRootPath() {
        return this.root;
    }

    void processSingleResource(FileStatus resource) {
        ResourceStatus resourceStatus;
        Path path;
        block19: {
            path = resource.getPath();
            resourceStatus = ResourceStatus.INIT;
            if (path.toString().endsWith(RENAMED_SUFFIX)) {
                LOG.info("Found a renamed directory that was left undeleted at " + path.toString() + ". Deleting.");
                try {
                    if (this.fs.delete(path, true)) {
                        resourceStatus = ResourceStatus.DELETED;
                    }
                }
                catch (IOException e) {
                    LOG.error("Error while processing a shared cache resource: " + path, (Throwable)e);
                }
            } else {
                String key = path.getName();
                try {
                    this.store.cleanResourceReferences(key);
                }
                catch (YarnException e) {
                    LOG.error("Exception thrown while removing dead appIds.", (Throwable)e);
                }
                if (this.store.isResourceEvictable(key, resource)) {
                    try {
                        if (this.store.removeResource(key)) {
                            boolean deleted = this.removeResourceFromCacheFileSystem(path);
                            if (deleted) {
                                resourceStatus = ResourceStatus.DELETED;
                            } else {
                                LOG.error("Failed to remove path from the file system. Skipping this resource: " + path);
                                resourceStatus = ResourceStatus.ERROR;
                            }
                            break block19;
                        }
                        resourceStatus = ResourceStatus.PROCESSED;
                    }
                    catch (IOException e) {
                        LOG.error("Failed to remove path from the file system. Skipping this resource: " + path, (Throwable)e);
                        resourceStatus = ResourceStatus.ERROR;
                    }
                } else {
                    resourceStatus = ResourceStatus.PROCESSED;
                }
            }
        }
        switch (resourceStatus) {
            case DELETED: {
                this.metrics.reportAFileDelete();
                break;
            }
            case PROCESSED: {
                this.metrics.reportAFileProcess();
                break;
            }
            case ERROR: {
                this.metrics.reportAFileError();
                break;
            }
            default: {
                LOG.error("Cleaner encountered an invalid status (" + resourceStatus + ") while processing resource: " + path.getName());
            }
        }
    }

    private boolean removeResourceFromCacheFileSystem(Path path) throws IOException {
        Path renamedPath = new Path(path.toString() + RENAMED_SUFFIX);
        if (this.fs.rename(path, renamedPath)) {
            LOG.info("Deleting " + path.toString());
            return this.fs.delete(renamedPath, true);
        }
        LOG.error("We were not able to rename the directory to " + renamedPath.toString() + ". We will leave it intact.");
        return false;
    }

    private static enum ResourceStatus {
        INIT,
        PROCESSED,
        DELETED,
        ERROR;

    }
}

