/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import com.obs.services.exception.ObsException;
import com.obs.services.model.KeyAndVersion;
import com.obs.services.model.ListObjectsRequest;
import com.obs.services.model.ObjectListing;
import com.obs.services.model.ObsObject;
import com.obs.services.model.fs.GetAttributeRequest;
import com.obs.services.model.fs.NewFolderRequest;
import com.obs.services.model.fs.ObsFSAttribute;
import com.obs.services.model.fs.RenameRequest;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.fs.obs.FileConflictException;
import org.apache.hadoop.fs.obs.OBSCommonUtils;
import org.apache.hadoop.fs.obs.OBSFileStatus;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class OBSPosixBucketUtils {
    private static final Logger LOG = LoggerFactory.getLogger(OBSPosixBucketUtils.class);

    private OBSPosixBucketUtils() {
    }

    static int fsGetObjectKeyDepth(String key) {
        int depth = 0;
        int idx = key.indexOf(47);
        while (idx >= 0) {
            ++depth;
            idx = key.indexOf(47, idx + 1);
        }
        return key.endsWith("/") ? depth - 1 : depth;
    }

    static boolean fsIsFolder(ObsFSAttribute attr) {
        int ifDir = 16384;
        int mode = attr.getMode();
        if (mode < 0) {
            return false;
        }
        return (mode & 0x4000) != 0;
    }

    static boolean renameBasedOnPosix(OBSFileSystem owner, Path src, Path dst) throws IOException {
        String dstKey;
        String srcKey;
        Path dstPath;
        block8: {
            dstPath = dst;
            srcKey = OBSCommonUtils.pathToKey(owner, src);
            dstKey = OBSCommonUtils.pathToKey(owner, dstPath);
            if (srcKey.isEmpty()) {
                LOG.error("rename: src [{}] is root directory", (Object)src);
                return false;
            }
            try {
                FileStatus dstStatus = owner.getFileStatus(dstPath);
                if (dstStatus.isDirectory()) {
                    String newDstString = OBSCommonUtils.maybeAddTrailingSlash(dstPath.toString());
                    String filename = srcKey.substring(OBSCommonUtils.pathToKey(owner, src.getParent()).length() + 1);
                    dstPath = new Path(newDstString + filename);
                    dstKey = OBSCommonUtils.pathToKey(owner, dstPath);
                    LOG.debug("rename: dest is an existing directory and will be changed to [{}]", (Object)dstPath);
                    if (owner.exists(dstPath)) {
                        LOG.error("rename: failed to rename " + src + " to " + dstPath + " because destination exists");
                        return false;
                    }
                    break block8;
                }
                if (srcKey.equals(dstKey)) {
                    LOG.warn("rename: src and dest refer to the same file or directory: {}", (Object)dstPath);
                    return true;
                }
                LOG.error("rename: failed to rename " + src + " to " + dstPath + " because destination exists");
                return false;
            }
            catch (FileNotFoundException e) {
                LOG.debug("rename: dest [{}] does not exist", (Object)dstPath);
            }
            catch (FileConflictException e) {
                FileStatus dstParentStatus;
                Path parent = dstPath.getParent();
                if (OBSCommonUtils.pathToKey(owner, parent).isEmpty() || (dstParentStatus = owner.getFileStatus(parent)).isDirectory()) break block8;
                throw new ParentNotDirectoryException(parent + " is not a directory");
            }
        }
        if (dstKey.startsWith(srcKey) && (dstKey.equals(srcKey) || dstKey.charAt(srcKey.length()) == '/')) {
            LOG.error("rename: dest [{}] cannot be a descendant of src [{}]", (Object)dstPath, (Object)src);
            return false;
        }
        return OBSPosixBucketUtils.innerFsRenameWithRetry(owner, src, dstPath, srcKey, dstKey);
    }

    private static boolean innerFsRenameWithRetry(OBSFileSystem owner, Path src, Path dst, String srcKey, String dstKey) throws IOException {
        boolean renameResult = true;
        for (int retryTime = 1; retryTime <= 3; ++retryTime) {
            try {
                LOG.debug("rename: {}-st rename from [{}] to [{}] ...", new Object[]{retryTime, srcKey, dstKey});
                OBSPosixBucketUtils.innerFsRenameFile(owner, srcKey, dstKey);
                renameResult = true;
                break;
            }
            catch (FileNotFoundException e) {
                if (owner.exists(dst)) {
                    LOG.warn("rename: successfully {}-st rename src [{}] to dest [{}] with SDK retry", new Object[]{retryTime, src, dst, e});
                    renameResult = true;
                    break;
                }
                LOG.error("rename: failed {}-st rename src [{}] to dest [{}]", new Object[]{retryTime, src, dst, e});
                renameResult = false;
                break;
            }
            catch (IOException e) {
                if (retryTime == 3) {
                    LOG.error("rename: failed {}-st rename src [{}] to dest [{}]", new Object[]{retryTime, src, dst, e});
                    throw e;
                }
                LOG.warn("rename: failed {}-st rename src [{}] to dest [{}]", new Object[]{retryTime, src, dst, e});
                if (owner.exists(dst) && owner.exists(src)) {
                    LOG.warn("rename: failed {}-st rename src [{}] to dest [{}] with SDK retry", new Object[]{retryTime, src, dst, e});
                    renameResult = false;
                    break;
                }
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        return renameResult;
    }

    static void fsRenameToNewFolder(OBSFileSystem owner, String src, String dst) throws IOException, ObsException {
        LOG.debug("RenameFolder path {} to {}", (Object)src, (Object)dst);
        try {
            RenameRequest renameObjectRequest = new RenameRequest();
            renameObjectRequest.setBucketName(owner.getBucket());
            renameObjectRequest.setObjectKey(src);
            renameObjectRequest.setNewObjectKey(dst);
            owner.getObsClient().renameFolder(renameObjectRequest);
            owner.getSchemeStatistics().incrementWriteOps(1);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("renameFile(" + src + ", " + dst + ")", src, e);
        }
    }

    static void innerFsRenameFile(OBSFileSystem owner, String srcKey, String dstKey) throws IOException {
        LOG.debug("RenameFile path {} to {}", (Object)srcKey, (Object)dstKey);
        try {
            RenameRequest renameObjectRequest = new RenameRequest();
            renameObjectRequest.setBucketName(owner.getBucket());
            renameObjectRequest.setObjectKey(srcKey);
            renameObjectRequest.setNewObjectKey(dstKey);
            owner.getObsClient().renameFile(renameObjectRequest);
            owner.getSchemeStatistics().incrementWriteOps(1);
        }
        catch (ObsException e) {
            if (e.getResponseCode() == 404) {
                throw new FileNotFoundException("No such file or directory: " + srcKey);
            }
            if (e.getResponseCode() == 409) {
                throw new FileConflictException("File conflicts during rename, " + e.getResponseStatus());
            }
            throw OBSCommonUtils.translateException("renameFile(" + srcKey + ", " + dstKey + ")", srcKey, e);
        }
    }

    static void fsRenameToNewObject(OBSFileSystem owner, String srcKey, String dstKey) throws IOException {
        String newSrcKey = srcKey;
        String newdstKey = dstKey;
        newSrcKey = OBSCommonUtils.maybeDeleteBeginningSlash(newSrcKey);
        newdstKey = OBSCommonUtils.maybeDeleteBeginningSlash(newdstKey);
        if (newSrcKey.endsWith("/")) {
            OBSPosixBucketUtils.fsRenameToNewFolder(owner, newSrcKey, newdstKey);
        } else {
            OBSPosixBucketUtils.innerFsRenameFile(owner, newSrcKey, newdstKey);
        }
    }

    private static int fsRemoveFile(OBSFileSystem owner, String sonObjectKey, List<KeyAndVersion> files) throws IOException {
        files.add(new KeyAndVersion(sonObjectKey));
        if (files.size() == owner.getMaxEntriesToDelete()) {
            OBSCommonUtils.removeKeys(owner, files, true, false);
            return owner.getMaxEntriesToDelete();
        }
        return 0;
    }

    static boolean fsDelete(OBSFileSystem owner, FileStatus status, boolean recursive) throws IOException, ObsException {
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        Path f = status.getPath();
        String key = OBSCommonUtils.pathToKey(owner, f);
        if (!status.isDirectory()) {
            LOG.debug("delete: Path is a file");
            OBSPosixBucketUtils.trashObjectIfNeed(owner, key);
        } else {
            LOG.debug("delete: Path is a directory: {} - recursive {}", (Object)f, (Object)recursive);
            key = OBSCommonUtils.maybeAddTrailingSlash(key);
            boolean isEmptyDir = OBSCommonUtils.isFolderEmpty(owner, key);
            if (key.equals("")) {
                return OBSCommonUtils.rejectRootDirectoryDelete(owner.getBucket(), isEmptyDir, recursive);
            }
            if (!recursive && !isEmptyDir) {
                LOG.warn("delete: Path is not empty: {} - recursive {}", (Object)f, (Object)recursive);
                throw new PathIsNotEmptyDirectoryException(f.toString());
            }
            if (isEmptyDir) {
                LOG.debug("delete: Deleting fake empty directory {} - recursive {}", (Object)f, (Object)recursive);
                OBSCommonUtils.deleteObject(owner, key);
            } else {
                LOG.debug("delete: Deleting objects for directory prefix {} to delete - recursive {}", (Object)f, (Object)recursive);
                OBSPosixBucketUtils.trashFolderIfNeed(owner, key, f);
            }
        }
        long endTime = System.currentTimeMillis();
        LOG.debug("delete Path:{} thread:{}, timeUsedInMilliSec:{}", new Object[]{f, threadId, endTime - startTime});
        return true;
    }

    private static void trashObjectIfNeed(OBSFileSystem owner, String key) throws ObsException, IOException {
        if (OBSPosixBucketUtils.needToTrash(owner, key)) {
            OBSPosixBucketUtils.mkTrash(owner, key);
            StringBuilder sb = new StringBuilder(owner.getTrashDir());
            sb.append(key);
            if (owner.exists(new Path(sb.toString()))) {
                SimpleDateFormat df = new SimpleDateFormat("-yyyyMMddHHmmss");
                sb.append(df.format(new Date()));
            }
            OBSPosixBucketUtils.fsRenameToNewObject(owner, key, sb.toString());
            LOG.debug("Moved: '" + key + "' to trash at: " + sb.toString());
        } else {
            OBSCommonUtils.deleteObject(owner, key);
        }
    }

    private static void trashFolderIfNeed(OBSFileSystem owner, String key, Path f) throws ObsException, IOException {
        if (OBSPosixBucketUtils.needToTrash(owner, key)) {
            OBSPosixBucketUtils.mkTrash(owner, key);
            StringBuilder sb = new StringBuilder(owner.getTrashDir());
            String subKey = OBSCommonUtils.maybeAddTrailingSlash(key);
            sb.append(subKey);
            if (owner.exists(new Path(sb.toString()))) {
                SimpleDateFormat df = new SimpleDateFormat("-yyyyMMddHHmmss");
                sb.insert(sb.length() - 1, df.format(new Date()));
            }
            String srcKey = OBSCommonUtils.maybeDeleteBeginningSlash(key);
            String dstKey = OBSCommonUtils.maybeDeleteBeginningSlash(sb.toString());
            OBSPosixBucketUtils.fsRenameToNewFolder(owner, srcKey, dstKey);
            LOG.debug("Moved: '" + key + "' to trash at: " + sb.toString());
        } else if (owner.isEnableMultiObjectDeleteRecursion()) {
            long delNum = OBSPosixBucketUtils.fsRecursivelyDeleteDir(owner, key, true);
            LOG.debug("Recursively delete {} files/dirs when deleting {}", (Object)delNum, (Object)key);
        } else {
            OBSPosixBucketUtils.fsNonRecursivelyDelete(owner, f);
        }
    }

    static long fsRecursivelyDeleteDir(OBSFileSystem owner, String parentKey, boolean deleteParent) throws IOException {
        long delNum = 0L;
        ArrayList<KeyAndVersion> subdirList = new ArrayList<KeyAndVersion>(owner.getMaxEntriesToDelete());
        ArrayList<KeyAndVersion> fileList = new ArrayList<KeyAndVersion>(owner.getMaxEntriesToDelete());
        ListObjectsRequest request = OBSCommonUtils.createListObjectsRequest(owner, parentKey, "/", owner.getMaxKeys());
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        while (true) {
            for (String commonPrefix : objects.getCommonPrefixes()) {
                if (commonPrefix.equals(parentKey)) continue;
                delNum += (long)OBSPosixBucketUtils.fsRemoveSubdir(owner, commonPrefix, subdirList);
            }
            for (ObsObject sonObject : objects.getObjects()) {
                String sonObjectKey = sonObject.getObjectKey();
                if (sonObjectKey.equals(parentKey)) continue;
                if (!sonObjectKey.endsWith("/")) {
                    delNum += (long)OBSPosixBucketUtils.fsRemoveFile(owner, sonObjectKey, fileList);
                    continue;
                }
                delNum += (long)OBSPosixBucketUtils.fsRemoveSubdir(owner, sonObjectKey, subdirList);
            }
            if (!objects.isTruncated()) break;
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        delNum += (long)fileList.size();
        OBSCommonUtils.removeKeys(owner, fileList, true, false);
        delNum += (long)subdirList.size();
        OBSCommonUtils.removeKeys(owner, subdirList, true, false);
        if (deleteParent) {
            OBSCommonUtils.deleteObject(owner, parentKey);
            ++delNum;
        }
        return delNum;
    }

    private static boolean needToTrash(OBSFileSystem owner, String key) {
        String newKey = key;
        newKey = OBSCommonUtils.maybeDeleteBeginningSlash(newKey);
        if (owner.isEnableTrash() && newKey.startsWith(owner.getTrashDir())) {
            return false;
        }
        return owner.isEnableTrash();
    }

    private static int fsRemoveSubdir(OBSFileSystem owner, String subdirKey, List<KeyAndVersion> subdirList) throws IOException {
        OBSPosixBucketUtils.fsRecursivelyDeleteDir(owner, subdirKey, false);
        subdirList.add(new KeyAndVersion(subdirKey));
        if (subdirList.size() == owner.getMaxEntriesToDelete()) {
            OBSCommonUtils.removeKeys(owner, subdirList, true, false);
            return owner.getMaxEntriesToDelete();
        }
        return 0;
    }

    private static void mkTrash(OBSFileSystem owner, String key) throws ObsException, IOException {
        String newKey = key;
        StringBuilder sb = new StringBuilder(owner.getTrashDir());
        newKey = OBSCommonUtils.maybeAddTrailingSlash(newKey);
        sb.append(newKey);
        sb.deleteCharAt(sb.length() - 1);
        sb.delete(sb.lastIndexOf("/"), sb.length());
        Path fastDeleteRecycleDirPath = new Path(sb.toString());
        if (!owner.exists(fastDeleteRecycleDirPath)) {
            owner.mkdirs(fastDeleteRecycleDirPath);
        }
    }

    private static void fsNonRecursivelyDelete(OBSFileSystem owner, Path parent) throws IOException, ObsException {
        FileStatus[] arFileStatus = OBSCommonUtils.innerListStatus(owner, parent, true);
        OBSPosixBucketUtils.fsRemoveKeys(owner, arFileStatus);
        OBSCommonUtils.deleteObject(owner, OBSCommonUtils.pathToKey(owner, parent));
    }

    private static void fsRemoveKeys(OBSFileSystem owner, FileStatus[] arFileStatus) throws ObsException, IOException {
        if (arFileStatus.length <= 0) {
            return;
        }
        for (FileStatus fileStatus : arFileStatus) {
            String key = OBSCommonUtils.pathToKey(owner, fileStatus.getPath());
            OBSCommonUtils.blockRootDelete(owner.getBucket(), key);
        }
        OBSPosixBucketUtils.fsRemoveKeysByDepth(owner, arFileStatus);
    }

    private static void fsRemoveKeysByDepth(OBSFileSystem owner, FileStatus[] arFileStatus) throws ObsException, IOException {
        if (arFileStatus.length <= 0) {
            return;
        }
        int depth = Integer.MAX_VALUE;
        ArrayList<KeyAndVersion> leafKeys = new ArrayList<KeyAndVersion>(owner.getMaxEntriesToDelete());
        for (int idx = arFileStatus.length - 1; idx >= 0; --idx) {
            if (leafKeys.size() >= owner.getMaxEntriesToDelete()) {
                OBSCommonUtils.removeKeys(owner, leafKeys, true, false);
            }
            String key = OBSCommonUtils.pathToKey(owner, arFileStatus[idx].getPath());
            if (!arFileStatus[idx].isDirectory()) {
                leafKeys.add(new KeyAndVersion(key, null));
                continue;
            }
            int keyDepth = OBSPosixBucketUtils.fsGetObjectKeyDepth(key);
            if (keyDepth == depth) {
                leafKeys.add(new KeyAndVersion(key, null));
                continue;
            }
            if (keyDepth < depth) {
                OBSCommonUtils.removeKeys(owner, leafKeys, true, false);
                depth = keyDepth;
                leafKeys.add(new KeyAndVersion(key, null));
                continue;
            }
            LOG.warn("The objects list is invalid because it isn't sorted by path depth.");
            throw new ObsException("System failure");
        }
        OBSCommonUtils.removeKeys(owner, leafKeys, true, false);
    }

    static void fsCreateFolder(OBSFileSystem owner, String objectName) throws ObsException {
        for (int retryTime = 1; retryTime < 3; ++retryTime) {
            try {
                OBSPosixBucketUtils.innerFsCreateFolder(owner, objectName);
                return;
            }
            catch (ObsException e) {
                LOG.warn("Failed to create folder [{}], retry time [{}], exception [{}]", new Object[]{objectName, retryTime, e});
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        OBSPosixBucketUtils.innerFsCreateFolder(owner, objectName);
    }

    private static void innerFsCreateFolder(OBSFileSystem owner, String objectName) throws ObsException {
        NewFolderRequest newFolderRequest = new NewFolderRequest(owner.getBucket(), objectName);
        newFolderRequest.setAcl(owner.getCannedACL());
        long len = newFolderRequest.getObjectKey().length();
        owner.getObsClient().newFolder(newFolderRequest);
        owner.getSchemeStatistics().incrementWriteOps(1);
        owner.getSchemeStatistics().incrementBytesWritten(len);
    }

    static OBSFileStatus innerFsGetObjectStatus(OBSFileSystem owner, Path f) throws IOException {
        Path path = OBSCommonUtils.qualify(owner, f);
        String key = OBSCommonUtils.pathToKey(owner, path);
        LOG.debug("Getting path status for {}  ({})", (Object)path, (Object)key);
        if (key.isEmpty()) {
            LOG.debug("Found root directory");
            return new OBSFileStatus(path, owner.getUsername());
        }
        try {
            GetAttributeRequest getAttrRequest = new GetAttributeRequest(owner.getBucket(), key);
            ObsFSAttribute meta = owner.getObsClient().getAttribute(getAttrRequest);
            owner.getSchemeStatistics().incrementReadOps(1);
            if (OBSPosixBucketUtils.fsIsFolder(meta)) {
                LOG.debug("Found file (with /): fake directory");
                return new OBSFileStatus(path, OBSCommonUtils.dateToLong(meta.getLastModified()), owner.getUsername());
            }
            LOG.debug("Found file (with /): real file? should not happen: {}", (Object)key);
            return new OBSFileStatus(meta.getContentLength(), OBSCommonUtils.dateToLong(meta.getLastModified()), path, owner.getDefaultBlockSize(path), owner.getUsername());
        }
        catch (ObsException e) {
            if (e.getResponseCode() == 404) {
                LOG.debug("Not Found: {}", (Object)path);
                throw new FileNotFoundException("No such file or directory: " + path);
            }
            if (e.getResponseCode() == 409) {
                throw new FileConflictException("file conflicts: " + e.getResponseStatus());
            }
            throw OBSCommonUtils.translateException("getFileStatus", path, e);
        }
    }

    static ContentSummary fsGetDirectoryContentSummary(OBSFileSystem owner, String key) throws IOException {
        String newKey = key;
        newKey = OBSCommonUtils.maybeAddTrailingSlash(newKey);
        long[] summary = new long[]{0L, 0L, 1L};
        LOG.debug("Summary key {}", (Object)newKey);
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(owner.getBucket());
        request.setPrefix(newKey);
        request.setMaxKeys(owner.getMaxKeys());
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        while (true) {
            if (!objects.getCommonPrefixes().isEmpty() || !objects.getObjects().isEmpty()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Found path as directory (with /): {}/{}", (Object)objects.getCommonPrefixes().size(), (Object)objects.getObjects().size());
                }
                for (String prefix : objects.getCommonPrefixes()) {
                    if (prefix.equals(newKey)) continue;
                    summary[2] = summary[2] + 1L;
                }
                for (ObsObject obj : objects.getObjects()) {
                    if (!obj.getObjectKey().endsWith("/")) {
                        summary[0] = summary[0] + obj.getMetadata().getContentLength();
                        summary[1] = summary[1] + 1L;
                        continue;
                    }
                    if (obj.getObjectKey().equals(newKey)) continue;
                    summary[2] = summary[2] + 1L;
                }
            }
            if (!objects.isTruncated()) break;
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        LOG.debug(String.format("file size [%d] - file count [%d] - directory count [%d] - file path [%s]", summary[0], summary[1], summary[2], newKey));
        return new ContentSummary.Builder().length(summary[0]).fileCount(summary[1]).directoryCount(summary[2]).spaceConsumed(summary[0]).build();
    }
}

