/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.jar.Attributes;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.GZIPInputStream;
import org.apache.commons.collections.map.CaseInsensitiveMap;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSDataOutputStreamBuilder;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FutureDataInputStreamBuilder;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathExistsException;
import org.apache.hadoop.fs.PathIsDirectoryException;
import org.apache.hadoop.fs.PathNotFoundException;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.shell.PathData;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.maprfs.AbstractMapRFileSystem;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.functional.FutureIO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Public
@InterfaceStability.Evolving
public class FileUtil {
    private static final Logger LOG = LoggerFactory.getLogger(FileUtil.class);
    public static final int SYMLINK_NO_PRIVILEGE = 2;
    private static final int BUFFER_SIZE = 8192;

    public static Path[] stat2Paths(FileStatus[] stats) {
        if (stats == null) {
            return null;
        }
        Path[] ret = new Path[stats.length];
        for (int i = 0; i < stats.length; ++i) {
            ret[i] = stats[i].getPath();
        }
        return ret;
    }

    public static Path[] stat2Paths(FileStatus[] stats, Path path) {
        if (stats == null) {
            return new Path[]{path};
        }
        return FileUtil.stat2Paths(stats);
    }

    public static void fullyDeleteOnExit(File file) {
        File[] files;
        file.deleteOnExit();
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            for (File child : files) {
                FileUtil.fullyDeleteOnExit(child);
            }
        }
    }

    public static boolean fullyDelete(File dir) {
        return FileUtil.fullyDelete(dir, false);
    }

    public static boolean fullyDelete(File dir, boolean tryGrantPermissions) {
        if (tryGrantPermissions) {
            File parent = dir.getParentFile();
            FileUtil.grantPermissions(parent);
        }
        if (FileUtil.deleteImpl(dir, false)) {
            return true;
        }
        if (!FileUtils.isSymlink((File)dir) && !FileUtil.fullyDeleteContents(dir, tryGrantPermissions)) {
            return false;
        }
        return FileUtil.deleteImpl(dir, true);
    }

    public static String readLink(File f) {
        if (f == null) {
            LOG.warn("Can not read a null symLink");
            return "";
        }
        try {
            return Shell.execCommand(Shell.getReadlinkCommand(f.toString())).trim();
        }
        catch (IOException x) {
            return "";
        }
    }

    public static Path fixSymlinkPath(PathData path) throws IOException {
        return path.stat.getSymlink().toString().startsWith("/") ? path.stat.getSymlink() : new Path(path.stat.getPath().getParent(), path.stat.getSymlink());
    }

    public static Path fixSymlinkFileStatus(FileStatus stat) throws IOException {
        return stat.getSymlink().toString().startsWith("/") ? stat.getSymlink() : new Path(stat.getPath().getParent(), stat.getSymlink());
    }

    public static PathData checkItemForSymlink(PathData item) throws IOException {
        return item.stat != null && item.stat.isSymlink() ? new PathData(FileUtil.fixSymlinkPath(item).toString(), item.fs.getConf()) : item;
    }

    public static PathData checkExistItem(PathData item) throws IOException {
        PathData checkedItem = FileUtil.checkItemForSymlink(item);
        if (!checkedItem.exists) {
            throw new PathNotFoundException(item.toString());
        }
        return checkedItem;
    }

    public static PathData checkPathForSymlink(Path path, Configuration conf) throws IOException {
        return FileUtil.checkItemForSymlink(new PathData(path.toString(), conf));
    }

    private static void grantPermissions(File f) {
        FileUtil.setExecutable(f, true);
        FileUtil.setReadable(f, true);
        FileUtil.setWritable(f, true);
    }

    private static boolean deleteImpl(File f, boolean doLog) {
        if (f == null) {
            LOG.warn("null file argument.");
            return false;
        }
        boolean wasDeleted = f.delete();
        if (wasDeleted) {
            return true;
        }
        boolean ex = f.exists();
        if (doLog && ex) {
            LOG.warn("Failed to delete file or dir [" + f.getAbsolutePath() + "]: it still exists.");
        }
        return !ex;
    }

    public static boolean fullyDeleteContents(File dir) {
        return FileUtil.fullyDeleteContents(dir, false);
    }

    public static boolean fullyDeleteContents(File dir, boolean tryGrantPermissions) {
        if (tryGrantPermissions) {
            FileUtil.grantPermissions(dir);
        }
        boolean deletionSucceeded = true;
        File[] contents = dir.listFiles();
        if (contents != null) {
            for (int i = 0; i < contents.length; ++i) {
                if (contents[i].isFile()) {
                    if (FileUtil.deleteImpl(contents[i], true)) continue;
                    deletionSucceeded = false;
                    continue;
                }
                boolean b = false;
                b = FileUtil.deleteImpl(contents[i], false);
                if (b || FileUtil.fullyDelete(contents[i], tryGrantPermissions)) continue;
                deletionSucceeded = false;
            }
        }
        return deletionSucceeded;
    }

    @Deprecated
    public static void fullyDelete(FileSystem fs, Path dir) throws IOException {
        fs.delete(dir, true);
    }

    private static void checkDependencies(FileSystem srcFS, Path src, FileSystem dstFS, Path dst) throws IOException {
        if (srcFS == dstFS) {
            String srcq = srcFS.makeQualified(src).toString() + "/";
            String dstq = dstFS.makeQualified(dst).toString() + "/";
            if (dstq.startsWith(srcq)) {
                if (srcq.length() == dstq.length()) {
                    throw new IOException("Cannot copy " + src + " to itself.");
                }
                throw new IOException("Cannot copy " + src + " to its subdirectory " + dst);
            }
        }
    }

    public static boolean copy(FileSystem srcFS, Path src, FileSystem dstFS, Path dst, boolean deleteSource, Configuration conf) throws IOException {
        return FileUtil.copy(srcFS, src, dstFS, dst, deleteSource, true, conf);
    }

    public static boolean copy(FileSystem srcFS, Path[] srcs, FileSystem dstFS, Path dst, boolean deleteSource, boolean overwrite, Configuration conf) throws IOException {
        boolean gotException = false;
        boolean returnVal = true;
        StringBuilder exceptions = new StringBuilder();
        if (srcs.length == 1) {
            return FileUtil.copy(srcFS, srcs[0], dstFS, dst, deleteSource, overwrite, conf);
        }
        try {
            FileStatus sdst = dstFS.getFileStatus(dst);
            if (!sdst.isDirectory()) {
                throw new IOException("copying multiple files, but last argument `" + dst + "' is not a directory");
            }
        }
        catch (FileNotFoundException e) {
            throw new IOException("`" + dst + "': specified destination directory does not exist", e);
        }
        for (Path src : srcs) {
            try {
                if (FileUtil.copy(srcFS, src, dstFS, dst, deleteSource, overwrite, conf)) continue;
                returnVal = false;
            }
            catch (IOException e) {
                gotException = true;
                exceptions.append(e.getMessage()).append("\n");
            }
        }
        if (gotException) {
            throw new IOException(exceptions.toString());
        }
        return returnVal;
    }

    public static boolean copy(FileSystem srcFS, Path src, FileSystem dstFS, Path dst, boolean deleteSource, boolean overwrite, Configuration conf) throws IOException {
        FileStatus fileStatus = srcFS.getFileStatus(src);
        return FileUtil.copy(srcFS, fileStatus, dstFS, dst, deleteSource, overwrite, conf);
    }

    public static boolean copy(FileSystem srcFS, FileStatus srcStatus, FileSystem dstFS, Path dst, boolean deleteSource, boolean overwrite, Configuration conf) throws IOException {
        if (srcStatus.isSymlink()) {
            srcStatus = srcFS.getFileStatus(srcStatus.getSymlink());
        }
        Path src = srcStatus.getPath();
        dst = FileUtil.checkDest(src.getName(), dstFS, dst, overwrite);
        if (srcStatus.isDirectory()) {
            FileUtil.checkDependencies(srcFS, src, dstFS, dst);
            if (!dstFS.mkdirs(dst)) {
                return false;
            }
            RemoteIterator<FileStatus> contents = srcFS.listStatusIterator(src);
            while (contents.hasNext()) {
                FileStatus next = contents.next();
                FileUtil.copy(srcFS, next, dstFS, new Path(dst, next.getPath().getName()), deleteSource, overwrite, conf);
            }
        } else {
            if (srcStatus.isTable()) {
                throw new IOException("Cannot copy MDP Tables");
            }
            InputStream in = null;
            FSDataOutputStream out = null;
            try {
                in = (InputStream)FutureIO.awaitFuture(((FutureDataInputStreamBuilder)((FutureDataInputStreamBuilder)srcFS.openFile(src).opt("fs.option.openfile.read.policy", "whole-file")).optLong("fs.option.openfile.length", srcStatus.getLen())).build());
                out = dstFS.create(dst, overwrite);
                org.apache.hadoop.io.IOUtils.copyBytes(in, (OutputStream)out, conf, true);
            }
            catch (IOException e) {
                org.apache.hadoop.io.IOUtils.cleanupWithLogger(LOG, in, out);
                throw e;
            }
        }
        if (srcFS instanceof AbstractMapRFileSystem && dstFS instanceof AbstractMapRFileSystem && dst != null) {
            AbstractMapRFileSystem mapRFileSystem = (AbstractMapRFileSystem)srcFS;
            mapRFileSystem.copyAce(src, dst);
        }
        if (deleteSource) {
            return srcFS.delete(src, true);
        }
        return true;
    }

    public static boolean copyAsMove(FileSystem srcFS, FileStatus srcStatus, FileSystem dstFS, Path dst, boolean deleteSource, boolean overwrite, Configuration conf) throws IOException {
        Path src = srcStatus.getPath();
        dst = FileUtil.checkDest(src.getName(), dstFS, dst, overwrite);
        if (srcStatus.isSymlink()) {
            if (srcFS instanceof AbstractMapRFileSystem && dstFS instanceof AbstractMapRFileSystem) {
                AbstractMapRFileSystem mapRFileSystem = (AbstractMapRFileSystem)srcFS;
                if (dstFS.exists(dst) && overwrite) {
                    dstFS.delete(dst, true);
                }
                mapRFileSystem.createSymlink(srcStatus.getSymlink(), dst, false);
                FileUtil.copyOwner(srcStatus, dst, mapRFileSystem);
                FileUtil.copyPermissions(srcStatus, dst, mapRFileSystem);
                mapRFileSystem.copyAce(srcStatus.getPath(), dst);
                if (deleteSource) {
                    return srcFS.delete(src, true);
                }
                return true;
            }
            srcStatus = srcFS.getFileStatus(srcStatus.getSymlink());
        }
        src = srcStatus.getPath();
        if (srcStatus.isDirectory()) {
            FileUtil.checkDependencies(srcFS, src, dstFS, dst);
            if (!dstFS.mkdirs(dst)) {
                return false;
            }
            FileStatus[] contents = srcFS.listStatus(src);
            for (int i = 0; i < contents.length; ++i) {
                FileUtil.copyAsMove(srcFS, contents[i], dstFS, new Path(dst, contents[i].getPath().getName()), deleteSource, overwrite, conf);
            }
        } else {
            if (srcStatus.isTable()) {
                throw new IOException("Cannot copy MDP Tables");
            }
            FSDataInputStream in = null;
            FSDataOutputStream out = null;
            try {
                in = srcFS.open(src);
                out = dstFS.create(dst, overwrite);
                org.apache.hadoop.io.IOUtils.copyBytes((InputStream)in, (OutputStream)out, conf, true);
            }
            catch (IOException e) {
                org.apache.hadoop.io.IOUtils.closeStream(out);
                org.apache.hadoop.io.IOUtils.closeStream(in);
                throw e;
            }
        }
        FileUtil.copyOwner(srcStatus, dst, dstFS);
        FileUtil.copyPermissions(srcStatus, dst, dstFS);
        if (srcFS instanceof AbstractMapRFileSystem && dstFS instanceof AbstractMapRFileSystem) {
            AbstractMapRFileSystem mapRFileSystem = (AbstractMapRFileSystem)srcFS;
            mapRFileSystem.copyAce(src, dst);
        }
        if (deleteSource) {
            return srcFS.delete(src, true);
        }
        return true;
    }

    private static void copyOwner(FileStatus srcStatus, Path dst, FileSystem fs) throws IOException {
        FileStatus dstFileStatus = fs.getFileStatus(dst);
        String srcFileOwner = srcStatus.getOwner();
        String srcFileGroup = srcStatus.getGroup();
        if (!dstFileStatus.getOwner().equals(srcFileOwner) || !dstFileStatus.getGroup().equals(srcFileGroup)) {
            fs.setOwner(dst, srcFileOwner, srcFileGroup);
        }
    }

    private static void copyPermissions(FileStatus srcStatus, Path dst, FileSystem fs) throws IOException {
        FileStatus dstFileStatus = fs.getFileStatus(dst);
        FsPermission srcFilePermission = srcStatus.getPermission();
        if (!dstFileStatus.getPermission().equals(srcFilePermission)) {
            fs.setPermission(dst, srcFilePermission);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean copyMerge(FileSystem srcFS, Path srcDir, FileSystem dstFS, Path dstFile, boolean deleteSource, Configuration conf, String addString) throws IOException {
        dstFile = FileUtil.checkDest(srcDir.getName(), dstFS, dstFile, false);
        if (!srcFS.getFileStatus(srcDir).isDirectory()) {
            return false;
        }
        try (FSDataOutputStream out = dstFS.create(dstFile);){
            Object[] contents = srcFS.listStatus(srcDir);
            Arrays.sort(contents);
            for (int i = 0; i < contents.length; ++i) {
                if (!((FileStatus)contents[i]).isFile()) continue;
                try (FSDataInputStream in = srcFS.open(((FileStatus)contents[i]).getPath());){
                    org.apache.hadoop.io.IOUtils.copyBytes((InputStream)in, (OutputStream)out, conf, false);
                    if (addString == null) continue;
                    ((OutputStream)out).write(addString.getBytes("UTF-8"));
                    continue;
                }
            }
        }
        if (deleteSource) {
            return srcFS.delete(srcDir, true);
        }
        return true;
    }

    public static boolean copy(File src, FileSystem dstFS, Path dst, boolean deleteSource, Configuration conf) throws IOException {
        dst = FileUtil.checkDest(src.getName(), dstFS, dst, false);
        if (src.isDirectory()) {
            if (!dstFS.mkdirs(dst)) {
                return false;
            }
            File[] contents = FileUtil.listFiles(src);
            for (int i = 0; i < contents.length; ++i) {
                FileUtil.copy(contents[i], dstFS, new Path(dst, contents[i].getName()), deleteSource, conf);
            }
        } else if (src.isFile()) {
            InputStream in = null;
            FSDataOutputStream out = null;
            try {
                in = Files.newInputStream(src.toPath(), new OpenOption[0]);
                out = dstFS.create(dst);
                org.apache.hadoop.io.IOUtils.copyBytes(in, (OutputStream)out, conf);
            }
            catch (IOException e) {
                org.apache.hadoop.io.IOUtils.closeStream(out);
                org.apache.hadoop.io.IOUtils.closeStream(in);
                throw e;
            }
        } else {
            if (!src.canRead()) {
                throw new IOException(src.toString() + ": Permission denied");
            }
            throw new IOException(src.toString() + ": No such file or directory");
        }
        if (deleteSource) {
            return FileUtil.fullyDelete(src);
        }
        return true;
    }

    public static boolean copy(FileSystem srcFS, Path src, File dst, boolean deleteSource, Configuration conf) throws IOException {
        FileStatus filestatus = srcFS.getFileStatus(src);
        return FileUtil.copy(srcFS, filestatus, dst, deleteSource, conf);
    }

    private static boolean copy(FileSystem srcFS, FileStatus srcStatus, File dst, boolean deleteSource, Configuration conf) throws IOException {
        Path src = srcStatus.getPath();
        if (srcStatus.isDirectory()) {
            if (!dst.mkdirs()) {
                return false;
            }
            FileStatus[] contents = srcFS.listStatus(src);
            for (int i = 0; i < contents.length; ++i) {
                FileUtil.copy(srcFS, contents[i], new File(dst, contents[i].getPath().getName()), deleteSource, conf);
            }
        } else {
            InputStream in = (InputStream)FutureIO.awaitFuture(((FutureDataInputStreamBuilder)srcFS.openFile(src).withFileStatus(srcStatus).opt("fs.option.openfile.read.policy", "whole-file")).build());
            org.apache.hadoop.io.IOUtils.copyBytes(in, Files.newOutputStream(dst.toPath(), new OpenOption[0]), conf);
        }
        if (deleteSource) {
            return srcFS.delete(src, true);
        }
        return true;
    }

    private static Path checkDest(String srcName, FileSystem dstFS, Path dst, boolean overwrite) throws IOException {
        FileStatus sdst;
        try {
            sdst = dstFS.getFileStatus(dst);
        }
        catch (FileNotFoundException e) {
            sdst = null;
        }
        if (null != sdst) {
            if (sdst.isDirectory()) {
                if (null == srcName) {
                    if (overwrite) {
                        return dst;
                    }
                    throw new PathIsDirectoryException(dst.toString());
                }
                return FileUtil.checkDest(null, dstFS, new Path(dst, srcName), overwrite);
            }
            if (!overwrite) {
                throw new PathExistsException(dst.toString(), "Target " + dst + " already exists");
            }
        }
        return dst;
    }

    public static boolean isRegularFile(File file) {
        return FileUtil.isRegularFile(file, true);
    }

    public static boolean isRegularFile(File file, boolean allowLinks) {
        if (file != null) {
            if (allowLinks) {
                return Files.isRegularFile(file.toPath(), new LinkOption[0]);
            }
            return Files.isRegularFile(file.toPath(), LinkOption.NOFOLLOW_LINKS);
        }
        return true;
    }

    public static String makeShellPath(String filename) throws IOException {
        return filename;
    }

    public static String makeShellPath(File file) throws IOException {
        return FileUtil.makeShellPath(file, false);
    }

    public static String makeSecureShellPath(File file) throws IOException {
        if (Shell.WINDOWS) {
            throw new UnsupportedOperationException("Not implemented for Windows");
        }
        return FileUtil.makeShellPath(file, false).replace("'", "'\\''");
    }

    public static String makeShellPath(File file, boolean makeCanonicalPath) throws IOException {
        if (makeCanonicalPath) {
            return FileUtil.makeShellPath(file.getCanonicalPath());
        }
        return FileUtil.makeShellPath(file.toString());
    }

    public static long getDU(File dir) {
        long size = 0L;
        if (!dir.exists()) {
            return 0L;
        }
        if (!dir.isDirectory()) {
            return dir.length();
        }
        File[] allFiles = dir.listFiles();
        if (allFiles != null) {
            for (File f : allFiles) {
                if (FileUtils.isSymlink((File)f)) continue;
                size += FileUtil.getDU(f);
            }
        }
        return size;
    }

    public static void unZip(InputStream inputStream, File toDir) throws IOException {
        try (ZipArchiveInputStream zip = new ZipArchiveInputStream(inputStream);){
            int numOfFailedLastModifiedSet = 0;
            String targetDirPath = toDir.getCanonicalPath() + File.separator;
            ZipArchiveEntry entry = zip.getNextZipEntry();
            while (entry != null) {
                if (!entry.isDirectory()) {
                    File file = new File(toDir, entry.getName());
                    if (!file.getCanonicalPath().startsWith(targetDirPath)) {
                        throw new IOException("expanding " + entry.getName() + " would create file outside of " + toDir);
                    }
                    File parent = file.getParentFile();
                    if (!parent.mkdirs() && !parent.isDirectory()) {
                        throw new IOException("Mkdirs failed to create " + parent.getAbsolutePath());
                    }
                    try (OutputStream out = Files.newOutputStream(file.toPath(), new OpenOption[0]);){
                        org.apache.hadoop.io.IOUtils.copyBytes((InputStream)zip, out, 8192);
                    }
                    if (!file.setLastModified(entry.getTime())) {
                        ++numOfFailedLastModifiedSet;
                    }
                    if (entry.getPlatform() == 3) {
                        Files.setPosixFilePermissions(file.toPath(), FileUtil.permissionsFromMode(entry.getUnixMode()));
                    }
                }
                entry = zip.getNextZipEntry();
            }
            if (numOfFailedLastModifiedSet > 0) {
                LOG.warn("Could not set last modfied time for {} file(s)", (Object)numOfFailedLastModifiedSet);
            }
        }
    }

    private static Set<PosixFilePermission> permissionsFromMode(int mode) {
        EnumSet<PosixFilePermission> permissions = EnumSet.noneOf(PosixFilePermission.class);
        FileUtil.addPermissions(permissions, mode, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_EXECUTE);
        FileUtil.addPermissions(permissions, mode >> 3, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE);
        FileUtil.addPermissions(permissions, mode >> 6, PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE);
        return permissions;
    }

    private static void addPermissions(Set<PosixFilePermission> permissions, int mode, PosixFilePermission r, PosixFilePermission w, PosixFilePermission x) {
        if (((long)mode & 1L) == 1L) {
            permissions.add(x);
        }
        if (((long)mode & 2L) == 2L) {
            permissions.add(w);
        }
        if (((long)mode & 4L) == 4L) {
            permissions.add(r);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void unZip(File inFile, File unzipDir) throws IOException {
        try (ZipFile zipFile = new ZipFile(inFile);){
            Enumeration entries = zipFile.getEntries();
            String targetDirPath = unzipDir.getCanonicalPath() + File.separator;
            while (entries.hasMoreElements()) {
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                if (entry.isDirectory()) continue;
                try (InputStream in = zipFile.getInputStream(entry);){
                    File file = new File(unzipDir, entry.getName());
                    if (!file.getCanonicalPath().startsWith(targetDirPath)) {
                        throw new IOException("expanding " + entry.getName() + " would create file outside of " + unzipDir);
                    }
                    if (!file.getParentFile().mkdirs() && !file.getParentFile().isDirectory()) {
                        throw new IOException("Mkdirs failed to create " + file.getParentFile().toString());
                    }
                    try (OutputStream out = Files.newOutputStream(file.toPath(), new OpenOption[0]);){
                        int i;
                        byte[] buffer = new byte[8192];
                        while ((i = in.read(buffer)) != -1) {
                            out.write(buffer, 0, i);
                        }
                    }
                    if (entry.getPlatform() != 3) continue;
                    Files.setPosixFilePermissions(file.toPath(), FileUtil.permissionsFromMode(entry.getUnixMode()));
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void runCommandOnStream(InputStream inputStream, String command) throws IOException, InterruptedException, ExecutionException {
        int exitCode;
        ExecutorService executor = null;
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        builder.command(Shell.WINDOWS ? "cmd" : "bash", Shell.WINDOWS ? "/c" : "-c", command);
        Process process = builder.start();
        try {
            executor = Executors.newFixedThreadPool(2);
            Future<?> output = executor.submit(() -> {
                block9: {
                    try {
                        if (LOG.isDebugEnabled()) {
                            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream(), StandardCharsets.UTF_8));){
                                String line;
                                while ((line = reader.readLine()) != null) {
                                    LOG.debug(line);
                                }
                                break block9;
                            }
                        }
                        IOUtils.copy((InputStream)process.getInputStream(), (OutputStream)new IOUtils.NullOutputStream());
                    }
                    catch (IOException e) {
                        LOG.debug(e.getMessage());
                    }
                }
            });
            Future<?> error = executor.submit(() -> {
                block9: {
                    try {
                        if (LOG.isDebugEnabled()) {
                            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream(), StandardCharsets.UTF_8));){
                                String line;
                                while ((line = reader.readLine()) != null) {
                                    LOG.debug(line);
                                }
                                break block9;
                            }
                        }
                        IOUtils.copy((InputStream)process.getErrorStream(), (OutputStream)new IOUtils.NullOutputStream());
                    }
                    catch (IOException e) {
                        LOG.debug(e.getMessage());
                    }
                }
            });
            try {
                IOUtils.copy((InputStream)inputStream, (OutputStream)process.getOutputStream());
            }
            finally {
                process.getOutputStream().close();
            }
            error.get();
            output.get();
        }
        finally {
            if (executor != null) {
                executor.shutdown();
            }
            exitCode = process.waitFor();
        }
        if (exitCode != 0) {
            throw new IOException(String.format("Error executing command. %s Process exited with exit code %d.", command, exitCode));
        }
    }

    public static void unTar(InputStream inputStream, File untarDir, boolean gzipped) throws IOException, InterruptedException, ExecutionException {
        if (!untarDir.mkdirs() && !untarDir.isDirectory()) {
            throw new IOException("Mkdirs failed to create " + untarDir);
        }
        if (Shell.WINDOWS) {
            FileUtil.unTarUsingJava(inputStream, untarDir, gzipped);
        } else {
            FileUtil.unTarUsingTar(inputStream, untarDir, gzipped);
        }
    }

    public static void unTar(File inFile, File untarDir) throws IOException {
        if (!untarDir.mkdirs() && !untarDir.isDirectory()) {
            throw new IOException("Mkdirs failed to create " + untarDir);
        }
        boolean gzipped = inFile.toString().endsWith("gz");
        if (Shell.WINDOWS) {
            FileUtil.unTarUsingJava(inFile, untarDir, gzipped);
        } else {
            FileUtil.unTarUsingTar(inFile, untarDir, gzipped);
        }
    }

    private static void unTarUsingTar(InputStream inputStream, File untarDir, boolean gzipped) throws IOException, InterruptedException, ExecutionException {
        StringBuilder untarCommand = new StringBuilder();
        if (gzipped) {
            untarCommand.append("gzip -dc | (");
        }
        untarCommand.append("cd '").append(FileUtil.makeSecureShellPath(untarDir)).append("' && ").append("tar -x ");
        if (gzipped) {
            untarCommand.append(")");
        }
        FileUtil.runCommandOnStream(inputStream, untarCommand.toString());
    }

    private static void unTarUsingTar(File inFile, File untarDir, boolean gzipped) throws IOException {
        StringBuffer untarCommand = new StringBuffer();
        String source = "'" + FileUtil.makeSecureShellPath(inFile) + "'";
        if (gzipped) {
            untarCommand.append(" gzip -dc ").append(source).append(" | (");
        }
        untarCommand.append("cd '").append(FileUtil.makeSecureShellPath(untarDir)).append("' && ").append("tar -xf ");
        if (gzipped) {
            untarCommand.append(" -)");
        } else {
            untarCommand.append(source);
        }
        LOG.debug("executing [{}]", (Object)untarCommand);
        String[] shellCmd = new String[]{"bash", "-c", untarCommand.toString()};
        Shell.ShellCommandExecutor shexec = new Shell.ShellCommandExecutor(shellCmd);
        shexec.execute();
        int exitcode = shexec.getExitCode();
        if (exitcode != 0) {
            throw new IOException("Error untarring file " + inFile + ". Tar process exited with exit code " + exitcode + " from command " + untarCommand);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void unTarUsingJava(File inFile, File untarDir, boolean gzipped) throws IOException {
        InputStream inputStream = null;
        TarArchiveInputStream tis = null;
        try {
            inputStream = gzipped ? new GZIPInputStream(Files.newInputStream(inFile.toPath(), new OpenOption[0])) : Files.newInputStream(inFile.toPath(), new OpenOption[0]);
            inputStream = new BufferedInputStream(inputStream);
            tis = new TarArchiveInputStream(inputStream);
            TarArchiveEntry entry = tis.getNextTarEntry();
            while (entry != null) {
                FileUtil.unpackEntries(tis, entry, untarDir);
                entry = tis.getNextTarEntry();
            }
        }
        catch (Throwable throwable) {
            org.apache.hadoop.io.IOUtils.cleanupWithLogger(LOG, new Closeable[]{tis, inputStream});
            throw throwable;
        }
        org.apache.hadoop.io.IOUtils.cleanupWithLogger(LOG, new Closeable[]{tis, inputStream});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void unTarUsingJava(InputStream inputStream, File untarDir, boolean gzipped) throws IOException {
        TarArchiveInputStream tis = null;
        try {
            if (gzipped) {
                inputStream = new GZIPInputStream(inputStream);
            }
            inputStream = new BufferedInputStream(inputStream);
            tis = new TarArchiveInputStream(inputStream);
            TarArchiveEntry entry = tis.getNextTarEntry();
            while (entry != null) {
                FileUtil.unpackEntries(tis, entry, untarDir);
                entry = tis.getNextTarEntry();
            }
        }
        catch (Throwable throwable) {
            org.apache.hadoop.io.IOUtils.cleanupWithLogger(LOG, new Closeable[]{tis, inputStream});
            throw throwable;
        }
        org.apache.hadoop.io.IOUtils.cleanupWithLogger(LOG, new Closeable[]{tis, inputStream});
    }

    private static void unpackEntries(TarArchiveInputStream tis, TarArchiveEntry entry, File outputDir) throws IOException {
        String canonicalTargetPath;
        String targetDirPath = outputDir.getCanonicalPath() + File.separator;
        File outputFile = new File(outputDir, entry.getName());
        if (!outputFile.getCanonicalPath().startsWith(targetDirPath)) {
            throw new IOException("expanding " + entry.getName() + " would create entry outside of " + outputDir);
        }
        if ((entry.isSymbolicLink() || entry.isLink()) && !(canonicalTargetPath = FileUtil.getCanonicalPath(entry.getLinkName(), outputDir)).startsWith(targetDirPath)) {
            throw new IOException("expanding " + entry.getName() + " would create entry outside of " + outputDir);
        }
        if (entry.isDirectory()) {
            File subDir = new File(outputDir, entry.getName());
            if (!subDir.mkdirs() && !subDir.isDirectory()) {
                throw new IOException("Mkdirs failed to create tar internal dir " + outputDir);
            }
            for (TarArchiveEntry e : entry.getDirectoryEntries()) {
                FileUtil.unpackEntries(tis, e, subDir);
            }
            return;
        }
        if (entry.isSymbolicLink()) {
            canonicalTargetPath = FileUtil.getCanonicalPath(entry.getLinkName(), outputDir);
            Files.createSymbolicLink(FileSystems.getDefault().getPath(outputDir.getPath(), entry.getName()), FileSystems.getDefault().getPath(canonicalTargetPath, new String[0]), new FileAttribute[0]);
            return;
        }
        if (!outputFile.getParentFile().exists() && !outputFile.getParentFile().mkdirs()) {
            throw new IOException("Mkdirs failed to create tar internal dir " + outputDir);
        }
        if (entry.isLink()) {
            canonicalTargetPath = FileUtil.getCanonicalPath(entry.getLinkName(), outputDir);
            File src = new File(canonicalTargetPath);
            HardLink.createHardLink(src, outputFile);
            return;
        }
        FileUtils.copyToFile((InputStream)tis, (File)outputFile);
    }

    private static String getCanonicalPath(String path, File parentDir) throws IOException {
        java.nio.file.Path targetPath = Paths.get(path, new String[0]);
        return (targetPath.isAbsolute() ? new File(path) : new File(parentDir, path)).getCanonicalPath();
    }

    public static int symLink(String target, String linkname) throws IOException {
        Shell.ShellCommandExecutor shExec;
        if (target == null || linkname == null) {
            LOG.warn("Can not create a symLink with a target = " + target + " and link =" + linkname);
            return 1;
        }
        File targetFile = new File(Path.getPathWithoutSchemeAndAuthority(new Path(target)).toString());
        File linkFile = new File(Path.getPathWithoutSchemeAndAuthority(new Path(linkname)).toString());
        String[] cmd = Shell.getSymlinkCommand(targetFile.toString(), linkFile.toString());
        try {
            shExec = Shell.WINDOWS && linkFile.getParentFile() != null && !new Path(target).isAbsolute() ? new Shell.ShellCommandExecutor(cmd, linkFile.getParentFile()) : new Shell.ShellCommandExecutor(cmd);
            shExec.execute();
        }
        catch (Shell.ExitCodeException ec) {
            int returnVal = ec.getExitCode();
            if (Shell.WINDOWS && returnVal == 2) {
                LOG.warn("Fail to create symbolic links on Windows. The default security settings in Windows disallow non-elevated administrators and all non-administrators from creating symbolic links. This behavior can be changed in the Local Security Policy management console");
            } else if (returnVal != 0) {
                LOG.warn("Command '" + StringUtils.join((CharSequence)" ", cmd) + "' failed " + returnVal + " with: " + ec.getMessage());
            }
            return returnVal;
        }
        catch (IOException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Error while create symlink " + linkname + " to " + target + ". Exception: " + StringUtils.stringifyException(e));
            }
            throw e;
        }
        return shExec.getExitCode();
    }

    public static int chmod(String filename, String perm) throws IOException, InterruptedException {
        return FileUtil.chmod(filename, perm, false);
    }

    public static int chmod(String filename, String perm, boolean recursive) throws IOException {
        Shell.ShellCommandExecutor shExec;
        block2: {
            String[] cmd = Shell.getSetPermissionCommand(perm, recursive);
            String[] args = new String[cmd.length + 1];
            System.arraycopy(cmd, 0, args, 0, cmd.length);
            args[cmd.length] = new File(filename).getPath();
            shExec = new Shell.ShellCommandExecutor(args);
            try {
                shExec.execute();
            }
            catch (IOException e) {
                if (!LOG.isDebugEnabled()) break block2;
                LOG.debug("Error while changing permission : " + filename + " Exception: " + StringUtils.stringifyException(e));
            }
        }
        return shExec.getExitCode();
    }

    public static void setOwner(File file, String username, String groupname) throws IOException {
        if (username == null && groupname == null) {
            throw new IOException("username == null && groupname == null");
        }
        String arg = (username == null ? "" : username) + (groupname == null ? "" : ":" + groupname);
        String[] cmd = Shell.getSetOwnerCommand(arg);
        FileUtil.execCommand(file, cmd);
    }

    public static boolean setReadable(File f, boolean readable) {
        if (Shell.WINDOWS) {
            try {
                String permission = readable ? "u+r" : "u-r";
                FileUtil.chmod(f.getCanonicalPath(), permission, false);
                return true;
            }
            catch (IOException ex) {
                return false;
            }
        }
        return f.setReadable(readable);
    }

    public static boolean setWritable(File f, boolean writable) {
        if (Shell.WINDOWS) {
            try {
                String permission = writable ? "u+w" : "u-w";
                FileUtil.chmod(f.getCanonicalPath(), permission, false);
                return true;
            }
            catch (IOException ex) {
                return false;
            }
        }
        return f.setWritable(writable);
    }

    public static boolean setExecutable(File f, boolean executable) {
        if (Shell.WINDOWS) {
            try {
                String permission = executable ? "u+x" : "u-x";
                FileUtil.chmod(f.getCanonicalPath(), permission, false);
                return true;
            }
            catch (IOException ex) {
                return false;
            }
        }
        return f.setExecutable(executable);
    }

    public static boolean canRead(File f) {
        if (Shell.WINDOWS) {
            try {
                return NativeIO.Windows.access(f.getCanonicalPath(), NativeIO.Windows.AccessRight.ACCESS_READ);
            }
            catch (IOException e) {
                return false;
            }
        }
        return f.canRead();
    }

    public static boolean canWrite(File f) {
        if (Shell.WINDOWS) {
            try {
                return NativeIO.Windows.access(f.getCanonicalPath(), NativeIO.Windows.AccessRight.ACCESS_WRITE);
            }
            catch (IOException e) {
                return false;
            }
        }
        return f.canWrite();
    }

    public static boolean canExecute(File f) {
        if (Shell.WINDOWS) {
            try {
                return NativeIO.Windows.access(f.getCanonicalPath(), NativeIO.Windows.AccessRight.ACCESS_EXECUTE);
            }
            catch (IOException e) {
                return false;
            }
        }
        return f.canExecute();
    }

    public static void setPermission(File f, FsPermission permission) throws IOException {
        FsAction other;
        FsAction user = permission.getUserAction();
        FsAction group = permission.getGroupAction();
        if (group != (other = permission.getOtherAction()) || NativeIO.isAvailable() || Shell.WINDOWS) {
            FileUtil.execSetPermission(f, permission);
            return;
        }
        boolean rv = true;
        rv = f.setReadable(group.implies(FsAction.READ), false);
        FileUtil.checkReturnValue(rv, f, permission);
        if (group.implies(FsAction.READ) != user.implies(FsAction.READ)) {
            rv = f.setReadable(user.implies(FsAction.READ), true);
            FileUtil.checkReturnValue(rv, f, permission);
        }
        rv = f.setWritable(group.implies(FsAction.WRITE), false);
        FileUtil.checkReturnValue(rv, f, permission);
        if (group.implies(FsAction.WRITE) != user.implies(FsAction.WRITE)) {
            rv = f.setWritable(user.implies(FsAction.WRITE), true);
            FileUtil.checkReturnValue(rv, f, permission);
        }
        rv = f.setExecutable(group.implies(FsAction.EXECUTE), false);
        FileUtil.checkReturnValue(rv, f, permission);
        if (group.implies(FsAction.EXECUTE) != user.implies(FsAction.EXECUTE)) {
            rv = f.setExecutable(user.implies(FsAction.EXECUTE), true);
            FileUtil.checkReturnValue(rv, f, permission);
        }
    }

    private static void checkReturnValue(boolean rv, File p, FsPermission permission) throws IOException {
        if (!rv) {
            throw new IOException("Failed to set permissions of path: " + p + " to " + String.format("%04o", permission.toShort()));
        }
    }

    private static void execSetPermission(File f, FsPermission permission) throws IOException {
        if (NativeIO.isAvailable()) {
            NativeIO.POSIX.chmod(f.getCanonicalPath(), permission.toShort());
        } else {
            FileUtil.execCommand(f, Shell.getSetPermissionCommand(String.format("%04o", permission.toShort()), false));
        }
    }

    static String execCommand(File f, String ... cmd) throws IOException {
        String[] args = new String[cmd.length + 1];
        System.arraycopy(cmd, 0, args, 0, cmd.length);
        args[cmd.length] = f.getCanonicalPath();
        String output = Shell.execCommand(args);
        return output;
    }

    public static final File createLocalTempFile(File basefile, String prefix, boolean isDeleteOnExit) throws IOException {
        File tmp = File.createTempFile(prefix + basefile.getName(), "", basefile.getParentFile());
        if (isDeleteOnExit) {
            tmp.deleteOnExit();
        }
        return tmp;
    }

    public static void replaceFile(File src, File target) throws IOException {
        if (!src.renameTo(target)) {
            int retries = 5;
            while (target.exists() && !target.delete() && retries-- >= 0) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException e) {
                    throw new IOException("replaceFile interrupted.");
                }
            }
            if (!src.renameTo(target)) {
                throw new IOException("Unable to rename " + src + " to " + target);
            }
        }
    }

    public static File[] listFiles(File dir) throws IOException {
        File[] files = dir.listFiles();
        if (files == null) {
            throw new IOException("Invalid directory or I/O error occurred for dir: " + dir.toString());
        }
        return files;
    }

    public static String[] list(File dir) throws IOException {
        if (!FileUtil.canRead(dir)) {
            throw new AccessDeniedException(dir.toString(), null, "Permission denied");
        }
        String[] fileNames = dir.list();
        if (fileNames == null) {
            throw new IOException("Invalid directory or I/O error occurred for dir: " + dir.toString());
        }
        return fileNames;
    }

    public static String[] createJarWithClassPath(String inputClassPath, Path pwd, Map<String, String> callerEnv) throws IOException {
        return FileUtil.createJarWithClassPath(inputClassPath, pwd, pwd, callerEnv);
    }

    public static String[] createJarWithClassPath(String inputClassPath, Path pwd, Path targetDir, Map<String, String> callerEnv) throws IOException {
        CaseInsensitiveMap env = Shell.WINDOWS ? new CaseInsensitiveMap((Map)callerEnv) : callerEnv;
        String[] classPathEntries = inputClassPath.split(File.pathSeparator);
        for (int i = 0; i < classPathEntries.length; ++i) {
            classPathEntries[i] = StringUtils.replaceTokens(classPathEntries[i], StringUtils.ENV_VAR_PATTERN, (Map<String, String>)env);
        }
        File workingDir = new File(pwd.toString());
        if (!workingDir.mkdirs()) {
            LOG.debug("mkdirs false for " + workingDir + ", execution will continue");
        }
        StringBuilder unexpandedWildcardClasspath = new StringBuilder();
        ArrayList<String> classPathEntryList = new ArrayList<String>(classPathEntries.length);
        for (String classPathEntry : classPathEntries) {
            if (classPathEntry.length() == 0) continue;
            if (classPathEntry.endsWith("*")) {
                List<Path> jars = FileUtil.getJarsInDirectory(classPathEntry);
                if (!jars.isEmpty()) {
                    for (Path jar : jars) {
                        classPathEntryList.add(jar.toUri().toURL().toExternalForm());
                    }
                    continue;
                }
                unexpandedWildcardClasspath.append(File.pathSeparator).append(classPathEntry);
                continue;
            }
            File fileCpEntry = null;
            fileCpEntry = !new Path(classPathEntry).isAbsolute() ? new File(targetDir.toString(), classPathEntry) : new File(classPathEntry);
            String classPathEntryUrl = fileCpEntry.toURI().toURL().toExternalForm();
            if (classPathEntry.endsWith("/") && !classPathEntryUrl.endsWith("/")) {
                classPathEntryUrl = classPathEntryUrl + "/";
            }
            classPathEntryList.add(classPathEntryUrl);
        }
        String jarClassPath = StringUtils.join((CharSequence)" ", classPathEntryList);
        Manifest jarManifest = new Manifest();
        jarManifest.getMainAttributes().putValue(Attributes.Name.MANIFEST_VERSION.toString(), "1.0");
        jarManifest.getMainAttributes().putValue(Attributes.Name.CLASS_PATH.toString(), jarClassPath);
        File classPathJar = File.createTempFile("classpath-", ".jar", workingDir);
        try (OutputStream fos = Files.newOutputStream(classPathJar.toPath(), new OpenOption[0]);
             BufferedOutputStream bos = new BufferedOutputStream(fos);){
            JarOutputStream jos = new JarOutputStream((OutputStream)bos, jarManifest);
            jos.close();
        }
        String[] jarCp = new String[]{classPathJar.getCanonicalPath(), unexpandedWildcardClasspath.toString()};
        return jarCp;
    }

    public static List<Path> getJarsInDirectory(String path) {
        return FileUtil.getJarsInDirectory(path, true);
    }

    public static List<Path> getJarsInDirectory(String path, boolean useLocal) {
        ArrayList<Path> paths = new ArrayList<Path>();
        try {
            if (!path.endsWith("*")) {
                path = path + File.separator + "*";
            }
            Path globPath = new Path(path).suffix("{.jar,.JAR}");
            FileContext context = useLocal ? FileContext.getLocalFSFileContext() : FileContext.getFileContext(globPath.toUri());
            FileStatus[] files = context.util().globStatus(globPath);
            if (files != null) {
                for (FileStatus file : files) {
                    paths.add(file.getPath());
                }
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return paths;
    }

    public static boolean compareFs(FileSystem srcFs, FileSystem destFs) {
        if (srcFs == null || destFs == null) {
            return false;
        }
        URI srcUri = srcFs.getUri();
        URI dstUri = destFs.getUri();
        if (srcUri.getScheme() == null) {
            return false;
        }
        if (!srcUri.getScheme().equals(dstUri.getScheme())) {
            return false;
        }
        String srcHost = srcUri.getHost();
        String dstHost = dstUri.getHost();
        if (srcHost != null && dstHost != null) {
            if (srcHost.equals(dstHost)) {
                return srcUri.getPort() == dstUri.getPort();
            }
            try {
                srcHost = InetAddress.getByName(srcHost).getCanonicalHostName();
                dstHost = InetAddress.getByName(dstHost).getCanonicalHostName();
            }
            catch (UnknownHostException ue) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Could not compare file-systems. Unknown host: ", (Throwable)ue);
                }
                return false;
            }
            if (!srcHost.equals(dstHost)) {
                return false;
            }
        } else {
            if (srcHost == null && dstHost != null) {
                return false;
            }
            if (srcHost != null) {
                return false;
            }
        }
        return srcUri.getPort() == dstUri.getPort();
    }

    public static FileSystem write(FileSystem fs, Path path, byte[] bytes) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(bytes);
        try (Object out = ((FSDataOutputStreamBuilder)fs.createFile(path).overwrite(true)).build();){
            ((FilterOutputStream)out).write(bytes);
        }
        return fs;
    }

    public static FileContext write(FileContext fileContext, Path path, byte[] bytes) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(bytes);
        try (Object out = ((FSDataOutputStreamBuilder)fileContext.create(path).overwrite(true)).build();){
            ((FilterOutputStream)out).write(bytes);
        }
        return fileContext;
    }

    public static FileSystem write(FileSystem fs, Path path, Iterable<? extends CharSequence> lines, Charset cs) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(lines);
        Objects.requireNonNull(cs);
        CharsetEncoder encoder = cs.newEncoder();
        try (Object out = ((FSDataOutputStreamBuilder)fs.createFile(path).overwrite(true)).build();
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)out, encoder));){
            for (CharSequence charSequence : lines) {
                writer.append(charSequence);
                writer.newLine();
            }
        }
        return fs;
    }

    public static FileContext write(FileContext fileContext, Path path, Iterable<? extends CharSequence> lines, Charset cs) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(lines);
        Objects.requireNonNull(cs);
        CharsetEncoder encoder = cs.newEncoder();
        try (Object out = ((FSDataOutputStreamBuilder)fileContext.create(path).overwrite(true)).build();
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)out, encoder));){
            for (CharSequence charSequence : lines) {
                writer.append(charSequence);
                writer.newLine();
            }
        }
        return fileContext;
    }

    public static FileSystem write(FileSystem fs, Path path, CharSequence charseq, Charset cs) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(charseq);
        Objects.requireNonNull(cs);
        CharsetEncoder encoder = cs.newEncoder();
        try (Object out = ((FSDataOutputStreamBuilder)fs.createFile(path).overwrite(true)).build();
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)out, encoder));){
            writer.append(charseq);
        }
        return fs;
    }

    public static FileContext write(FileContext fs, Path path, CharSequence charseq, Charset cs) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(charseq);
        Objects.requireNonNull(cs);
        CharsetEncoder encoder = cs.newEncoder();
        try (Object out = ((FSDataOutputStreamBuilder)fs.create(path).overwrite(true)).build();
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)out, encoder));){
            writer.append(charseq);
        }
        return fs;
    }

    public static FileSystem write(FileSystem fs, Path path, CharSequence charseq) throws IOException {
        return FileUtil.write(fs, path, charseq, StandardCharsets.UTF_8);
    }

    public static FileContext write(FileContext fileContext, Path path, CharSequence charseq) throws IOException {
        return FileUtil.write(fileContext, path, charseq, StandardCharsets.UTF_8);
    }

    @InterfaceAudience.LimitedPrivate(value={"ViewDistributedFileSystem"})
    @InterfaceStability.Unstable
    public static void rename(FileSystem srcFs, Path src, Path dst, Options.Rename ... options) throws IOException {
        srcFs.rename(src, dst, options);
    }

    public static void maybeIgnoreMissingDirectory(FileSystem fs, Path path, FileNotFoundException e) throws FileNotFoundException {
        boolean b;
        try {
            b = !fs.hasPathCapability(path, "fs.capability.directory.listing.inconsistent");
        }
        catch (IOException ex) {
            e.addSuppressed(ex);
            throw e;
        }
        if (b) {
            throw e;
        }
        LOG.info("Ignoring missing directory {}", (Object)path);
        LOG.debug("Directory missing", (Throwable)e);
    }

    @Deprecated
    public static class HardLink
    extends org.apache.hadoop.fs.HardLink {
    }
}

