/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.tools;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.tools.DistCpOptions.FileAttribute;
import org.apache.hadoop.tools.util.DistCpUtils;
import org.apache.hadoop.tools.util.ProducerConsumer;
import org.apache.hadoop.tools.util.WorkReport;
import org.apache.hadoop.tools.util.WorkRequest;
import org.apache.hadoop.tools.util.WorkRequestProcessor;
import org.apache.hadoop.mapreduce.security.TokenCache;
import org.apache.hadoop.security.Credentials;

import com.google.common.annotations.VisibleForTesting;

import java.io.*;
import java.util.ArrayList;
import java.util.Set;
import java.util.List;
import java.util.Map;
import java.util.HashMap;

import static org.apache.hadoop.tools.DistCpConstants
        .HDFS_RESERVED_RAW_DIRECTORY_NAME;
import static org.apache.hadoop.tools.util.DistCpUtils.getOriginalFileStatus;

/**
 * The SimpleCopyListing is responsible for making the exhaustive list of
 * all files/directories under its specified list of input-paths.
 * These are written into the specified copy-listing file.
 * Note: The SimpleCopyListing doesn't handle wild-cards in the input-paths.
 */
public class SimpleCopyListing extends CopyListing {
  private static final Logger LOG = LoggerFactory.getLogger(SimpleCopyListing.class);

  private long totalPaths = 0;
  private long totalDirs = 0;
  private long totalBytesToCopy = 0;
  private int numListstatusThreads = 1;
  private final int maxRetries = 3;
  private CopyFilter copyFilter;
  public static Map<String, Set<String>> loopLocator = new HashMap<>();

  /**
   * Protected constructor, to initialize configuration.
   *
   * @param configuration The input configuration, with which the source/target FileSystems may be accessed.
   * @param credentials - Credentials object on which the FS delegation tokens are cached. If null
   * delegation token caching is skipped
   */
  protected SimpleCopyListing(Configuration configuration, Credentials credentials) {
    super(configuration, credentials);
    numListstatusThreads = getConf().getInt(
        DistCpConstants.CONF_LABEL_LISTSTATUS_THREADS,
        DistCpConstants.DEFAULT_LISTSTATUS_THREADS);
    copyFilter = CopyFilter.getCopyFilter(getConf());
    copyFilter.initialize();
  }

  @VisibleForTesting
  protected SimpleCopyListing(Configuration configuration, Credentials credentials,
                              int numListstatusThreads) {
    super(configuration, credentials);
    this.numListstatusThreads = numListstatusThreads;
  }

  @Override
  protected void validatePaths(DistCpOptions options)
      throws IOException, InvalidInputException {

    Path targetPath = options.getTargetPath();
    FileSystem targetFS = targetPath.getFileSystem(getConf());
    boolean targetIsFile = targetFS.isFile(targetPath);
    targetPath = targetFS.makeQualified(targetPath);
    final boolean targetIsReservedRaw =
        Path.getPathWithoutSchemeAndAuthority(targetPath).toString().
            startsWith(HDFS_RESERVED_RAW_DIRECTORY_NAME);

    //If target is a file, then source has to be single file
    if (targetIsFile) {
      if (options.getSourcePaths().size() > 1) {
        throw new InvalidInputException("Multiple source being copied to a file: " +
            targetPath);
      }

      Path srcPath = options.getSourcePaths().get(0);
      FileSystem sourceFS = srcPath.getFileSystem(getConf());
      if (!sourceFS.isFile(srcPath)) {
        throw new InvalidInputException("Cannot copy " + srcPath +
            ", which is not a file to " + targetPath);
      }
    }

    if (options.shouldAtomicCommit() && targetFS.exists(targetPath)) {
      throw new InvalidInputException("Target path for atomic-commit already exists: " +
        targetPath + ". Cannot atomic-commit to pre-existing target-path.");
    }

    for (Path path: options.getSourcePaths()) {
      FileSystem fs = path.getFileSystem(getConf());
      if (!fs.exists(path)) {
        throw new InvalidInputException(path + " doesn't exist");
      }
      if (Path.getPathWithoutSchemeAndAuthority(path).toString().
          startsWith(HDFS_RESERVED_RAW_DIRECTORY_NAME)) {
        if (!targetIsReservedRaw) {
          final String msg = "The source path '" + path + "' starts with " +
              HDFS_RESERVED_RAW_DIRECTORY_NAME + " but the target path '" +
              targetPath + "' does not. Either all or none of the paths must " +
              "have this prefix.";
          throw new InvalidInputException(msg);
        }
      } else if (targetIsReservedRaw) {
        final String msg = "The target path '" + targetPath + "' starts with " +
                HDFS_RESERVED_RAW_DIRECTORY_NAME + " but the source path '" +
                path + "' does not. Either all or none of the paths must " +
                "have this prefix.";
        throw new InvalidInputException(msg);
      }
    }

    if (targetIsReservedRaw) {
      options.preserveRawXattrs();
      getConf().setBoolean(DistCpConstants.CONF_LABEL_PRESERVE_RAWXATTRS, true);
    }

    /* This is requires to allow map tasks to access each of the source
       clusters. This would retrieve the delegation token for each unique
       file system and add them to job's private credential store
     */
    Credentials credentials = getCredentials();
    if (credentials != null) {
      Path[] inputPaths = options.getSourcePaths().toArray(new Path[1]);
      TokenCache.obtainTokensForNamenodes(credentials, inputPaths, getConf());
    }
  }

  /** {@inheritDoc} */
  @Override
  public void doBuildListing(Path pathToListingFile, DistCpOptions options) throws IOException {
    doBuildListing(getWriter(pathToListingFile), options);
  }
  /**
   * Collect the list of 
   *   {@literal <sourceRelativePath, sourceFileStatus>}
   * to be copied and write to the sequence file. In essence, any file or
   * directory that need to be copied or sync-ed is written as an entry to the
   * sequence file, with the possible exception of the source root:
   *     when either -update (sync) or -overwrite switch is specified, and if
   *     the the source root is a directory, then the source root entry is not 
   *     written to the sequence file, because only the contents of the source
   *     directory need to be copied in this case.
   * See {@link org.apache.hadoop.tools.util.DistCpUtils#getRelativePath} for
   *     how relative path is computed.
   * See computeSourceRootPath method for how the root path of the source is
   *     computed.
   * @param fileListWriter
   * @param options
   * @throws IOException
   */
  @VisibleForTesting
  public void doBuildListing(SequenceFile.Writer fileListWriter,
      DistCpOptions options) throws IOException {
    if (options.getNumListstatusThreads() > 0) {
      numListstatusThreads = options.getNumListstatusThreads();
    }

    try {
      for (Path path: options.getSourcePaths()) {
        FileSystem sourceFS = path.getFileSystem(getConf());
        final boolean preserveAcls = options.shouldPreserve(FileAttribute.ACL);
        final boolean preserveXAttrs = options.shouldPreserve(FileAttribute.XATTR);
        final boolean preserveRawXAttrs = options.shouldPreserveRawXattrs();
        final boolean keepLinks = options.shouldKeepLinks();
        path = makeQualified(path);

        FileStatus rootStatus = sourceFS.getFileStatus(path);
        FileListingEntry listingEntryRoot = getOriginalFileStatus(rootStatus, getConf(), keepLinks, loopLocator);
        FileStatus[] sourceFiles = sourceFS.listStatus(listingEntryRoot.getSourceRealPath().getPath());
        Path sourcePathRoot = computeSourceRootPath(listingEntryRoot.getSourceRealPath(), options);
        String relativePathPrefix = listingEntryRoot.getSourceRealPath().getPath().equals(sourcePathRoot) ? "" : Path.SEPARATOR + listingEntryRoot.getSourceRealPath().getPath().getName();
        boolean explore = (sourceFiles != null && sourceFiles.length > 0);
        if (!explore || rootStatus.isDirectory()
                || (rootStatus.isSymlink() && listingEntryRoot.getSourceRealPath().isDirectory())
                || (rootStatus.isSymlink() && keepLinks)) {
          DistCpUtils.toCopyListingFileStatus(sourceFS, listingEntryRoot,
                preserveAcls, preserveXAttrs, preserveRawXAttrs,
                options.getBlocksPerChunk());
          writeToFileListingRoot(fileListWriter, listingEntryRoot,
              sourcePathRoot, options, relativePathPrefix);
        }
        if (explore && !(keepLinks && rootStatus.isSymlink())) {
          ArrayList<FileListingEntry> sourceDirs = new ArrayList<FileListingEntry>();
          for (FileStatus sourceStatus: sourceFiles) {
            if (LOG.isDebugEnabled()) {
              LOG.debug("Recording source-path: " + sourceStatus.getPath() + " for copy.");
            }
            FileListingEntry listingEntry = getOriginalFileStatus(sourceStatus, getConf(), keepLinks, loopLocator);
            DistCpUtils.toCopyListingFileStatus(sourceFS, listingEntry,
                    preserveAcls && listingEntry.getSourceRealPath().isDirectory(),
                    preserveXAttrs && listingEntry.getSourceRealPath().isDirectory(),
                    preserveRawXAttrs && listingEntry.getSourceRealPath().isDirectory(),
                    options.getBlocksPerChunk());
            writeToFileListing(fileListWriter, listingEntry, relativePathPrefix + DistCpUtils.getRelativePath(listingEntry));
            if (listingEntry.getSourceRealPath().isDirectory()) {
              if (LOG.isDebugEnabled()) {
                LOG.debug("Adding source dir for traverse: " + listingEntry.getSourceRealPath().getPath());
              }
              sourceDirs.add(listingEntry);
            }
          }
          traverseDirectory(fileListWriter, sourceFS, sourceDirs,
                  relativePathPrefix, options);
        }
      }
      fileListWriter.close();
      printStats();
      LOG.info("Build file listing completed.");
      fileListWriter = null;
    } finally {
      IOUtils.cleanupWithLogger(LOG, fileListWriter);
    }
  }

  private Path computeSourceRootPath(FileStatus sourceStatus,
                                     DistCpOptions options) throws IOException {

    Path target = options.getTargetPath();
    FileSystem targetFS = target.getFileSystem(getConf());
    final boolean targetPathExists = options.getTargetPathExists();

    boolean solitaryFile = options.getSourcePaths().size() == 1
            && !sourceStatus.isDirectory()
            && !(sourceStatus.isSymlink() && getOriginalFileStatus(sourceStatus, getConf(), options.shouldKeepLinks(), loopLocator).getSourceRealPath().isDirectory());

    if (solitaryFile) {
      return sourceStatus.getPath();
    } else {
      boolean specialHandling = (options.getSourcePaths().size() == 1 && !targetPathExists) ||
          options.shouldSyncFolder() || options.shouldOverwrite();

      boolean simpleFile = !sourceStatus.isDirectory() &&
              !(sourceStatus.isSymlink() && getOriginalFileStatus(sourceStatus, getConf(), options.shouldKeepLinks(), loopLocator).getSourceRealPath().isDirectory());
      if((specialHandling && (sourceStatus.isDirectory() || !simpleFile))
              || simpleFile) {
        return sourceStatus.getPath();
      } else {
        return sourceStatus.getPath().getParent();
      }
    }
  }

  /**
   * Provide an option to skip copy of a path, Allows for exclusion
   * of files such as {@link org.apache.hadoop.mapreduce.lib.output.FileOutputCommitter#SUCCEEDED_FILE_NAME}
   * @param path - Path being considered for copy while building the file listing
   * @return - True if the path should be considered for copy, false otherwise
   */
  protected boolean shouldCopy(Path path) {
    return copyFilter.shouldCopy(path);
  }

  /** {@inheritDoc} */
  @Override
  protected long getBytesToCopy() {
    return totalBytesToCopy;
  }

  /** {@inheritDoc} */
  @Override
  protected long getNumberOfPaths() {
    return totalPaths;
  }

  private Path makeQualified(Path path) throws IOException {
    final FileSystem fs = path.getFileSystem(getConf());
    return path.makeQualified(fs.getUri(), fs.getWorkingDirectory());
  }

  private SequenceFile.Writer getWriter(Path pathToListFile) throws IOException {
    FileSystem fs = pathToListFile.getFileSystem(getConf());
    if (fs.exists(pathToListFile)) {
      fs.delete(pathToListFile, false);
    }
    return SequenceFile.createWriter(getConf(),
            SequenceFile.Writer.file(pathToListFile),
            SequenceFile.Writer.keyClass(Text.class),
            SequenceFile.Writer.valueClass(CopyListingFileStatus.class),
            SequenceFile.Writer.compression(SequenceFile.CompressionType.NONE));
  }

  /*
   *  Private class to implement WorkRequestProcessor interface. It processes
   *  each directory (represented by FileStatus item) and returns a list of all
   *  file-system objects in that directory (files and directories). In case of
   *  retriable exceptions it increments retry counter and returns the same
   *  directory for later retry.
   */
  private static class FileStatusProcessor
      implements WorkRequestProcessor<FileListingEntry, List<FileListingEntry>> {
    private FileSystem fileSystem;

    public FileStatusProcessor(FileSystem fileSystem) {
      this.fileSystem = fileSystem;
    }

    /*
     *  Processor for FileSystem.listStatus().
     *
     *  @param workRequest  Input work item that contains FileStatus item which
     *                      is a parent directory we want to list.
     *  @return Outputs WorkReport<FileStatus[]> with a list of objects in the
     *          directory (array of objects, empty if parent directory is
     *          empty). In case of intermittent exception we increment retry
     *          counter and return the list containing the parent directory).
     */
    public WorkReport<List<FileListingEntry>> processItem(
        WorkRequest<FileListingEntry> workRequest) {
      FileListingEntry parent = workRequest.getItem();
      int retry = workRequest.getRetry();
      WorkReport<List<FileListingEntry>> result = null;
      try {
        if (retry > 0) {
          int sleepSeconds = 2;
          for (int i = 1; i < retry; i++) {
            sleepSeconds *= 2;
          }
          try {
            Thread.sleep(1000 * sleepSeconds);
          } catch (InterruptedException ie) {
            LOG.debug("Interrupted while sleeping in exponential backoff.");
          }
        }
        FileStatus[] childFileStatusList = fileSystem.listStatus(parent.getSourceRealPath().getPath());
        List<FileListingEntry> childEntryList = new ArrayList<FileListingEntry>();
        for(FileStatus status: childFileStatusList) {
          FileListingEntry childListingEntry = getOriginalFileStatus(status, new Configuration(), parent.isKeepLink(), loopLocator);
          childListingEntry.setParent(parent);
          childEntryList.add(childListingEntry);
        }
        result = new WorkReport<List<FileListingEntry>>(childEntryList, 0, true);
      } catch (FileNotFoundException fnf) {
        LOG.error("FileNotFoundException exception in listStatus: " +
                  fnf.getMessage());
        result = new WorkReport<List<FileListingEntry>>(new ArrayList<FileListingEntry>(), 0, true, fnf);
      } catch (Exception e) {
        LOG.error("Exception in listStatus. Will send for retry.");
        List<FileListingEntry> childEntryList = new ArrayList<FileListingEntry>();
        childEntryList.add(parent);
        result = new WorkReport<List<FileListingEntry>>(childEntryList, retry + 1, false, e);
      }
      return result;
    }
  }

  private void printStats() {
    LOG.info("Paths (files+dirs) cnt = " + totalPaths +
             "; dirCnt = " + totalDirs);
  }

  private void maybePrintStats() {
    if (totalPaths % 100000 == 0) {
      printStats();
    }
  }

  private void traverseDirectory(SequenceFile.Writer fileListWriter,
                                 FileSystem sourceFS,
                                 ArrayList<FileListingEntry> sourceDirs,
                                 String relativePathPrefix,
                                 DistCpOptions options)
                                 throws IOException {
    final boolean preserveAcls = options.shouldPreserve(FileAttribute.ACL);
    final boolean preserveXAttrs = options.shouldPreserve(FileAttribute.XATTR);
    final boolean preserveRawXattrs = options.shouldPreserveRawXattrs();

    assert numListstatusThreads > 0;
    LOG.debug("Starting thread pool of " + numListstatusThreads +
              " listStatus workers.");
    ProducerConsumer<FileListingEntry, List<FileListingEntry>> workers =
        new ProducerConsumer<FileListingEntry, List<FileListingEntry>>(numListstatusThreads);
    for (int i = 0; i < numListstatusThreads; i++) {
      workers.addWorker(
          new FileStatusProcessor(sourceFS));
    }

    for (FileListingEntry entry : sourceDirs) {
      workers.put(new WorkRequest<FileListingEntry>(entry, 0));
      maybePrintStats();
    }

    while (workers.hasWork()) {
      try {
        WorkReport<List<FileListingEntry>> workResult = workers.take();
        int retry = workResult.getRetry();
        List<FileListingEntry> childs = workResult.getItem();
        for (FileListingEntry child: childs) {
          if (LOG.isDebugEnabled()) {
            LOG.debug("Recording source-path: " + child.getSourceRealPath().getPath() + " for copy.");
          }
          if (retry == 0) {
            DistCpUtils.toCopyListingFileStatus(sourceFS, child,
                preserveAcls && child.getSourceRealPath().isDirectory(),
                preserveXAttrs && child.getSourceRealPath().isDirectory(),
                preserveRawXattrs && child.getSourceRealPath().isDirectory(),
                    options.getBlocksPerChunk());
            writeToFileListing(fileListWriter, child, relativePathPrefix + DistCpUtils.getRelativePath(child));
          }
          if (retry < maxRetries) {
            if (child.getSourceRealPath().isDirectory()) {
              if (LOG.isDebugEnabled()) {
                LOG.debug("Traversing into source dir: " + child.getSourceRealPath().getPath());
              }
              workers.put(new WorkRequest<FileListingEntry>(child, retry));
              maybePrintStats();
            }
          } else {
            LOG.error("Giving up on " + child.getSourceRealPath().getPath() +
                      " after " + retry + " retries.");
          }
        }
      } catch (InterruptedException ie) {
        LOG.error("Could not get item from childQueue. Retrying...");
      }
    }
    workers.shutdown();
  }

  private void writeToFileListingRoot(SequenceFile.Writer fileListWriter,
                                      FileListingEntry listingEntry, Path sourcePathRoot,
                                      DistCpOptions options, String relativePathPrefix) throws IOException {
    boolean syncOrOverwrite = options.shouldSyncFolder() ||
        options.shouldOverwrite();
    if (listingEntry.getSourceRealPath().getPath().equals(sourcePathRoot) &&
            listingEntry.getSourceRealPath().isDirectory() && syncOrOverwrite) {
      // Skip the root-paths when syncOrOverwrite
      if (LOG.isDebugEnabled()) {
        LOG.debug("Skip " + listingEntry.getSourceRealPath().getPath());
      }
      return;
    }
    writeToFileListing(fileListWriter, listingEntry, relativePathPrefix);
  }

  private void writeToFileListing(SequenceFile.Writer fileListWriter,
                                  FileListingEntry listingEntry, String relativePath) throws IOException {
    for(CopyListingFileStatus fileStatus: listingEntry.getCopyListingFileStatus()) {
      if (LOG.isDebugEnabled()) {
        LOG.debug("RELATIVE TARGET PATH: " + relativePath
                + ", REAL FILE PATH: " + fileStatus.getPath());
      }

      if (!shouldCopy(fileStatus.getPath())) {
        return;
      }

      fileListWriter.append(new Text(relativePath), fileStatus);
      fileListWriter.sync();

      if (!fileStatus.isDirectory()) {
        totalBytesToCopy += fileStatus.getSizeToCopy();
      } else {
        totalDirs++;
      }
      totalPaths++;
    }
  }
}
