/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.io.Closeable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Locale;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.AbfsStatistic;
import org.apache.hadoop.fs.azurebfs.AzureBlobFileSystemStore;
import org.apache.hadoop.fs.azurebfs.constants.AbfsHttpConstants;
import org.apache.hadoop.fs.azurebfs.constants.HttpOperationType;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsDriverException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsInvalidChecksumException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidUriException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.SASTokenProviderException;
import org.apache.hadoop.fs.azurebfs.contracts.services.AppendRequestParameters;
import org.apache.hadoop.fs.azurebfs.contracts.services.AzureServiceErrorCode;
import org.apache.hadoop.fs.azurebfs.extensions.EncryptionContextProvider;
import org.apache.hadoop.fs.azurebfs.extensions.ExtensionHelper;
import org.apache.hadoop.fs.azurebfs.extensions.SASTokenProvider;
import org.apache.hadoop.fs.azurebfs.oauth2.AccessTokenProvider;
import org.apache.hadoop.fs.azurebfs.security.ContextEncryptionAdapter;
import org.apache.hadoop.fs.azurebfs.services.AbfsApacheHttpClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsClientContext;
import org.apache.hadoop.fs.azurebfs.services.AbfsClientRenameResult;
import org.apache.hadoop.fs.azurebfs.services.AbfsCounters;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpHeader;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsPerfTracker;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperationType;
import org.apache.hadoop.fs.azurebfs.services.AbfsRetryPolicy;
import org.apache.hadoop.fs.azurebfs.services.AbfsThrottlingIntercept;
import org.apache.hadoop.fs.azurebfs.services.AbfsThrottlingInterceptFactory;
import org.apache.hadoop.fs.azurebfs.services.AbfsUriQueryBuilder;
import org.apache.hadoop.fs.azurebfs.services.AuthType;
import org.apache.hadoop.fs.azurebfs.services.ExponentialRetryPolicy;
import org.apache.hadoop.fs.azurebfs.services.KeepAliveCache;
import org.apache.hadoop.fs.azurebfs.services.SharedKeyCredentials;
import org.apache.hadoop.fs.azurebfs.services.StaticRetryPolicy;
import org.apache.hadoop.fs.azurebfs.utils.EncryptionType;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.fs.store.LogExactlyOnce;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.ssl.DelegatingSSLSocketFactory;
import org.apache.hadoop.thirdparty.com.google.common.base.Strings;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.FutureCallback;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.Futures;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableScheduledFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListeningScheduledExecutorService;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.MoreExecutors;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AbfsClient
implements Closeable {
    public static final Logger LOG = LoggerFactory.getLogger(AbfsClient.class);
    public static final String HUNDRED_CONTINUE_USER_AGENT = " 100-continue;";
    private final URL baseUrl;
    private final SharedKeyCredentials sharedKeyCredentials;
    private AbfsHttpConstants.ApiVersion xMsVersion = AbfsHttpConstants.ApiVersion.getCurrentVersion();
    private final ExponentialRetryPolicy exponentialRetryPolicy;
    private final StaticRetryPolicy staticRetryPolicy;
    private final String filesystem;
    private final AbfsConfiguration abfsConfiguration;
    private final String userAgent;
    private final AbfsPerfTracker abfsPerfTracker;
    private String clientProvidedEncryptionKey = null;
    private String clientProvidedEncryptionKeySHA = null;
    private final String accountName;
    private final AuthType authType;
    private AccessTokenProvider tokenProvider;
    private SASTokenProvider sasTokenProvider;
    private final AbfsCounters abfsCounters;
    private EncryptionContextProvider encryptionContextProvider = null;
    private EncryptionType encryptionType = EncryptionType.NONE;
    private final AbfsThrottlingIntercept intercept;
    private final ListeningScheduledExecutorService executorService;
    private boolean renameResilience;
    private KeepAliveCache keepAliveCache;
    private AbfsApacheHttpClient abfsApacheHttpClient;
    private static final LogExactlyOnce ABFS_METADATA_INCOMPLETE_RENAME_FAILURE = new LogExactlyOnce(LOG);

    private AbfsClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, EncryptionContextProvider encryptionContextProvider, AbfsClientContext abfsClientContext) throws IOException {
        this.baseUrl = baseUrl;
        this.sharedKeyCredentials = sharedKeyCredentials;
        String baseUrlString = baseUrl.toString();
        this.filesystem = baseUrlString.substring(baseUrlString.lastIndexOf("/") + 1);
        this.abfsConfiguration = abfsConfiguration;
        this.exponentialRetryPolicy = abfsClientContext.getExponentialRetryPolicy();
        this.staticRetryPolicy = abfsClientContext.getStaticRetryPolicy();
        this.accountName = abfsConfiguration.getAccountName().substring(0, abfsConfiguration.getAccountName().indexOf("."));
        this.authType = abfsConfiguration.getAuthType(this.accountName);
        this.intercept = AbfsThrottlingInterceptFactory.getInstance(this.accountName, abfsConfiguration);
        this.renameResilience = abfsConfiguration.getRenameResilience();
        if (encryptionContextProvider != null) {
            this.encryptionContextProvider = encryptionContextProvider;
            this.xMsVersion = AbfsHttpConstants.ApiVersion.APR_10_2021;
            this.encryptionType = EncryptionType.ENCRYPTION_CONTEXT;
        } else if (abfsConfiguration.getEncodedClientProvidedEncryptionKey() != null) {
            this.clientProvidedEncryptionKey = abfsConfiguration.getEncodedClientProvidedEncryptionKey();
            this.clientProvidedEncryptionKeySHA = abfsConfiguration.getEncodedClientProvidedEncryptionKeySHA();
            this.encryptionType = EncryptionType.GLOBAL_KEY;
        }
        String sslProviderName = null;
        if (this.baseUrl.toString().startsWith("https")) {
            try {
                LOG.trace("Initializing DelegatingSSLSocketFactory with {} SSL Channel Mode", (Object)this.abfsConfiguration.getPreferredSSLFactoryOption());
                DelegatingSSLSocketFactory.initializeDefaultFactory((DelegatingSSLSocketFactory.SSLChannelMode)this.abfsConfiguration.getPreferredSSLFactoryOption());
                sslProviderName = DelegatingSSLSocketFactory.getDefaultFactory().getProviderName();
            }
            catch (IOException e) {
                LOG.trace("NonCritFailure: DelegatingSSLSocketFactory Init failed : {}", (Object)e.getMessage());
            }
        }
        if (abfsConfiguration.getPreferredHttpOperationType() == HttpOperationType.APACHE_HTTP_CLIENT) {
            this.keepAliveCache = new KeepAliveCache(abfsConfiguration);
            this.abfsApacheHttpClient = new AbfsApacheHttpClient(DelegatingSSLSocketFactory.getDefaultFactory(), abfsConfiguration.getHttpReadTimeout(), this.keepAliveCache);
        }
        this.userAgent = this.initializeUserAgent(abfsConfiguration, sslProviderName);
        this.abfsPerfTracker = abfsClientContext.getAbfsPerfTracker();
        this.abfsCounters = abfsClientContext.getAbfsCounters();
        ThreadFactory tf = new ThreadFactoryBuilder().setNameFormat("AbfsClient Lease Ops").setDaemon(true).build();
        this.executorService = MoreExecutors.listeningDecorator((ScheduledExecutorService)HadoopExecutors.newScheduledThreadPool((int)this.abfsConfiguration.getNumLeaseThreads(), (ThreadFactory)tf));
    }

    public AbfsClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, AccessTokenProvider tokenProvider, EncryptionContextProvider encryptionContextProvider, AbfsClientContext abfsClientContext) throws IOException {
        this(baseUrl, sharedKeyCredentials, abfsConfiguration, encryptionContextProvider, abfsClientContext);
        this.tokenProvider = tokenProvider;
    }

    public AbfsClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, SASTokenProvider sasTokenProvider, EncryptionContextProvider encryptionContextProvider, AbfsClientContext abfsClientContext) throws IOException {
        this(baseUrl, sharedKeyCredentials, abfsConfiguration, encryptionContextProvider, abfsClientContext);
        this.sasTokenProvider = sasTokenProvider;
    }

    @Override
    public void close() throws IOException {
        if (this.keepAliveCache != null) {
            this.keepAliveCache.close();
        }
        if (this.abfsApacheHttpClient != null) {
            this.abfsApacheHttpClient.close();
        }
        if (this.tokenProvider instanceof Closeable) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{(Closeable)((Object)this.tokenProvider)});
        }
        HadoopExecutors.shutdown((ExecutorService)this.executorService, (Logger)LOG, (long)0L, (TimeUnit)TimeUnit.SECONDS);
    }

    public String getFileSystem() {
        return this.filesystem;
    }

    protected AbfsPerfTracker getAbfsPerfTracker() {
        return this.abfsPerfTracker;
    }

    ExponentialRetryPolicy getExponentialRetryPolicy() {
        return this.exponentialRetryPolicy;
    }

    StaticRetryPolicy getStaticRetryPolicy() {
        return this.staticRetryPolicy;
    }

    public AbfsRetryPolicy getRetryPolicy(String failureReason) {
        return "CT".equals(failureReason) && this.getAbfsConfiguration().getStaticRetryForConnectionTimeoutEnabled() ? this.getStaticRetryPolicy() : this.getExponentialRetryPolicy();
    }

    SharedKeyCredentials getSharedKeyCredentials() {
        return this.sharedKeyCredentials;
    }

    public void setEncryptionType(EncryptionType encryptionType) {
        this.encryptionType = encryptionType;
    }

    public EncryptionType getEncryptionType() {
        return this.encryptionType;
    }

    AbfsThrottlingIntercept getIntercept() {
        return this.intercept;
    }

    @VisibleForTesting
    protected List<AbfsHttpHeader> createDefaultHeaders() {
        return this.createDefaultHeaders(this.xMsVersion);
    }

    private List<AbfsHttpHeader> createDefaultHeaders(AbfsHttpConstants.ApiVersion xMsVersion) {
        ArrayList<AbfsHttpHeader> requestHeaders = new ArrayList<AbfsHttpHeader>();
        requestHeaders.add(new AbfsHttpHeader("x-ms-version", xMsVersion.toString()));
        requestHeaders.add(new AbfsHttpHeader("Accept", "application/json, application/octet-stream"));
        requestHeaders.add(new AbfsHttpHeader("Accept-Charset", "utf-8"));
        requestHeaders.add(new AbfsHttpHeader("Content-Type", ""));
        requestHeaders.add(new AbfsHttpHeader("User-Agent", this.userAgent));
        return requestHeaders;
    }

    private void addEncryptionKeyRequestHeaders(String path, List<AbfsHttpHeader> requestHeaders, boolean isCreateFileRequest, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        String encodedKeySHA256;
        String encodedKey;
        switch (this.encryptionType) {
            case GLOBAL_KEY: {
                encodedKey = this.clientProvidedEncryptionKey;
                encodedKeySHA256 = this.clientProvidedEncryptionKeySHA;
                break;
            }
            case ENCRYPTION_CONTEXT: {
                if (isCreateFileRequest) {
                    requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-context", contextEncryptionAdapter.getEncodedContext()));
                }
                encodedKey = contextEncryptionAdapter.getEncodedKey();
                encodedKeySHA256 = contextEncryptionAdapter.getEncodedKeySHA();
                break;
            }
            default: {
                return;
            }
        }
        requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-key", encodedKey));
        requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-key-sha256", encodedKeySHA256));
        requestHeaders.add(new AbfsHttpHeader("x-ms-encryption-algorithm", "AES256"));
    }

    AbfsUriQueryBuilder createDefaultUriQueryBuilder() {
        AbfsUriQueryBuilder abfsUriQueryBuilder = new AbfsUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("timeout", "90");
        return abfsUriQueryBuilder;
    }

    public AbfsRestOperation createFilesystem(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = new AbfsUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.CreateFileSystem, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setFilesystemProperties(String properties, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-properties", properties));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetFileSystemProperties, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation listPath(String relativePath, boolean recursive, int listMaxResults, String continuation, TracingContext tracingContext) throws IOException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        abfsUriQueryBuilder.addQuery("directory", AbfsClient.getDirectoryQueryParameter(relativePath));
        abfsUriQueryBuilder.addQuery("recursive", String.valueOf(recursive));
        abfsUriQueryBuilder.addQuery("continuation", continuation);
        abfsUriQueryBuilder.addQuery("maxResults", String.valueOf(listMaxResults));
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(this.abfsConfiguration.isUpnUsed()));
        this.appendSASTokenToQuery(relativePath, "list", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.ListPaths, "GET", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation getFilesystemProperties(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetFileSystemProperties, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation deleteFilesystem(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", "filesystem");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.DeleteFileSystem, "DELETE", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation createPath(String path, boolean isFile, boolean overwrite, AzureBlobFileSystemStore.Permissions permissions, boolean isAppendBlob, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        if (isFile) {
            this.addEncryptionKeyRequestHeaders(path, requestHeaders, true, contextEncryptionAdapter, tracingContext);
        }
        if (!overwrite) {
            requestHeaders.add(new AbfsHttpHeader("If-None-Match", "*"));
        }
        if (permissions.hasPermission().booleanValue()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-permissions", permissions.getPermission()));
        }
        if (permissions.hasUmask().booleanValue()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-umask", permissions.getUmask()));
        }
        if (eTag != null && !eTag.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("resource", isFile ? "file" : "directory");
        if (isAppendBlob) {
            abfsUriQueryBuilder.addQuery("blobtype", "appendblob");
        }
        String operation = isFile ? "create-file" : "create-directory";
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.CreatePath, "PUT", url, requestHeaders);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            String existingResource;
            if (!op.hasResult()) {
                throw ex;
            }
            if (!isFile && op.getResult().getStatusCode() == 409 && (existingResource = op.getResult().getResponseHeader("x-ms-existing-resource-type")) != null && existingResource.equals("directory")) {
                return op;
            }
            throw ex;
        }
        return op;
    }

    public AbfsRestOperation acquireLease(String path, int duration, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "acquire"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-duration", Integer.toString(duration)));
        requestHeaders.add(new AbfsHttpHeader("x-ms-proposed-lease-id", UUID.randomUUID().toString()));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeasePath, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation renewLease(String path, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "renew"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeasePath, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation releaseLease(String path, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "release"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeasePath, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation breakLease(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "break"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-break-period", "0"));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeasePath, "POST", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsClientRenameResult renamePath(String source, String destination, String continuation, TracingContext tracingContext, String sourceEtag, boolean isMetadataIncompleteState, boolean isNamespaceEnabled) throws IOException {
        boolean shouldAttemptRecovery;
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        boolean hasEtag = !StringUtils.isEmpty((CharSequence)sourceEtag);
        boolean bl = shouldAttemptRecovery = this.renameResilience && isNamespaceEnabled;
        if (!hasEtag && shouldAttemptRecovery) {
            try {
                AbfsRestOperation srcStatusOp = this.getPathStatus(source, false, tracingContext, null);
                if (srcStatusOp.hasResult()) {
                    AbfsHttpOperation result = srcStatusOp.getResult();
                    sourceEtag = AzureBlobFileSystemStore.extractEtagHeader(result);
                    boolean isDir = this.checkIsDir(result);
                    shouldAttemptRecovery = !isDir;
                    LOG.debug("Retrieved etag of source for rename recovery: {}; isDir={}", (Object)sourceEtag, (Object)isDir);
                }
            }
            catch (AbfsRestOperationException e) {
                throw new AbfsRestOperationException(e.getStatusCode(), AzureServiceErrorCode.SOURCE_PATH_NOT_FOUND.getErrorCode(), e.getMessage(), e);
            }
        }
        String encodedRenameSource = AbfsClient.urlEncode("/" + this.getFileSystem() + source);
        if (this.authType == AuthType.SAS) {
            AbfsUriQueryBuilder srcQueryBuilder = new AbfsUriQueryBuilder();
            this.appendSASTokenToQuery(source, "rename-source", srcQueryBuilder);
            encodedRenameSource = encodedRenameSource + srcQueryBuilder.toString();
        }
        LOG.trace("Rename source queryparam added {}", (Object)encodedRenameSource);
        requestHeaders.add(new AbfsHttpHeader("x-ms-rename-source", encodedRenameSource));
        requestHeaders.add(new AbfsHttpHeader("If-None-Match", "*"));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("continuation", continuation);
        this.appendSASTokenToQuery(destination, "rename-destination", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(destination, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.createRenameRestOperation(url, requestHeaders);
        try {
            this.incrementAbfsRenamePath();
            op.execute(tracingContext);
            return new AbfsClientRenameResult(op, isMetadataIncompleteState, isMetadataIncompleteState);
        }
        catch (AzureBlobFileSystemException e) {
            if (!op.hasResult()) {
                throw e;
            }
            if (op.getResult().getStorageErrorCode().equals(AzureServiceErrorCode.RENAME_DESTINATION_PARENT_PATH_NOT_FOUND.getErrorCode()) && !isMetadataIncompleteState) {
                ABFS_METADATA_INCOMPLETE_RENAME_FAILURE.info("Rename Failure attempting to resolve tracking metadata state and retrying.", new Object[0]);
                shouldAttemptRecovery = true;
                isMetadataIncompleteState = true;
                String sourceEtagAfterFailure = sourceEtag;
                if (StringUtils.isEmpty((CharSequence)sourceEtagAfterFailure)) {
                    AbfsRestOperation sourceStatusOp = this.getPathStatus(source, false, tracingContext, null);
                    isMetadataIncompleteState = true;
                    AbfsHttpOperation sourceStatusResult = sourceStatusOp.getResult();
                    sourceEtagAfterFailure = AzureBlobFileSystemStore.extractEtagHeader(sourceStatusResult);
                }
                this.renamePath(source, destination, continuation, tracingContext, sourceEtagAfterFailure, isMetadataIncompleteState, isNamespaceEnabled);
            }
            isMetadataIncompleteState = false;
            boolean etagCheckSucceeded = false;
            if (shouldAttemptRecovery) {
                etagCheckSucceeded = this.renameIdempotencyCheckOp(source, sourceEtag, op, destination, tracingContext);
            }
            if (!etagCheckSucceeded) {
                throw e;
            }
            return new AbfsClientRenameResult(op, true, isMetadataIncompleteState);
        }
    }

    private boolean checkIsDir(AbfsHttpOperation result) {
        String resourceType = result.getResponseHeader("x-ms-resource-type");
        return resourceType != null && resourceType.equalsIgnoreCase("directory");
    }

    @VisibleForTesting
    AbfsRestOperation createRenameRestOperation(URL url, List<AbfsHttpHeader> requestHeaders) {
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.RenamePath, "PUT", url, requestHeaders);
        return op;
    }

    private void incrementAbfsRenamePath() {
        this.abfsCounters.incrementCounter(AbfsStatistic.RENAME_PATH_ATTEMPTS, 1L);
    }

    public boolean renameIdempotencyCheckOp(String source, String sourceEtag, AbfsRestOperation op, String destination, TracingContext tracingContext) {
        Preconditions.checkArgument((boolean)op.hasResult(), (Object)"Operations has null HTTP response");
        LOG.debug("rename({}, {}) failure {}; retry={} etag {}", new Object[]{source, destination, op.getResult().getStatusCode(), op.isARetriedRequest(), sourceEtag});
        if (!op.isARetriedRequest() || op.getResult().getStatusCode() != 404) {
            return false;
        }
        if (StringUtils.isNotEmpty((CharSequence)sourceEtag)) {
            LOG.info("rename {} to {} failed, checking etag of destination", (Object)source, (Object)destination);
            try {
                AbfsRestOperation destStatusOp = this.getPathStatus(destination, false, tracingContext, null);
                AbfsHttpOperation result = destStatusOp.getResult();
                boolean recovered = result.getStatusCode() == 200 && sourceEtag.equals(AzureBlobFileSystemStore.extractEtagHeader(result));
                LOG.info("File rename has taken place: recovery {}", (Object)(recovered ? "succeeded" : "failed"));
                return recovered;
            }
            catch (AzureBlobFileSystemException ex) {
                LOG.debug("Failed to get status of path {}", (Object)destination, (Object)ex);
            }
        } else {
            LOG.debug("No source etag; unable to probe for the operation's success");
        }
        return false;
    }

    @VisibleForTesting
    boolean isSourceDestEtagEqual(String sourceEtag, AbfsHttpOperation result) {
        return sourceEtag.equals(AzureBlobFileSystemStore.extractEtagHeader(result));
    }

    public AbfsRestOperation append(String path, byte[] buffer, AppendRequestParameters reqParams, String cachedSasToken, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        if (reqParams.isExpectHeaderEnabled()) {
            requestHeaders.add(new AbfsHttpHeader("Expect", "100-continue"));
        }
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        if (reqParams.getLeaseId() != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", reqParams.getLeaseId()));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "append");
        abfsUriQueryBuilder.addQuery("position", Long.toString(reqParams.getPosition()));
        if (reqParams.getMode() == AppendRequestParameters.Mode.FLUSH_MODE || reqParams.getMode() == AppendRequestParameters.Mode.FLUSH_CLOSE_MODE) {
            abfsUriQueryBuilder.addQuery("flush", "true");
            if (reqParams.getMode() == AppendRequestParameters.Mode.FLUSH_CLOSE_MODE) {
                abfsUriQueryBuilder.addQuery("close", "true");
            }
        }
        if (reqParams.isRetryDueToExpect()) {
            String userAgentRetry = this.userAgent;
            userAgentRetry = userAgentRetry.replace(HUNDRED_CONTINUE_USER_AGENT, "");
            requestHeaders.removeIf(header -> header.getName().equalsIgnoreCase("User-Agent"));
            requestHeaders.add(new AbfsHttpHeader("User-Agent", userAgentRetry));
        }
        if (this.isChecksumValidationEnabled()) {
            this.addCheckSumHeaderForWrite(requestHeaders, reqParams, buffer);
        }
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.Append, "PUT", url, requestHeaders, buffer, reqParams.getoffset(), reqParams.getLength(), sasTokenForReuse);
        try {
            op.execute(tracingContext);
        }
        catch (AbfsRestOperationException e) {
            int responseStatusCode = e.getStatusCode();
            if (this.checkUserError(responseStatusCode) && reqParams.isExpectHeaderEnabled()) {
                LOG.debug("User error, retrying without 100 continue enabled for the given path {}", (Object)path);
                reqParams.setExpectHeaderEnabled(false);
                reqParams.setRetryDueToExpect(true);
                return this.append(path, buffer, reqParams, cachedSasToken, contextEncryptionAdapter, tracingContext);
            }
            if (!op.hasResult()) {
                throw e;
            }
            if (this.isMd5ChecksumError(e)) {
                throw new AbfsInvalidChecksumException(e);
            }
            if (reqParams.isAppendBlob() && this.appendSuccessCheckOp(op, path, reqParams.getPosition() + (long)reqParams.getLength(), tracingContext)) {
                AbfsRestOperation successOp = this.getAbfsRestOperation(AbfsRestOperationType.Append, "PUT", url, requestHeaders, buffer, reqParams.getoffset(), reqParams.getLength(), sasTokenForReuse);
                successOp.hardSetResult(200);
                return successOp;
            }
            throw e;
        }
        catch (AzureBlobFileSystemException e) {
            LOG.debug("Append request failed with non server issues for path: {}, offset: {}, position: {}", new Object[]{path, reqParams.getoffset(), reqParams.getPosition()});
            throw e;
        }
        return op;
    }

    private boolean checkUserError(int responseStatusCode) {
        return responseStatusCode >= 400 && responseStatusCode < 500;
    }

    private boolean isMd5ChecksumError(AbfsRestOperationException e) {
        AzureServiceErrorCode storageErrorCode = e.getErrorCode();
        return storageErrorCode == AzureServiceErrorCode.MD5_MISMATCH;
    }

    public boolean appendSuccessCheckOp(AbfsRestOperation op, String path, long length, TracingContext tracingContext) throws AzureBlobFileSystemException {
        String fileLength;
        AbfsRestOperation destStatusOp;
        if (op.isARetriedRequest() && op.getResult().getStatusCode() == 400 && (destStatusOp = this.getPathStatus(path, false, tracingContext, null)).getResult().getStatusCode() == 200 && length <= Long.parseLong(fileLength = destStatusOp.getResult().getResponseHeader("Content-Length"))) {
            LOG.debug("Returning success response from append blob idempotency code");
            return true;
        }
        return false;
    }

    public AbfsRestOperation flush(String path, long position, boolean retainUncommittedData, boolean isClose, String cachedSasToken, String leaseId, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        if (leaseId != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "flush");
        abfsUriQueryBuilder.addQuery("position", Long.toString(position));
        abfsUriQueryBuilder.addQuery("retainUncommittedData", String.valueOf(retainUncommittedData));
        abfsUriQueryBuilder.addQuery("close", String.valueOf(isClose));
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.Flush, "PUT", url, requestHeaders, sasTokenForReuse);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setPathProperties(String path, String properties, TracingContext tracingContext, ContextEncryptionAdapter contextEncryptionAdapter) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-properties", properties));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setProperties");
        this.appendSASTokenToQuery(path, "set-properties", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetPathProperties, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation getPathStatus(String path, boolean includeProperties, TracingContext tracingContext, ContextEncryptionAdapter contextEncryptionAdapter) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String operation = "get-properties";
        if (!includeProperties) {
            abfsUriQueryBuilder.addQuery("action", "getStatus");
            operation = "get-status";
        } else {
            this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        }
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(this.abfsConfiguration.isUpnUsed()));
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetPathStatus, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation read(String path, long position, byte[] buffer, int bufferOffset, int bufferLength, String eTag, String cachedSasToken, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        AbfsHttpHeader rangeHeader = new AbfsHttpHeader("Range", String.format("bytes=%d-%d", position, position + (long)bufferLength - 1L));
        requestHeaders.add(rangeHeader);
        requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        if (this.isChecksumValidationEnabled(requestHeaders, rangeHeader, bufferLength)) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-range-get-content-md5", "true"));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "read", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.ReadFile, "GET", url, requestHeaders, buffer, bufferOffset, bufferLength, sasTokenForReuse);
        op.execute(tracingContext);
        if (this.isChecksumValidationEnabled(requestHeaders, rangeHeader, bufferLength)) {
            this.verifyCheckSumForRead(buffer, op.getResult(), bufferOffset);
        }
        return op;
    }

    public AbfsRestOperation deletePath(String path, boolean recursive, String continuation, TracingContext tracingContext, boolean isNamespaceEnabled) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.isPaginatedDelete(recursive, isNamespaceEnabled) != false && this.xMsVersion.compareTo(AbfsHttpConstants.ApiVersion.AUG_03_2023) < 0 ? this.createDefaultHeaders(AbfsHttpConstants.ApiVersion.AUG_03_2023) : this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        if (this.isPaginatedDelete(recursive, isNamespaceEnabled).booleanValue()) {
            abfsUriQueryBuilder.addQuery("paginated", "true");
        }
        abfsUriQueryBuilder.addQuery("recursive", String.valueOf(recursive));
        abfsUriQueryBuilder.addQuery("continuation", continuation);
        String operation = recursive ? "delete-recursive" : "delete";
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = new AbfsRestOperation(AbfsRestOperationType.DeletePath, this, "DELETE", url, requestHeaders, this.abfsConfiguration);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException e) {
            if (!op.hasResult()) {
                throw e;
            }
            AbfsRestOperation idempotencyOp = this.deleteIdempotencyCheckOp(op);
            if (idempotencyOp.getResult().getStatusCode() == op.getResult().getStatusCode()) {
                throw e;
            }
            return idempotencyOp;
        }
        return op;
    }

    public AbfsRestOperation deleteIdempotencyCheckOp(AbfsRestOperation op) {
        Preconditions.checkArgument((boolean)op.hasResult(), (Object)"Operations has null HTTP response");
        if (op.isARetriedRequest() && op.getResult().getStatusCode() == 404) {
            AbfsRestOperation successOp = this.getAbfsRestOperation(AbfsRestOperationType.DeletePath, "DELETE", op.getUrl(), op.getRequestHeaders());
            successOp.hardSetResult(200);
            LOG.debug("Returning success response from delete idempotency logic");
            return successOp;
        }
        return op;
    }

    public AbfsRestOperation setOwner(String path, String owner, String group, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        if (owner != null && !owner.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-owner", owner));
        }
        if (group != null && !group.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-group", group));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setAccessControl");
        this.appendSASTokenToQuery(path, "set-owner", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetOwner, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setPermission(String path, String permission, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-permissions", permission));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setAccessControl");
        this.appendSASTokenToQuery(path, "set-permission", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetPermissions, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation setAcl(String path, String aclSpecString, TracingContext tracingContext) throws AzureBlobFileSystemException {
        return this.setAcl(path, aclSpecString, "", tracingContext);
    }

    public AbfsRestOperation setAcl(String path, String aclSpecString, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("X-HTTP-Method-Override", "PATCH"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-acl", aclSpecString));
        if (eTag != null && !eTag.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "setAccessControl");
        this.appendSASTokenToQuery(path, "set-acl", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetAcl, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation getAclStatus(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        return this.getAclStatus(path, this.abfsConfiguration.isUpnUsed(), tracingContext);
    }

    public AbfsRestOperation getAclStatus(String path, boolean useUPN, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "getAccessControl");
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(useUPN));
        this.appendSASTokenToQuery(path, "get-acl", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetAcl, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation checkAccess(String path, String rwx, TracingContext tracingContext) throws AzureBlobFileSystemException {
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("action", "checkAccess");
        abfsUriQueryBuilder.addQuery("fsAction", rwx);
        this.appendSASTokenToQuery(path, "check-access", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.CheckAccess, "HEAD", url, this.createDefaultHeaders());
        op.execute(tracingContext);
        return op;
    }

    public static String getDirectoryQueryParameter(String path) {
        String directory = path;
        if (Strings.isNullOrEmpty((String)directory)) {
            directory = "";
        } else if (directory.charAt(0) == '/') {
            directory = directory.substring(1);
        }
        return directory;
    }

    private String appendSASTokenToQuery(String path, String operation, AbfsUriQueryBuilder queryBuilder) throws SASTokenProviderException {
        return this.appendSASTokenToQuery(path, operation, queryBuilder, null);
    }

    private String appendSASTokenToQuery(String path, String operation, AbfsUriQueryBuilder queryBuilder, String cachedSasToken) throws SASTokenProviderException {
        String sasToken = null;
        if (this.authType == AuthType.SAS) {
            try {
                LOG.trace("Fetch SAS token for {} on {}", (Object)operation, (Object)path);
                if (cachedSasToken == null) {
                    sasToken = this.sasTokenProvider.getSASToken(this.accountName, this.filesystem, path, operation);
                    if (sasToken == null || sasToken.isEmpty()) {
                        throw new UnsupportedOperationException("SASToken received is empty or null");
                    }
                } else {
                    sasToken = cachedSasToken;
                    LOG.trace("Using cached SAS token.");
                }
                if (sasToken.charAt(0) == '?') {
                    sasToken = sasToken.substring(1);
                }
                queryBuilder.setSASToken(sasToken);
                LOG.trace("SAS token fetch complete for {} on {}", (Object)operation, (Object)path);
            }
            catch (Exception ex) {
                throw new SASTokenProviderException(String.format("Failed to acquire a SAS token for %s on %s due to %s", operation, path, ex.toString()));
            }
        }
        return sasToken;
    }

    private URL createRequestUrl(String query) throws AzureBlobFileSystemException {
        return this.createRequestUrl("", query);
    }

    @VisibleForTesting
    protected URL createRequestUrl(String path, String query) throws AzureBlobFileSystemException {
        URL url;
        String base = this.baseUrl.toString();
        String encodedPath = path;
        try {
            encodedPath = AbfsClient.urlEncode(path);
        }
        catch (AzureBlobFileSystemException ex) {
            LOG.debug("Unexpected error.", (Throwable)ex);
            throw new InvalidUriException(path);
        }
        StringBuilder sb = new StringBuilder();
        sb.append(base);
        sb.append(encodedPath);
        sb.append(query);
        try {
            url = new URL(sb.toString());
        }
        catch (MalformedURLException ex) {
            throw new InvalidUriException(sb.toString());
        }
        return url;
    }

    public static String urlEncode(String value) throws AzureBlobFileSystemException {
        String encodedString;
        try {
            encodedString = URLEncoder.encode(value, "utf-8").replace("+", "%20").replace("%2F", "/");
        }
        catch (UnsupportedEncodingException ex) {
            throw new InvalidUriException(value);
        }
        return encodedString;
    }

    public synchronized String getAccessToken() throws IOException {
        if (this.tokenProvider != null) {
            return "Bearer " + this.tokenProvider.getToken().getAccessToken();
        }
        return null;
    }

    protected Boolean getIsPaginatedDeleteEnabled() {
        return this.abfsConfiguration.isPaginatedDeleteEnabled();
    }

    private Boolean isPaginatedDelete(boolean isRecursiveDelete, boolean isNamespaceEnabled) {
        return this.getIsPaginatedDeleteEnabled() != false && isNamespaceEnabled && isRecursiveDelete;
    }

    public AuthType getAuthType() {
        return this.authType;
    }

    public EncryptionContextProvider getEncryptionContextProvider() {
        return this.encryptionContextProvider;
    }

    @VisibleForTesting
    String initializeUserAgent(AbfsConfiguration abfsConfiguration, String sslProviderName) {
        StringBuilder sb = new StringBuilder();
        sb.append("APN/1.0");
        sb.append(" ");
        sb.append(AbfsHttpConstants.CLIENT_VERSION);
        sb.append(" ");
        sb.append("(");
        sb.append(System.getProperty("java.vendor").replaceAll(" ", ""));
        sb.append(" ");
        sb.append("JavaJRE");
        sb.append(" ");
        sb.append(System.getProperty("java.version"));
        sb.append(";");
        sb.append(" ");
        sb.append(System.getProperty("os.name").replaceAll(" ", ""));
        sb.append(" ");
        sb.append(System.getProperty("os.version"));
        sb.append("/");
        sb.append(System.getProperty("os.arch"));
        sb.append(";");
        this.appendIfNotEmpty(sb, sslProviderName, true);
        this.appendIfNotEmpty(sb, ExtensionHelper.getUserAgentSuffix(this.tokenProvider, ""), true);
        if (abfsConfiguration.isExpectHeaderEnabled()) {
            sb.append(" ");
            sb.append("100-continue");
            sb.append(";");
        }
        sb.append(" ").append((Object)abfsConfiguration.getPreferredHttpOperationType()).append(";");
        sb.append(" ");
        sb.append(abfsConfiguration.getClusterName());
        sb.append("/");
        sb.append(abfsConfiguration.getClusterType());
        sb.append(")");
        this.appendIfNotEmpty(sb, abfsConfiguration.getCustomUserAgentPrefix(), false);
        return String.format(Locale.ROOT, sb.toString(), new Object[0]);
    }

    private void appendIfNotEmpty(StringBuilder sb, String regEx, boolean shouldAppendSemiColon) {
        if (regEx == null || regEx.trim().isEmpty()) {
            return;
        }
        sb.append(" ");
        sb.append(regEx);
        if (shouldAppendSemiColon) {
            sb.append(";");
        }
    }

    private void addCheckSumHeaderForWrite(List<AbfsHttpHeader> requestHeaders, AppendRequestParameters reqParams, byte[] buffer) throws AbfsRestOperationException {
        String md5Hash = this.computeMD5Hash(buffer, reqParams.getoffset(), reqParams.getLength());
        requestHeaders.add(new AbfsHttpHeader("Content-MD5", md5Hash));
    }

    private void verifyCheckSumForRead(byte[] buffer, AbfsHttpOperation result, int bufferOffset) throws AbfsRestOperationException {
        String md5HashActual;
        int numberOfBytesRead = (int)result.getBytesReceived();
        if (numberOfBytesRead == 0) {
            return;
        }
        String md5HashComputed = this.computeMD5Hash(buffer, bufferOffset, numberOfBytesRead);
        if (!md5HashComputed.equals(md5HashActual = result.getResponseHeader("Content-MD5"))) {
            LOG.debug("Md5 Mismatch Error in Read Operation. Server returned Md5: {}, Client computed Md5: {}", (Object)md5HashActual, (Object)md5HashComputed);
            throw new AbfsInvalidChecksumException(result.getRequestId());
        }
    }

    private boolean isChecksumValidationEnabled(List<AbfsHttpHeader> requestHeaders, AbfsHttpHeader rangeHeader, int bufferLength) {
        return this.getAbfsConfiguration().getIsChecksumValidationEnabled() && requestHeaders.contains(rangeHeader) && bufferLength <= 0x400000;
    }

    private boolean isChecksumValidationEnabled() {
        return this.getAbfsConfiguration().getIsChecksumValidationEnabled();
    }

    @VisibleForTesting
    public String computeMD5Hash(byte[] data, int off, int len) throws AbfsRestOperationException {
        try {
            MessageDigest md5Digest = MessageDigest.getInstance("MD5");
            md5Digest.update(data, off, len);
            byte[] md5Bytes = md5Digest.digest();
            return Base64.getEncoder().encodeToString(md5Bytes);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new AbfsDriverException(ex);
        }
    }

    @VisibleForTesting
    URL getBaseUrl() {
        return this.baseUrl;
    }

    @VisibleForTesting
    public SASTokenProvider getSasTokenProvider() {
        return this.sasTokenProvider;
    }

    @VisibleForTesting
    void setEncryptionContextProvider(EncryptionContextProvider provider) {
        this.encryptionContextProvider = provider;
    }

    protected AbfsCounters getAbfsCounters() {
        return this.abfsCounters;
    }

    public AbfsHttpConstants.ApiVersion getxMsVersion() {
        return this.xMsVersion;
    }

    protected AbfsConfiguration getAbfsConfiguration() {
        return this.abfsConfiguration;
    }

    public int getNumLeaseThreads() {
        return this.abfsConfiguration.getNumLeaseThreads();
    }

    public <V> ListenableScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit timeUnit) {
        return this.executorService.schedule(callable, delay, timeUnit);
    }

    public ListenableFuture<?> submit(Runnable runnable) {
        return this.executorService.submit(runnable);
    }

    public <V> void addCallback(ListenableFuture<V> future, FutureCallback<V> callback) {
        Futures.addCallback(future, callback, (Executor)this.executorService);
    }

    @VisibleForTesting
    protected AccessTokenProvider getTokenProvider() {
        return this.tokenProvider;
    }

    AbfsRestOperation getAbfsRestOperation(AbfsRestOperationType operationType, String httpMethod, URL url, List<AbfsHttpHeader> requestHeaders, byte[] buffer, int bufferOffset, int bufferLength, String sasTokenForReuse) {
        return new AbfsRestOperation(operationType, this, httpMethod, url, requestHeaders, buffer, bufferOffset, bufferLength, sasTokenForReuse, this.abfsConfiguration);
    }

    AbfsRestOperation getAbfsRestOperation(AbfsRestOperationType operationType, String httpMethod, URL url, List<AbfsHttpHeader> requestHeaders) {
        return new AbfsRestOperation(operationType, this, httpMethod, url, requestHeaders, this.abfsConfiguration);
    }

    AbfsRestOperation getAbfsRestOperation(AbfsRestOperationType operationType, String httpMethod, URL url, List<AbfsHttpHeader> requestHeaders, String sasTokenForReuse) {
        return new AbfsRestOperation(operationType, this, httpMethod, url, requestHeaders, sasTokenForReuse, this.abfsConfiguration);
    }

    @VisibleForTesting
    AbfsApacheHttpClient getAbfsApacheHttpClient() {
        return this.abfsApacheHttpClient;
    }

    @VisibleForTesting
    KeepAliveCache getKeepAliveCache() {
        return this.keepAliveCache;
    }
}

