/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInfo;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.shaded.org.apache.commons.collections.CollectionUtils;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.collect.ArrayListMultimap;
import org.apache.hadoop.thirdparty.com.google.common.collect.ListMultimap;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DirectoryScanner
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(DirectoryScanner.class);
    private static final int DEFAULT_MAP_SIZE = 32768;
    private static final int RECONCILE_BLOCKS_BATCH_SIZE = 1000;
    private final FsDatasetSpi<?> dataset;
    private final ExecutorService reportCompileThreadPool;
    private final ScheduledExecutorService masterThread;
    private final long scanPeriodMsecs;
    private final long throttleLimitMsPerSec;
    private final AtomicBoolean shouldRun = new AtomicBoolean();
    private boolean retainDiffs = false;
    @VisibleForTesting
    final AtomicLong timeRunningMs = new AtomicLong(0L);
    @VisibleForTesting
    final AtomicLong timeWaitingMs = new AtomicLong(0L);
    @VisibleForTesting
    final BlockPoolReport diffs = new BlockPoolReport();
    @VisibleForTesting
    final Map<String, Stats> stats;

    @VisibleForTesting
    public void setRetainDiffs(boolean b) {
        this.retainDiffs = b;
    }

    public DirectoryScanner(FsDatasetSpi<?> dataset, Configuration conf) {
        this.dataset = dataset;
        this.stats = new HashMap<String, Stats>(32768);
        int interval = (int)conf.getTimeDuration("dfs.datanode.directoryscan.interval", 21600L, TimeUnit.SECONDS);
        this.scanPeriodMsecs = TimeUnit.SECONDS.toMillis(interval);
        int throttle = conf.getInt("dfs.datanode.directoryscan.throttle.limit.ms.per.sec", -1);
        if ((long)throttle >= TimeUnit.SECONDS.toMillis(1L)) {
            LOG.warn("{} set to value above 1000 ms/sec. Assuming default value of {}", (Object)"dfs.datanode.directoryscan.throttle.limit.ms.per.sec", (Object)-1);
            throttle = -1;
        }
        this.throttleLimitMsPerSec = throttle;
        int threads = conf.getInt("dfs.datanode.directoryscan.threads", 1);
        this.reportCompileThreadPool = Executors.newFixedThreadPool(threads, (ThreadFactory)new Daemon.DaemonFactory());
        this.masterThread = new ScheduledThreadPoolExecutor(1, (ThreadFactory)new Daemon.DaemonFactory());
    }

    void start() {
        this.shouldRun.set(true);
        long firstScanTime = ThreadLocalRandom.current().nextLong(this.scanPeriodMsecs);
        LOG.info("Periodic Directory Tree Verification scan starting in {}ms with interval of {}ms and throttle limit of {}ms/s", new Object[]{firstScanTime, this.scanPeriodMsecs, this.throttleLimitMsPerSec});
        this.masterThread.scheduleAtFixedRate(this, firstScanTime, this.scanPeriodMsecs, TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    boolean getRunStatus() {
        return this.shouldRun.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void clear() {
        BlockPoolReport blockPoolReport = this.diffs;
        synchronized (blockPoolReport) {
            this.diffs.clear();
        }
        this.stats.clear();
    }

    @Override
    public void run() {
        if (!this.shouldRun.get()) {
            LOG.warn("This cycle terminating immediately because 'shouldRun' has been deactivated");
            return;
        }
        try {
            this.reconcile();
        }
        catch (Exception e) {
            LOG.error("Exception during DirectoryScanner execution - will continue next cycle", (Throwable)e);
        }
        catch (Error er) {
            LOG.error("System Error during DirectoryScanner execution - permanently terminating periodic scanner", (Throwable)er);
            throw er;
        }
    }

    void shutdown() {
        LOG.info("Shutdown has been called");
        if (!this.shouldRun.getAndSet(false)) {
            LOG.warn("Shutdown has been called, but periodic scanner not started");
        }
        if (this.masterThread != null) {
            this.masterThread.shutdown();
        }
        if (this.reportCompileThreadPool != null) {
            this.reportCompileThreadPool.shutdownNow();
        }
        if (this.masterThread != null) {
            try {
                this.masterThread.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOG.error("interrupted while waiting for masterThread to terminate", (Throwable)e);
            }
        }
        if (this.reportCompileThreadPool != null) {
            try {
                this.reportCompileThreadPool.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOG.error("interrupted while waiting for reportCompileThreadPool to terminate", (Throwable)e);
            }
        }
        if (!this.retainDiffs) {
            this.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void reconcile() throws IOException {
        LOG.debug("reconcile start DirectoryScanning");
        this.scan();
        int loopCount = 0;
        BlockPoolReport blockPoolReport = this.diffs;
        synchronized (blockPoolReport) {
            for (Map.Entry<String, FsVolumeSpi.ScanInfo> entry : this.diffs.getEntries()) {
                this.dataset.checkAndUpdate(entry.getKey(), entry.getValue());
                if (loopCount % 1000 == 0) {
                    try {
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                ++loopCount;
            }
        }
        if (!this.retainDiffs) {
            this.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scan() {
        BlockPoolReport blockPoolReport = new BlockPoolReport();
        this.clear();
        Collection<ScanInfoVolumeReport> volumeReports = this.getVolumeReports();
        for (ScanInfoVolumeReport volumeReport : volumeReports) {
            for (String blockPoolId : volumeReport.getBlockPoolIds()) {
                List<FsVolumeSpi.ScanInfo> scanInfos = volumeReport.getScanInfo(blockPoolId);
                blockPoolReport.addAll(blockPoolId, scanInfos);
            }
        }
        blockPoolReport.sortBlocks();
        for (String bpid : blockPoolReport.getBlockPoolIds()) {
            List<FsVolumeSpi.ScanInfo> blockpoolReport = blockPoolReport.getScanInfo(bpid);
            Stats statsRecord = new Stats(bpid);
            this.stats.put(bpid, statsRecord);
            ArrayList<FsVolumeSpi.ScanInfo> diffRecord = new ArrayList<FsVolumeSpi.ScanInfo>();
            statsRecord.totalBlocks = blockpoolReport.size();
            List<ReplicaInfo> bl = this.dataset.getFinalizedBlocks(bpid);
            Collections.sort(bl);
            int d = 0;
            int m = 0;
            while (m < bl.size() && d < blockpoolReport.size()) {
                ReplicaInfo memBlock = bl.get(m);
                FsVolumeSpi.ScanInfo info = blockpoolReport.get(d);
                if (info.getBlockId() < memBlock.getBlockId()) {
                    if (!this.dataset.isDeletingBlock(bpid, info.getBlockId())) {
                        ++statsRecord.missingMemoryBlocks;
                        this.addDifference(diffRecord, statsRecord, info);
                    }
                    ++d;
                    continue;
                }
                if (info.getBlockId() > memBlock.getBlockId()) {
                    this.addDifference(diffRecord, statsRecord, memBlock.getBlockId(), info.getVolume());
                    ++m;
                    continue;
                }
                if (info.getBlockFile() == null) {
                    this.addDifference(diffRecord, statsRecord, info);
                } else if (info.getGenStamp() != memBlock.getGenerationStamp() || info.getBlockLength() != memBlock.getNumBytes()) {
                    ++statsRecord.mismatchBlocks;
                    this.addDifference(diffRecord, statsRecord, info);
                } else if (memBlock.compareWith(info) != 0) {
                    ++statsRecord.duplicateBlocks;
                    this.addDifference(diffRecord, statsRecord, info);
                }
                if (++d < blockpoolReport.size()) {
                    FsVolumeSpi.ScanInfo nextInfo = blockpoolReport.get(d);
                    if (nextInfo.getBlockId() == info.getBlockId()) continue;
                    ++m;
                    continue;
                }
                ++m;
            }
            while (m < bl.size()) {
                ReplicaInfo current = bl.get(m++);
                this.addDifference(diffRecord, statsRecord, current.getBlockId(), current.getVolume());
            }
            while (d < blockpoolReport.size()) {
                if (!this.dataset.isDeletingBlock(bpid, blockpoolReport.get(d).getBlockId())) {
                    ++statsRecord.missingMemoryBlocks;
                    this.addDifference(diffRecord, statsRecord, blockpoolReport.get(d));
                }
                ++d;
            }
            BlockPoolReport blockPoolReport2 = this.diffs;
            synchronized (blockPoolReport2) {
                this.diffs.addAll(bpid, diffRecord);
            }
            LOG.info("Scan Results: {}", (Object)statsRecord);
        }
    }

    private void addDifference(Collection<FsVolumeSpi.ScanInfo> diffRecord, Stats statsRecord, FsVolumeSpi.ScanInfo info) {
        statsRecord.missingMetaFile = statsRecord.missingMetaFile + (info.getMetaFile() == null ? 1L : 0L);
        statsRecord.missingBlockFile = statsRecord.missingBlockFile + (info.getBlockFile() == null ? 1L : 0L);
        diffRecord.add(info);
    }

    private void addDifference(Collection<FsVolumeSpi.ScanInfo> diffRecord, Stats statsRecord, long blockId, FsVolumeSpi vol) {
        ++statsRecord.missingBlockFile;
        ++statsRecord.missingMetaFile;
        diffRecord.add(new FsVolumeSpi.ScanInfo(blockId, null, null, null, vol));
    }

    @VisibleForTesting
    public Collection<ScanInfoVolumeReport> getVolumeReports() {
        ArrayList<ScanInfoVolumeReport> volReports = new ArrayList<ScanInfoVolumeReport>();
        ArrayList<Future<ScanInfoVolumeReport>> compilersInProgress = new ArrayList<Future<ScanInfoVolumeReport>>();
        try (FsDatasetSpi.FsVolumeReferences volumes = this.dataset.getFsVolumeReferences();){
            for (FsVolumeSpi fsVolumeSpi : volumes) {
                if (fsVolumeSpi.getStorageType() == StorageType.PROVIDED) continue;
                ReportCompiler reportCompiler = new ReportCompiler(fsVolumeSpi);
                Future<ScanInfoVolumeReport> result = this.reportCompileThreadPool.submit(reportCompiler);
                compilersInProgress.add(result);
            }
            for (Future future : compilersInProgress) {
                try {
                    ScanInfoVolumeReport result = (ScanInfoVolumeReport)future.get();
                    if (CollectionUtils.addIgnoreNull(volReports, (Object)result)) continue;
                    volReports.clear();
                    break;
                }
                catch (Exception ex) {
                    LOG.warn("Error compiling report. Continuing.", (Throwable)ex);
                }
            }
        }
        catch (IOException e) {
            LOG.error("Unexpected IOException by closing FsVolumeReference", (Throwable)e);
        }
        return volReports;
    }

    public class ReportCompiler
    implements Callable<ScanInfoVolumeReport> {
        private final FsVolumeSpi volume;
        private final StopWatch throttleTimer = new StopWatch();
        private final StopWatch perfTimer = new StopWatch();

        public ReportCompiler(FsVolumeSpi volume) {
            this.volume = volume;
        }

        @Override
        public ScanInfoVolumeReport call() throws IOException {
            String[] bpList = this.volume.getBlockPoolList();
            ScanInfoVolumeReport result = new ScanInfoVolumeReport(this.volume, Arrays.asList(bpList));
            this.perfTimer.start();
            this.throttleTimer.start();
            for (String bpid : bpList) {
                ArrayList<FsVolumeSpi.ScanInfo> report = new ArrayList<FsVolumeSpi.ScanInfo>(32768);
                this.perfTimer.reset().start();
                this.throttleTimer.reset().start();
                try {
                    this.volume.compileReport(bpid, report, this);
                    result.addAll(bpid, report);
                }
                catch (InterruptedException ex) {
                    result = null;
                    break;
                }
            }
            LOG.trace("Scanner volume report: {}", (Object)result);
            return result;
        }

        public void throttle() throws InterruptedException {
            this.accumulateTimeRunning();
            if (DirectoryScanner.this.throttleLimitMsPerSec > 0L) {
                long runningTime = this.throttleTimer.now(TimeUnit.MILLISECONDS);
                if (runningTime >= DirectoryScanner.this.throttleLimitMsPerSec) {
                    long sleepTime;
                    if (runningTime >= 1000L) {
                        LOG.warn("Unable to throttle within the second. Blocking for 1s.");
                        sleepTime = 1000L;
                    } else {
                        long overTime = runningTime - DirectoryScanner.this.throttleLimitMsPerSec;
                        sleepTime = 1000L - DirectoryScanner.this.throttleLimitMsPerSec + overTime;
                    }
                    Thread.sleep(sleepTime);
                    this.throttleTimer.reset().start();
                }
                this.accumulateTimeWaiting();
            }
        }

        private void accumulateTimeRunning() {
            DirectoryScanner.this.timeRunningMs.getAndAdd(this.perfTimer.now(TimeUnit.MILLISECONDS));
            this.perfTimer.reset().start();
        }

        private void accumulateTimeWaiting() {
            DirectoryScanner.this.timeWaitingMs.getAndAdd(this.perfTimer.now(TimeUnit.MILLISECONDS));
            this.perfTimer.reset().start();
        }
    }

    @VisibleForTesting
    public static class BlockPoolReport {
        private static final long serialVersionUID = 1L;
        private final Set<String> blockPools;
        private final ListMultimap<String, FsVolumeSpi.ScanInfo> map;

        BlockPoolReport() {
            this.blockPools = new HashSet<String>(2);
            this.map = ArrayListMultimap.create((int)2, (int)32768);
        }

        BlockPoolReport(Collection<String> blockPools) {
            this.blockPools = new HashSet<String>(blockPools);
            this.map = ArrayListMultimap.create((int)blockPools.size(), (int)32768);
        }

        public void addAll(String bpid, Collection<FsVolumeSpi.ScanInfo> scanInfos) {
            this.blockPools.add(bpid);
            this.map.putAll((Object)bpid, scanInfos);
        }

        public void sortBlocks() {
            for (String bpid : this.map.keySet()) {
                List list = this.map.get((Object)bpid);
                Collections.sort(list);
            }
        }

        public Set<String> getBlockPoolIds() {
            return Collections.unmodifiableSet(this.blockPools);
        }

        public List<FsVolumeSpi.ScanInfo> getScanInfo(String bpid) {
            return this.map.get((Object)bpid);
        }

        public Collection<Map.Entry<String, FsVolumeSpi.ScanInfo>> getEntries() {
            return Collections.unmodifiableCollection(this.map.entries());
        }

        public void clear() {
            this.map.clear();
            this.blockPools.clear();
        }

        public String toString() {
            return "BlockPoolReport [blockPools=" + this.blockPools + ", map=" + this.map + "]";
        }
    }

    @VisibleForTesting
    public static class ScanInfoVolumeReport {
        private static final long serialVersionUID = 1L;
        private final FsVolumeSpi volume;
        private final BlockPoolReport blockPoolReport;

        ScanInfoVolumeReport(FsVolumeSpi volume) {
            this.volume = volume;
            this.blockPoolReport = new BlockPoolReport();
        }

        ScanInfoVolumeReport(FsVolumeSpi volume, Collection<String> blockPools) {
            this.volume = volume;
            this.blockPoolReport = new BlockPoolReport(blockPools);
        }

        public void addAll(String bpid, Collection<FsVolumeSpi.ScanInfo> scanInfos) {
            this.blockPoolReport.addAll(bpid, scanInfos);
        }

        public Set<String> getBlockPoolIds() {
            return this.blockPoolReport.getBlockPoolIds();
        }

        public List<FsVolumeSpi.ScanInfo> getScanInfo(String bpid) {
            return this.blockPoolReport.getScanInfo(bpid);
        }

        public FsVolumeSpi getVolume() {
            return this.volume;
        }

        public String toString() {
            return "ScanInfoVolumeReport [volume=" + this.volume + ", blockPoolReport=" + this.blockPoolReport + "]";
        }
    }

    @VisibleForTesting
    static class Stats {
        final String bpid;
        long totalBlocks = 0L;
        long missingMetaFile = 0L;
        long missingBlockFile = 0L;
        long missingMemoryBlocks = 0L;
        long mismatchBlocks = 0L;
        long duplicateBlocks = 0L;

        public Stats(String bpid) {
            this.bpid = bpid;
        }

        public String toString() {
            return "BlockPool " + this.bpid + " Total blocks: " + this.totalBlocks + ", missing metadata files: " + this.missingMetaFile + ", missing block files: " + this.missingBlockFile + ", missing blocks in memory: " + this.missingMemoryBlocks + ", mismatched blocks: " + this.mismatchBlocks + ", duplicated blocks: " + this.duplicateBlocks;
        }
    }
}

