/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.QuotaCounts;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotTestHelper;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestQuotaByStorageType {
    private static final int BLOCKSIZE = 1024;
    private static final short REPLICATION = 3;
    private static final long seed = 0L;
    private static final Path dir = new Path("/TestQuotaByStorageType");
    private Configuration conf;
    private MiniDFSCluster cluster;
    private FSDirectory fsdir;
    private DistributedFileSystem dfs;
    private FSNamesystem fsn;
    protected static final Logger LOG = LoggerFactory.getLogger(TestQuotaByStorageType.class);

    @Before
    public void setUp() throws Exception {
        this.conf = new Configuration();
        this.conf.setLong("dfs.blocksize", 1024L);
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(3).storageTypes(new StorageType[]{StorageType.SSD, StorageType.DEFAULT}).build();
        this.cluster.waitActive();
        this.fsdir = this.cluster.getNamesystem().getFSDirectory();
        this.dfs = this.cluster.getFileSystem();
        this.fsn = this.cluster.getNamesystem();
    }

    @After
    public void tearDown() throws Exception {
        if (this.cluster != null) {
            this.cluster.shutdown();
        }
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithFileCreateOneSSD() throws Exception {
        this.testQuotaByStorageTypeWithFileCreateCase("ONE_SSD", StorageType.SSD, (short)1);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithFileCreateAllSSD() throws Exception {
        this.testQuotaByStorageTypeWithFileCreateCase("ALL_SSD", StorageType.SSD, (short)3);
    }

    void testQuotaByStorageTypeWithFileCreateCase(String storagePolicy, StorageType storageType, short replication) throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, storagePolicy);
        this.dfs.setQuotaByStorageType(foo, storageType, 10240L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long file1Len = 2560L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long storageTypeConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)storageType);
        Assert.assertEquals((long)(file1Len * (long)replication), (long)storageTypeConsumed);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithFileCreateAppend() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        this.dfs.setQuotaByStorageType(foo, StorageType.SSD, 4096L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long ssdConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        int appendLen = 2048;
        DFSTestUtil.appendFile((FileSystem)this.dfs, createdFile1, appendLen);
        ssdConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)(file1Len += (long)appendLen), (long)ssdConsumed);
        ContentSummary cs = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)cs.getSpaceConsumed(), (long)(file1Len * 3L));
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.SSD), (long)file1Len);
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.DISK), (long)(file1Len * 2L));
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithFileCreateDelete() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        this.dfs.setQuotaByStorageType(foo, StorageType.SSD, 10240L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long file1Len = 2560L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long storageTypeConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)storageTypeConsumed);
        this.dfs.delete(createdFile1, false);
        storageTypeConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)0L, (long)storageTypeConsumed);
        QuotaCounts counts = new QuotaCounts.Builder().build();
        fnode.computeQuotaUsage(this.fsn.getBlockManager().getStoragePolicySuite(), counts, true);
        Assert.assertEquals((String)fnode.dumpTreeRecursively().toString(), (long)0L, (long)counts.getTypeSpaces().get((Enum)StorageType.SSD));
        ContentSummary cs = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)cs.getSpaceConsumed(), (long)0L);
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.SSD), (long)0L);
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.DISK), (long)0L);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithFileCreateRename() throws Exception {
        Path foo = new Path(dir, "foo");
        this.dfs.mkdirs(foo);
        Path createdFile1foo = new Path(foo, "created_file1.data");
        Path bar = new Path(dir, "bar");
        this.dfs.mkdirs(bar);
        Path createdFile1bar = new Path(bar, "created_file1.data");
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        this.dfs.setStoragePolicy(bar, "ONE_SSD");
        this.dfs.setQuotaByStorageType(foo, StorageType.SSD, 4096L);
        this.dfs.setQuotaByStorageType(bar, StorageType.SSD, 2048L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long file1Len = 3072L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1foo, bufLen, file1Len, 1024L, (short)3, 0L);
        long ssdConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        try {
            this.dfs.rename(createdFile1foo, createdFile1bar);
            Assert.fail((String)"Should have failed with QuotaByStorageTypeExceededException ");
        }
        catch (Throwable t) {
            LOG.info("Got expected exception ", t);
        }
        ContentSummary cs = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)cs.getSpaceConsumed(), (long)(file1Len * 3L));
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.SSD), (long)file1Len);
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.DISK), (long)(file1Len * 2L));
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeExceptionWithFileCreate() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        this.dfs.setQuotaByStorageType(foo, StorageType.SSD, 4096L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long currentSSDConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)currentSSDConsumed);
        Path createdFile2 = new Path(foo, "created_file2.data");
        long file2Len = 1536L;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile2, bufLen, file2Len, 1024L, (short)3, 0L);
        currentSSDConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)(file1Len + file2Len), (long)currentSSDConsumed);
        Path createdFile3 = new Path(foo, "created_file3.data");
        long file3Len = 1024L;
        try {
            DFSTestUtil.createFile((FileSystem)this.dfs, createdFile3, bufLen, file3Len, 1024L, (short)3, 0L);
            Assert.fail((String)"Should have failed with QuotaByStorageTypeExceededException ");
        }
        catch (Throwable t) {
            LOG.info("Got expected exception ", t);
            currentSSDConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
            Assert.assertEquals((long)(file1Len + file2Len), (long)currentSSDConsumed);
        }
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeParentOffChildOff() throws Exception {
        Path parent = new Path(dir, "parent");
        Path child = new Path(parent, "child");
        this.dfs.mkdirs(parent);
        this.dfs.mkdirs(child);
        this.dfs.setStoragePolicy(parent, "ONE_SSD");
        Path createdFile1 = new Path(child, "created_file1.data");
        long file1Len = 2560L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        INode fnode = this.fsdir.getINode4Write("/");
        long ssdConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeParentOffChildOn() throws Exception {
        Path parent = new Path(dir, "parent");
        Path child = new Path(parent, "child");
        this.dfs.mkdirs(parent);
        this.dfs.mkdirs(child);
        this.dfs.setStoragePolicy(parent, "ONE_SSD");
        this.dfs.setQuotaByStorageType(child, StorageType.SSD, 2048L);
        Path createdFile1 = new Path(child, "created_file1.data");
        long file1Len = 2560L;
        int bufLen = 64;
        try {
            DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
            Assert.fail((String)"Should have failed with QuotaByStorageTypeExceededException ");
        }
        catch (Throwable t) {
            LOG.info("Got expected exception ", t);
        }
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeParentOnChildOff() throws Exception {
        short replication = 1;
        Path parent = new Path(dir, "parent");
        Path child = new Path(parent, "child");
        this.dfs.mkdirs(parent);
        this.dfs.mkdirs(child);
        this.dfs.setStoragePolicy(parent, "ONE_SSD");
        this.dfs.setQuotaByStorageType(parent, StorageType.SSD, 3072L);
        Path createdFile1 = new Path(child, "created_file1.data");
        long file1Len = 2560L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, replication, 0L);
        INode fnode = this.fsdir.getINode4Write(parent.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long currentSSDConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)currentSSDConsumed);
        Path createdFile2 = new Path(child, "created_file2.data");
        long file2Len = 1024L;
        try {
            DFSTestUtil.createFile((FileSystem)this.dfs, createdFile2, bufLen, file2Len, 1024L, replication, 0L);
            Assert.fail((String)"Should have failed with QuotaByStorageTypeExceededException ");
        }
        catch (Throwable t) {
            LOG.info("Got expected exception ", t);
            currentSSDConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
            Assert.assertEquals((long)file1Len, (long)currentSSDConsumed);
        }
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeParentOnChildOn() throws Exception {
        Path parent = new Path(dir, "parent");
        Path child = new Path(parent, "child");
        this.dfs.mkdirs(parent);
        this.dfs.mkdirs(child);
        this.dfs.setStoragePolicy(parent, "ONE_SSD");
        this.dfs.setQuotaByStorageType(parent, StorageType.SSD, 2048L);
        this.dfs.setQuotaByStorageType(child, StorageType.SSD, 3072L);
        Path createdFile1 = new Path(child, "created_file1.data");
        long file1Len = 2560L;
        int bufLen = 64;
        try {
            DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
            Assert.fail((String)"Should have failed with QuotaByStorageTypeExceededException ");
        }
        catch (Throwable t) {
            LOG.info("Got expected exception ", t);
        }
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithTraditionalQuota() throws Exception {
        Path foo = new Path(dir, "foo");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        this.dfs.setQuotaByStorageType(foo, StorageType.SSD, 10240L);
        this.dfs.setQuota(foo, 0x7FFFFFFFFFFFFFFEL, 30720L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        Path createdFile = new Path(foo, "created_file.data");
        long fileLen = 2560L;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile, 64, fileLen, 1024L, (short)3, 0L);
        QuotaCounts cnt = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed();
        Assert.assertEquals((long)2L, (long)cnt.getNameSpace());
        Assert.assertEquals((long)(fileLen * 3L), (long)cnt.getStorageSpace());
        this.dfs.delete(createdFile, true);
        QuotaCounts cntAfterDelete = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed();
        Assert.assertEquals((long)1L, (long)cntAfterDelete.getNameSpace());
        Assert.assertEquals((long)0L, (long)cntAfterDelete.getStorageSpace());
        QuotaCounts counts = new QuotaCounts.Builder().build();
        fnode.computeQuotaUsage(this.fsn.getBlockManager().getStoragePolicySuite(), counts, true);
        Assert.assertEquals((String)fnode.dumpTreeRecursively().toString(), (long)1L, (long)counts.getNameSpace());
        Assert.assertEquals((String)fnode.dumpTreeRecursively().toString(), (long)0L, (long)counts.getStorageSpace());
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeAndTraditionalQuotaException1() throws Exception {
        this.testQuotaByStorageTypeOrTraditionalQuotaExceededCase(12L, 4L, 5L, (short)3);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeAndTraditionalQuotaException2() throws Exception {
        this.testQuotaByStorageTypeOrTraditionalQuotaExceededCase(15L, 4L, 5L, (short)3);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeAndTraditionalQuotaException3() throws Exception {
        this.testQuotaByStorageTypeOrTraditionalQuotaExceededCase(12L, 5L, 5L, (short)3);
    }

    private void testQuotaByStorageTypeOrTraditionalQuotaExceededCase(long storageSpaceQuotaInBlocks, long ssdQuotaInBlocks, long testFileLenInBlocks, short replication) throws Exception {
        String METHOD_NAME = GenericTestUtils.getMethodName();
        Path testDir = new Path(dir, METHOD_NAME);
        this.dfs.mkdirs(testDir);
        this.dfs.setStoragePolicy(testDir, "ONE_SSD");
        long ssdQuota = 1024L * ssdQuotaInBlocks;
        long storageSpaceQuota = 1024L * storageSpaceQuotaInBlocks;
        this.dfs.setQuota(testDir, 0x7FFFFFFFFFFFFFFEL, storageSpaceQuota);
        this.dfs.setQuotaByStorageType(testDir, StorageType.SSD, ssdQuota);
        INode testDirNode = this.fsdir.getINode4Write(testDir.toString());
        Assert.assertTrue((boolean)testDirNode.isDirectory());
        Assert.assertTrue((boolean)testDirNode.isQuotaSet());
        Path createdFile = new Path(testDir, "created_file.data");
        long fileLen = testFileLenInBlocks * 1024L;
        try {
            DFSTestUtil.createFile((FileSystem)this.dfs, createdFile, 64, fileLen, 1024L, replication, 0L);
            Assert.fail((String)"Should have failed with DSQuotaExceededException or QuotaByStorageTypeExceededException ");
        }
        catch (Throwable t) {
            LOG.info("Got expected exception ", t);
            long currentSSDConsumed = testDirNode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
            Assert.assertEquals((long)Math.min(ssdQuota, storageSpaceQuota / (long)replication), (long)currentSSDConsumed);
        }
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithSnapshot() throws Exception {
        Path sub1 = new Path(dir, "Sub1");
        this.dfs.mkdirs(sub1);
        this.dfs.setStoragePolicy(sub1, "ONE_SSD");
        this.dfs.setQuotaByStorageType(sub1, StorageType.SSD, 4096L);
        INode sub1Node = this.fsdir.getINode4Write(sub1.toString());
        Assert.assertTrue((boolean)sub1Node.isDirectory());
        Assert.assertTrue((boolean)sub1Node.isQuotaSet());
        Path file1 = new Path(sub1, "file1");
        long file1Len = 2048L;
        DFSTestUtil.createFile((FileSystem)this.dfs, file1, file1Len, (short)3, 0L);
        SnapshotTestHelper.createSnapshot(this.dfs, sub1, "s1");
        long ssdConsumed = sub1Node.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        this.dfs.delete(file1, false);
        ssdConsumed = sub1Node.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        QuotaCounts counts1 = new QuotaCounts.Builder().build();
        sub1Node.computeQuotaUsage(this.fsn.getBlockManager().getStoragePolicySuite(), counts1, true);
        Assert.assertEquals((String)sub1Node.dumpTreeRecursively().toString(), (long)file1Len, (long)counts1.getTypeSpaces().get((Enum)StorageType.SSD));
        ContentSummary cs1 = this.dfs.getContentSummary(sub1);
        Assert.assertEquals((long)cs1.getSpaceConsumed(), (long)(file1Len * 3L));
        Assert.assertEquals((long)cs1.getTypeConsumed(StorageType.SSD), (long)file1Len);
        Assert.assertEquals((long)cs1.getTypeConsumed(StorageType.DISK), (long)(file1Len * 2L));
        this.dfs.deleteSnapshot(sub1, "s1");
        ssdConsumed = sub1Node.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)0L, (long)ssdConsumed);
        QuotaCounts counts2 = new QuotaCounts.Builder().build();
        sub1Node.computeQuotaUsage(this.fsn.getBlockManager().getStoragePolicySuite(), counts2, true);
        Assert.assertEquals((String)sub1Node.dumpTreeRecursively().toString(), (long)0L, (long)counts2.getTypeSpaces().get((Enum)StorageType.SSD));
        ContentSummary cs2 = this.dfs.getContentSummary(sub1);
        Assert.assertEquals((long)cs2.getSpaceConsumed(), (long)0L);
        Assert.assertEquals((long)cs2.getTypeConsumed(StorageType.SSD), (long)0L);
        Assert.assertEquals((long)cs2.getTypeConsumed(StorageType.DISK), (long)0L);
    }

    @Test(timeout=60000L)
    public void testQuotaByStorageTypeWithFileCreateTruncate() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        this.dfs.setQuotaByStorageType(foo, StorageType.SSD, 4096L);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((boolean)fnode.isQuotaSet());
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long ssdConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        int newFile1Len = 1024;
        this.dfs.truncate(createdFile1, (long)newFile1Len);
        ssdConsumed = fnode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)newFile1Len, (long)ssdConsumed);
        ContentSummary cs = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)cs.getSpaceConsumed(), (long)(newFile1Len * 3));
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.SSD), (long)newFile1Len);
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.DISK), (long)(newFile1Len * 2));
    }

    @Test
    public void testQuotaByStorageTypePersistenceInEditLog() throws IOException {
        String METHOD_NAME = GenericTestUtils.getMethodName();
        Path testDir = new Path(dir, METHOD_NAME);
        Path createdFile1 = new Path(testDir, "created_file1.data");
        this.dfs.mkdirs(testDir);
        this.dfs.setStoragePolicy(testDir, "ONE_SSD");
        long SSD_QUOTA = 4096L;
        this.dfs.setQuotaByStorageType(testDir, StorageType.SSD, 4096L);
        INode testDirNode = this.fsdir.getINode4Write(testDir.toString());
        Assert.assertTrue((boolean)testDirNode.isDirectory());
        Assert.assertTrue((boolean)testDirNode.isQuotaSet());
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long ssdConsumed = testDirNode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        this.cluster.restartNameNode(true);
        INode testDirNodeAfterNNRestart = this.fsdir.getINode4Write(testDir.toString());
        Assert.assertTrue((boolean)testDirNode.isDirectory());
        Assert.assertTrue((boolean)testDirNode.isQuotaSet());
        QuotaCounts qc = testDirNodeAfterNNRestart.getQuotaCounts();
        Assert.assertEquals((long)4096L, (long)qc.getTypeSpace(StorageType.SSD));
        for (StorageType t : StorageType.getTypesSupportingQuota()) {
            if (t == StorageType.SSD) continue;
            Assert.assertEquals((long)-1L, (long)qc.getTypeSpace(t));
        }
        long ssdConsumedAfterNNRestart = testDirNodeAfterNNRestart.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumedAfterNNRestart);
    }

    @Test
    public void testQuotaByStorageTypePersistenceInFsImage() throws IOException {
        String METHOD_NAME = GenericTestUtils.getMethodName();
        Path testDir = new Path(dir, METHOD_NAME);
        Path createdFile1 = new Path(testDir, "created_file1.data");
        this.dfs.mkdirs(testDir);
        this.dfs.setStoragePolicy(testDir, "ONE_SSD");
        long SSD_QUOTA = 4096L;
        this.dfs.setQuotaByStorageType(testDir, StorageType.SSD, 4096L);
        INode testDirNode = this.fsdir.getINode4Write(testDir.toString());
        Assert.assertTrue((boolean)testDirNode.isDirectory());
        Assert.assertTrue((boolean)testDirNode.isQuotaSet());
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        long ssdConsumed = testDirNode.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumed);
        this.dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        this.dfs.saveNamespace();
        this.dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        this.cluster.restartNameNode(true);
        INode testDirNodeAfterNNRestart = this.fsdir.getINode4Write(testDir.toString());
        Assert.assertTrue((boolean)testDirNode.isDirectory());
        Assert.assertTrue((boolean)testDirNode.isQuotaSet());
        QuotaCounts qc = testDirNodeAfterNNRestart.getQuotaCounts();
        Assert.assertEquals((long)4096L, (long)qc.getTypeSpace(StorageType.SSD));
        for (StorageType t : StorageType.getTypesSupportingQuota()) {
            if (t == StorageType.SSD) continue;
            Assert.assertEquals((long)-1L, (long)qc.getTypeSpace(t));
        }
        long ssdConsumedAfterNNRestart = testDirNodeAfterNNRestart.asDirectory().getDirectoryWithQuotaFeature().getSpaceConsumed().getTypeSpaces().get((Enum)StorageType.SSD);
        Assert.assertEquals((long)file1Len, (long)ssdConsumedAfterNNRestart);
    }

    @Test(timeout=60000L)
    public void testContentSummaryWithoutQuotaByStorageType() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        this.dfs.setStoragePolicy(foo, "ONE_SSD");
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((!fnode.isQuotaSet() ? 1 : 0) != 0);
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        ContentSummary cs = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)cs.getSpaceConsumed(), (long)(file1Len * 3L));
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.SSD), (long)file1Len);
        Assert.assertEquals((long)cs.getTypeConsumed(StorageType.DISK), (long)(file1Len * 2L));
    }

    @Test(timeout=60000L)
    public void testContentSummaryWithoutStoragePolicy() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile1 = new Path(foo, "created_file1.data");
        this.dfs.mkdirs(foo);
        INode fnode = this.fsdir.getINode4Write(foo.toString());
        Assert.assertTrue((boolean)fnode.isDirectory());
        Assert.assertTrue((!fnode.isQuotaSet() ? 1 : 0) != 0);
        long file1Len = 2048L;
        int bufLen = 64;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile1, bufLen, file1Len, 1024L, (short)3, 0L);
        ContentSummary cs = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)cs.getSpaceConsumed(), (long)(file1Len * 3L));
        for (StorageType t : StorageType.values()) {
            Assert.assertEquals((long)cs.getTypeConsumed(t), (long)0L);
            Assert.assertEquals((long)cs.getTypeQuota(t), (long)-1L);
        }
    }
}

