/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io.compress;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.compress.BZip2Codec;
import org.apache.hadoop.io.compress.CodecPool;
import org.apache.hadoop.io.compress.CompressionCodec;
import org.apache.hadoop.io.compress.Decompressor;
import org.apache.hadoop.io.compress.SplitCompressionInputStream;
import org.apache.hadoop.io.compress.SplittableCompressionCodec;
import org.apache.hadoop.io.compress.bzip2.BZip2TextFileWriter;
import org.apache.hadoop.io.compress.bzip2.BZip2Utils;
import org.apache.hadoop.thirdparty.com.google.common.primitives.Bytes;
import org.apache.hadoop.util.Preconditions;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AssertionsForClassTypes;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public final class TestBZip2Codec {
    private static final long HEADER_LEN = 2L;
    private Configuration conf;
    private FileSystem fs;
    private BZip2Codec codec;
    private Decompressor decompressor;
    private Path tempFile;

    @Before
    public void setUp() throws Exception {
        this.conf = new Configuration();
        Path workDir = new Path(System.getProperty("test.build.data", "target"), "data/" + this.getClass().getSimpleName());
        Path inputDir = new Path(workDir, "input");
        this.tempFile = new Path(inputDir, "test.txt.bz2");
        this.fs = workDir.getFileSystem(this.conf);
        this.codec = new BZip2Codec();
        this.codec.setConf(new Configuration(false));
        this.decompressor = CodecPool.getDecompressor((CompressionCodec)this.codec);
    }

    @After
    public void tearDown() throws Exception {
        CodecPool.returnDecompressor((Decompressor)this.decompressor);
        this.fs.delete(this.tempFile, false);
    }

    @Test
    public void createInputStreamWithStartAndEnd() throws Exception {
        byte[] data1 = TestBZip2Codec.newAlternatingByteArray(BZip2TextFileWriter.BLOCK_SIZE, 97, 98);
        byte[] data2 = TestBZip2Codec.newAlternatingByteArray(BZip2TextFileWriter.BLOCK_SIZE, 99, 100);
        byte[] data3 = TestBZip2Codec.newAlternatingByteArray(BZip2TextFileWriter.BLOCK_SIZE, 101, 102);
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.write(data1);
            writer.write(data2);
            writer.write(data3);
        }
        long fileSize = this.fs.getFileStatus(this.tempFile).getLen();
        List<Long> nextBlockOffsets = BZip2Utils.getNextBlockMarkerOffsets(this.tempFile, this.conf);
        long block2Start = nextBlockOffsets.get(0);
        long block3Start = nextBlockOffsets.get(1);
        try (SplitCompressionInputStream stream = this.newCompressionStream(this.tempFile, 0L, fileSize, SplittableCompressionCodec.READ_MODE.BYBLOCK);){
            Assert.assertEquals((long)0L, (long)stream.getPos());
            TestBZip2Codec.assertCasesWhereReadDoesNotAdvanceStream(stream);
            TestBZip2Codec.assertReadingAtPositionZero(stream, data1);
            TestBZip2Codec.assertCasesWhereReadDoesNotAdvanceStream(stream);
            TestBZip2Codec.assertReadingPastEndOfBlock(stream, block2Start, data2);
            TestBZip2Codec.assertReadingPastEndOfBlock(stream, block3Start, data3);
            Assert.assertEquals((long)-1L, (long)stream.read());
        }
        stream = this.newCompressionStream(this.tempFile, 1L, fileSize - 1L, SplittableCompressionCodec.READ_MODE.BYBLOCK);
        try {
            Assert.assertEquals((long)block2Start, (long)stream.getPos());
            TestBZip2Codec.assertCasesWhereReadDoesNotAdvanceStream(stream);
            TestBZip2Codec.assertReadingPastEndOfBlock(stream, block2Start, data2);
            TestBZip2Codec.assertCasesWhereReadDoesNotAdvanceStream(stream);
            TestBZip2Codec.assertReadingPastEndOfBlock(stream, block3Start, data3);
            Assert.assertEquals((long)-1L, (long)stream.read());
        }
        finally {
            if (stream != null) {
                stream.close();
            }
        }
        byte[] allData = Bytes.concat((byte[][])new byte[][]{data1, data2, data3});
        this.assertReadingWithContinuousMode(this.tempFile, 0L, fileSize, allData);
        this.assertReadingWithContinuousMode(this.tempFile, 2L, fileSize - 2L, allData);
    }

    private void assertReadingWithContinuousMode(Path file, long start, long length, byte[] expectedData) throws IOException {
        try (SplitCompressionInputStream stream = this.newCompressionStream(file, start, length, SplittableCompressionCodec.READ_MODE.CONTINUOUS);){
            Assert.assertEquals((long)2L, (long)stream.getPos());
            TestBZip2Codec.assertRead((InputStream)stream, expectedData);
            Assert.assertEquals((long)-1L, (long)stream.read());
            Assert.assertEquals((long)2L, (long)stream.getPos());
        }
    }

    private SplitCompressionInputStream newCompressionStream(Path file, long start, long length, SplittableCompressionCodec.READ_MODE readMode) throws IOException {
        FSDataInputStream rawIn = this.fs.open(file);
        rawIn.seek(start);
        long end = start + length;
        return this.codec.createInputStream((InputStream)rawIn, this.decompressor, start, end, readMode);
    }

    private static byte[] newAlternatingByteArray(int size, int ... choices) {
        Preconditions.checkArgument((choices.length > 1 ? 1 : 0) != 0);
        byte[] result = new byte[size];
        for (int i = 0; i < size; ++i) {
            result[i] = (byte)choices[i % choices.length];
        }
        return result;
    }

    private static void assertCasesWhereReadDoesNotAdvanceStream(SplitCompressionInputStream in) throws IOException {
        long initialPos = in.getPos();
        Assert.assertEquals((long)0L, (long)in.read(new byte[0]));
        Assertions.assertThatNullPointerException().isThrownBy(() -> in.read(null, 0, 1));
        AssertionsForClassTypes.assertThatExceptionOfType(IndexOutOfBoundsException.class).isThrownBy(() -> in.read(new byte[5], -1, 2));
        AssertionsForClassTypes.assertThatExceptionOfType(IndexOutOfBoundsException.class).isThrownBy(() -> in.read(new byte[5], 0, -1));
        AssertionsForClassTypes.assertThatExceptionOfType(IndexOutOfBoundsException.class).isThrownBy(() -> in.read(new byte[5], 1, 5));
        Assert.assertEquals((long)initialPos, (long)in.getPos());
    }

    private static void assertReadingAtPositionZero(SplitCompressionInputStream in, byte[] expectedData) throws IOException {
        byte[] buffer = new byte[expectedData.length];
        Assert.assertEquals((long)1L, (long)in.read(buffer, 0, 1));
        Assert.assertEquals((long)expectedData[0], (long)buffer[0]);
        Assert.assertEquals((long)0L, (long)in.getPos());
        IOUtils.readFully((InputStream)in, (byte[])buffer, (int)1, (int)(expectedData.length - 1));
        Assert.assertArrayEquals((byte[])expectedData, (byte[])buffer);
        Assert.assertEquals((long)0L, (long)in.getPos());
    }

    private static void assertReadingPastEndOfBlock(SplitCompressionInputStream in, long endOfBlockPos, byte[] expectedData) throws IOException {
        byte[] buffer = new byte[expectedData.length];
        Assert.assertEquals((long)1L, (long)in.read(buffer));
        Assert.assertEquals((long)expectedData[0], (long)buffer[0]);
        Assert.assertEquals((long)(endOfBlockPos + 1L), (long)in.getPos());
        IOUtils.readFully((InputStream)in, (byte[])buffer, (int)1, (int)(expectedData.length - 1));
        Assert.assertArrayEquals((byte[])expectedData, (byte[])buffer);
        Assert.assertEquals((long)(endOfBlockPos + 1L), (long)in.getPos());
    }

    private static void assertRead(InputStream in, byte[] expectedData) throws IOException {
        byte[] buffer = new byte[expectedData.length];
        IOUtils.readFully((InputStream)in, (byte[])buffer);
        Assert.assertArrayEquals((byte[])expectedData, (byte[])buffer);
    }
}

