/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.yarn.zk;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.drill.exec.proto.CoordinationProtos;
import org.apache.drill.exec.work.foreman.DrillbitStatusListener;
import org.apache.drill.yarn.appMaster.AMWrapperException;
import org.apache.drill.yarn.appMaster.EventContext;
import org.apache.drill.yarn.appMaster.Pollable;
import org.apache.drill.yarn.appMaster.RegistryHandler;
import org.apache.drill.yarn.appMaster.Task;
import org.apache.drill.yarn.appMaster.TaskLifecycleListener;
import org.apache.drill.yarn.zk.ZKClusterCoordinatorDriver;
import org.apache.drill.yarn.zk.ZKRuntimeException;

public class ZKRegistry
implements TaskLifecycleListener,
DrillbitStatusListener,
Pollable {
    public static final String CONTROLLER_PROPERTY = "zk";
    public static final int UPDATE_PERIOD_MS = 20000;
    public static final String ENDPOINT_PROPERTY = "endpoint";
    private static final Log LOG = LogFactory.getLog(ZKRegistry.class);
    private Map<String, DrillbitTracker> registry = new HashMap<String, DrillbitTracker>();
    private ZKClusterCoordinatorDriver zkDriver;
    private RegistryHandler registryHandler;
    private long lastUpdateTime;

    public ZKRegistry(ZKClusterCoordinatorDriver zkDriver) {
        this.zkDriver = zkDriver;
    }

    public void start(RegistryHandler controller) {
        this.registryHandler = controller;
        try {
            this.zkDriver.build();
        }
        catch (ZKRuntimeException e) {
            LOG.error((Object)"Failed to start ZK monitoring", (Throwable)e);
            throw new AMWrapperException("Failed to start ZK monitoring", e);
        }
        for (CoordinationProtos.DrillbitEndpoint dbe : this.zkDriver.getInitialEndpoints()) {
            String key = this.toKey(dbe);
            this.registry.put(key, new DrillbitTracker(key, dbe));
            controller.reserveHost(dbe.getAddress());
            LOG.warn((Object)("Host " + dbe.getAddress() + " already running a Drillbit outside of YARN."));
        }
        this.zkDriver.addDrillbitListener(this);
    }

    private String toKey(CoordinationProtos.DrillbitEndpoint dbe) {
        return ZKClusterCoordinatorDriver.asString(dbe);
    }

    private String toKey(Task task) {
        return this.zkDriver.toKey(task.getHostName());
    }

    public void drillbitRegistered(Set<CoordinationProtos.DrillbitEndpoint> registeredDrillbits) {
        List<AckEvent> updates = this.registerDrillbits(registeredDrillbits);
        for (AckEvent event : updates) {
            if (event.task == null) {
                this.registryHandler.reserveHost(event.endpoint.getAddress());
                continue;
            }
            this.registryHandler.startAck(event.task, ENDPOINT_PROPERTY, event.endpoint);
        }
    }

    private synchronized List<AckEvent> registerDrillbits(Set<CoordinationProtos.DrillbitEndpoint> registeredDrillbits) {
        ArrayList<AckEvent> events = new ArrayList<AckEvent>();
        for (CoordinationProtos.DrillbitEndpoint dbe : registeredDrillbits) {
            AckEvent event = this.drillbitRegistered(dbe);
            if (event == null) continue;
            events.add(event);
        }
        return events;
    }

    private AckEvent drillbitRegistered(CoordinationProtos.DrillbitEndpoint dbe) {
        String key = this.toKey(dbe);
        DrillbitTracker tracker = this.registry.get(key);
        if (tracker == null) {
            LOG.info((Object)("Registration of unmanaged drillbit: " + key));
            tracker = new DrillbitTracker(key, dbe);
            this.registry.put(key, tracker);
            return new AckEvent(null, dbe);
        }
        if (tracker.state == DrillbitTracker.State.REGISTERED) {
            LOG.info((Object)("Re-registration of known drillbit: " + key));
            return null;
        }
        LOG.info((Object)("Drillbit registered: " + key + ", task: " + tracker.task.toString()));
        tracker.endpoint = dbe;
        tracker.becomeRegistered();
        return new AckEvent(tracker.task, dbe);
    }

    public void drillbitUnregistered(Set<CoordinationProtos.DrillbitEndpoint> unregisteredDrillbits) {
        List<AckEvent> updates = this.unregisterDrillbits(unregisteredDrillbits);
        for (AckEvent event : updates) {
            this.registryHandler.completionAck(event.task, ENDPOINT_PROPERTY);
        }
    }

    private synchronized List<AckEvent> unregisterDrillbits(Set<CoordinationProtos.DrillbitEndpoint> unregisteredDrillbits) {
        ArrayList<AckEvent> events = new ArrayList<AckEvent>();
        for (CoordinationProtos.DrillbitEndpoint dbe : unregisteredDrillbits) {
            AckEvent event = this.drillbitUnregistered(dbe);
            if (event == null) continue;
            events.add(event);
        }
        return events;
    }

    private AckEvent drillbitUnregistered(CoordinationProtos.DrillbitEndpoint dbe) {
        String key = this.toKey(dbe);
        DrillbitTracker tracker = this.registry.get(key);
        assert (tracker != null);
        if (tracker == null) {
            LOG.error((Object)("Internal error - Unexpected drillbit unregistration: " + key));
            return null;
        }
        if (tracker.state == DrillbitTracker.State.UNMANAGED) {
            assert (tracker.task == null);
            LOG.info((Object)("Unmanaged drillbit unregistered: " + key));
            this.registry.remove(key);
            this.registryHandler.releaseHost(dbe.getAddress());
            return null;
        }
        LOG.info((Object)("Drillbit unregistered: " + key + ", task: " + tracker.task.toString()));
        tracker.becomeUnregistered();
        return new AckEvent(tracker.task, dbe);
    }

    @Override
    public synchronized void stateChange(TaskLifecycleListener.Event event, EventContext context) {
        switch (event) {
            case ALLOCATED: {
                this.taskCreated(context.task);
                break;
            }
            case ENDED: {
                this.taskEnded(context.task);
                break;
            }
        }
    }

    private void taskCreated(Task task) {
        String key = this.toKey(task);
        DrillbitTracker tracker = this.registry.get(key);
        if (tracker == null) {
            this.registry.put(key, new DrillbitTracker(key, task));
        } else if (tracker.state == DrillbitTracker.State.UNMANAGED) {
            LOG.info((Object)("Unmanaged drillbit became managed: " + key));
            tracker.task = task;
            tracker.becomeRegistered();
            this.registryHandler.startAck(task, ENDPOINT_PROPERTY, tracker.endpoint);
        } else {
            LOG.error((Object)(task.getLabel() + " - Drillbit registry in wrong state " + (Object)((Object)tracker.state) + " for new task: " + key));
        }
    }

    public synchronized boolean isRegistered(Task task) {
        String key = this.toKey(task);
        DrillbitTracker tracker = this.registry.get(key);
        if (tracker == null) {
            return false;
        }
        return tracker.state == DrillbitTracker.State.REGISTERED;
    }

    private void taskEnded(Task task) {
        if (task.getHostName() == null) {
            return;
        }
        String key = this.toKey(task);
        DrillbitTracker tracker = this.registry.get(key);
        assert (tracker != null);
        assert (tracker.state == DrillbitTracker.State.DEREGISTERED);
        this.registry.remove(key);
    }

    @Override
    public void tick(long curTime) {
        if (this.lastUpdateTime + 20000L < curTime) {
            return;
        }
        this.lastUpdateTime = curTime;
        if (this.zkDriver.hasFailed()) {
            int secs = (int)((this.zkDriver.getLostConnectionDurationMs() + 500L) / 1000L);
            LOG.error((Object)("ZooKeeper connection lost, failing after " + secs + " seconds."));
            this.registryHandler.registryDown();
        }
    }

    public void finish(RegistryHandler handler) {
        this.zkDriver.removeDrillbitListener(this);
        this.zkDriver.close();
    }

    public synchronized List<String> listUnmanagedDrillits() {
        ArrayList<String> drillbits = new ArrayList<String>();
        for (DrillbitTracker item : this.registry.values()) {
            if (item.state != DrillbitTracker.State.UNMANAGED) continue;
            drillbits.add(item.key);
        }
        return drillbits;
    }

    @VisibleForTesting
    protected Map<String, DrillbitTracker> getRegistryForTesting() {
        return this.registry;
    }

    public static class AckEvent {
        Task task;
        CoordinationProtos.DrillbitEndpoint endpoint;

        public AckEvent(Task task, CoordinationProtos.DrillbitEndpoint endpoint) {
            this.task = task;
            this.endpoint = endpoint;
        }
    }

    protected static class DrillbitTracker {
        protected final String key;
        protected State state;
        protected Task task;
        protected CoordinationProtos.DrillbitEndpoint endpoint;

        public DrillbitTracker(String key, CoordinationProtos.DrillbitEndpoint endpoint) {
            this.key = key;
            this.state = State.UNMANAGED;
            this.endpoint = endpoint;
        }

        public DrillbitTracker(String key, Task task) {
            this.key = key;
            this.task = task;
            this.state = State.NEW;
        }

        private void becomeRegistered() {
            this.state = State.REGISTERED;
        }

        public void becomeUnregistered() {
            assert (this.state == State.REGISTERED);
            this.state = State.DEREGISTERED;
            this.endpoint = null;
        }

        public static enum State {
            UNMANAGED,
            NEW,
            REGISTERED,
            DEREGISTERED;

        }
    }
}

