/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.parse.repl;

import hive.com.google.common.collect.Lists;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.security.auth.login.LoginException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.ReplChangeManager;
import org.apache.hadoop.hive.ql.ErrorMsg;
import org.apache.hadoop.hive.ql.metadata.HiveFatalException;
import org.apache.hadoop.hive.shims.ShimLoader;
import org.apache.hadoop.hive.shims.Utils;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CopyUtils {
    private static final Logger LOG = LoggerFactory.getLogger(CopyUtils.class);
    private static final String RAW_RESERVED_VIRTUAL_PATH = "/.reserved/raw/";
    private static final int MAX_COPY_RETRY = 5;
    private final HiveConf hiveConf;
    private final long maxCopyFileSize;
    private final long maxNumberOfFiles;
    private final boolean hiveInTest;
    private final String copyAsUser;

    public CopyUtils(String distCpDoAsUser, HiveConf hiveConf) {
        this.hiveConf = hiveConf;
        this.maxNumberOfFiles = hiveConf.getLongVar(HiveConf.ConfVars.HIVE_EXEC_COPYFILE_MAXNUMFILES);
        this.maxCopyFileSize = hiveConf.getLongVar(HiveConf.ConfVars.HIVE_EXEC_COPYFILE_MAXSIZE);
        this.hiveInTest = hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_IN_TEST);
        this.copyAsUser = distCpDoAsUser;
    }

    public void copyAndVerify(FileSystem destinationFs, Path destRoot, List<ReplChangeManager.FileInfo> srcFiles) throws IOException, LoginException, HiveFatalException {
        Map<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> map = this.fsToFileMap(srcFiles, destRoot);
        for (Map.Entry<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> entry : map.entrySet()) {
            FileSystem sourceFs = entry.getKey();
            Map<Path, List<ReplChangeManager.FileInfo>> destMap = entry.getValue();
            for (Map.Entry<Path, List<ReplChangeManager.FileInfo>> destMapEntry : destMap.entrySet()) {
                Path destination = destMapEntry.getKey();
                List<ReplChangeManager.FileInfo> fileInfoList = destMapEntry.getValue();
                boolean useRegularCopy = this.regularCopy(destinationFs, sourceFs, fileInfoList);
                if (!destinationFs.exists(destination) && !FileUtils.mkdir(destinationFs, destination, this.hiveConf)) {
                    LOG.error("Failed to create destination directory: " + destination);
                    throw new IOException("Destination directory creation failed");
                }
                this.doCopyRetry(sourceFs, fileInfoList, destinationFs, destination, useRegularCopy);
            }
        }
    }

    private void doCopyRetry(FileSystem sourceFs, List<ReplChangeManager.FileInfo> srcFileList, FileSystem destinationFs, Path destination, boolean useRegularCopy) throws IOException, LoginException, HiveFatalException {
        boolean isCopyError = false;
        List<Object> pathList = Lists.transform(srcFileList, ReplChangeManager.FileInfo::getEffectivePath);
        for (int repeat = 0; !pathList.isEmpty() && repeat < 5; ++repeat) {
            try {
                if (repeat > 0 && (pathList = this.getFilesToRetry(sourceFs, srcFileList, destinationFs, destination, isCopyError)).isEmpty()) break;
                LOG.info("Attempt: " + (repeat + 1) + ". Copying files: " + pathList);
                isCopyError = true;
                this.doCopyOnce(sourceFs, pathList, destinationFs, destination, useRegularCopy);
                isCopyError = false;
                continue;
            }
            catch (IOException e) {
                LOG.info("file operation failed", (Throwable)e);
                if (repeat >= 4) break;
                if (e instanceof FileNotFoundException) continue;
                int sleepTime = FileUtils.getSleepTime(repeat);
                LOG.info("Sleep for " + sleepTime + " milliseconds before retry " + (repeat + 1));
                try {
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException timerEx) {
                    LOG.info("sleep interrupted", (Object)timerEx.getMessage());
                }
                FileSystem.closeAllForUGI((UserGroupInformation)Utils.getUGI());
                sourceFs = ((Path)pathList.get(0)).getFileSystem((Configuration)this.hiveConf);
                destinationFs = destination.getFileSystem((Configuration)this.hiveConf);
            }
        }
        if (!pathList.isEmpty()) {
            LOG.error("File copy failed even after several attempts. Files list: " + pathList);
            throw new IOException(ErrorMsg.REPL_FILE_SYSTEM_OPERATION_RETRY.getMsg());
        }
    }

    private List<Path> getFilesToRetry(FileSystem sourceFs, List<ReplChangeManager.FileInfo> srcFileList, FileSystem destinationFs, Path destination, boolean isCopyError) throws IOException, HiveFatalException {
        ArrayList<Path> pathList = new ArrayList<Path>();
        for (ReplChangeManager.FileInfo srcFile : srcFileList) {
            if (srcFile.isCopyDone()) continue;
            Path srcPath = srcFile.getEffectivePath();
            Path destPath = new Path(destination, srcPath.getName());
            if (destinationFs.exists(destPath)) {
                if (this.isSourceFileMismatch(sourceFs, srcFile)) {
                    destinationFs.delete(destPath, true);
                    srcFile.setIsUseSourcePath(false);
                } else if (!isCopyError) {
                    srcFile.setCopyDone(true);
                    continue;
                }
            } else if (this.isSourceFileMismatch(sourceFs, srcFile)) {
                srcFile.setIsUseSourcePath(false);
            }
            srcPath = srcFile.getEffectivePath();
            if (null == srcPath) {
                LOG.error("File copy failed and likely source file is deleted or modified.Source File: " + srcFile.getSourcePath());
                throw new HiveFatalException(ErrorMsg.REPL_FILE_MISSING_FROM_SRC_AND_CM_PATH.getMsg());
            }
            if (!srcFile.isUseSourcePath() && !sourceFs.exists(srcFile.getCmPath())) {
                LOG.error("File Copy Failed. Both source and CM files are missing from source. Missing Source File: " + srcFile.getSourcePath() + ", CM File: " + srcFile.getCmPath() + ". Try setting higher value for hive.repl.cm.retain in source warehouse. Also, bootstrap the system again to get back the consistent replicated state.");
                throw new HiveFatalException(ErrorMsg.REPL_FILE_MISSING_FROM_SRC_AND_CM_PATH.getMsg());
            }
            pathList.add(srcPath);
        }
        return pathList;
    }

    private boolean isSourceFileMismatch(FileSystem sourceFs, ReplChangeManager.FileInfo srcFile) {
        String sourceChecksumString;
        if (srcFile.isUseSourcePath() && (sourceChecksumString = srcFile.getCheckSum()) != null) {
            String verifySourceChecksumString;
            try {
                verifySourceChecksumString = ReplChangeManager.checksumFor(srcFile.getSourcePath(), sourceFs);
            }
            catch (IOException e) {
                LOG.debug("Unable to calculate checksum for source file: " + srcFile.getSourcePath());
                return true;
            }
            if (!sourceChecksumString.equals(verifySourceChecksumString)) {
                return true;
            }
        }
        return false;
    }

    private void doCopyOnce(FileSystem sourceFs, List<Path> srcList, FileSystem destinationFs, Path destination, boolean useRegularCopy) throws IOException, LoginException {
        boolean usePrivilegedUser;
        UserGroupInformation ugi = Utils.getUGI();
        String currentUser = ugi.getShortUserName();
        boolean bl = usePrivilegedUser = this.copyAsUser != null && !currentUser.equals(this.copyAsUser);
        if (useRegularCopy) {
            this.doRegularCopyOnce(sourceFs, srcList, destinationFs, destination, usePrivilegedUser);
        } else {
            this.doDistCpCopyOnce(sourceFs, srcList, destination, usePrivilegedUser);
        }
    }

    private void doDistCpCopyOnce(FileSystem sourceFs, List<Path> srcList, Path destination, boolean usePrivilegedUser) throws IOException {
        if (this.hiveConf.getBoolVar(HiveConf.ConfVars.REPL_ADD_RAW_RESERVED_NAMESPACE)) {
            srcList = srcList.stream().map(path -> {
                URI uri = path.toUri();
                return new Path(uri.getScheme(), uri.getAuthority(), RAW_RESERVED_VIRTUAL_PATH + uri.getPath());
            }).collect(Collectors.toList());
            URI destinationUri = destination.toUri();
            destination = new Path(destinationUri.getScheme(), destinationUri.getAuthority(), RAW_RESERVED_VIRTUAL_PATH + destinationUri.getPath());
        }
        FileUtils.distCp(sourceFs, srcList, destination, false, usePrivilegedUser ? this.copyAsUser : null, this.hiveConf, ShimLoader.getHadoopShims());
    }

    private void doRegularCopyOnce(FileSystem sourceFs, List<Path> srcList, FileSystem destinationFs, Path destination, boolean usePrivilegedUser) throws IOException {
        Path[] paths = srcList.toArray(new Path[0]);
        if (usePrivilegedUser) {
            Path finalDestination = destination;
            UserGroupInformation proxyUser = UserGroupInformation.createProxyUser((String)this.copyAsUser, (UserGroupInformation)UserGroupInformation.getLoginUser());
            try {
                proxyUser.doAs(() -> {
                    FileUtil.copy((FileSystem)sourceFs, (Path[])paths, (FileSystem)destinationFs, (Path)finalDestination, (boolean)false, (boolean)true, (Configuration)this.hiveConf);
                    return true;
                });
            }
            catch (InterruptedException e) {
                throw new IOException(e);
            }
        } else {
            FileUtil.copy((FileSystem)sourceFs, (Path[])paths, (FileSystem)destinationFs, (Path)destination, (boolean)false, (boolean)true, (Configuration)this.hiveConf);
        }
    }

    public void doCopy(Path destination, List<Path> srcPaths) throws IOException, LoginException {
        Map<FileSystem, List<Path>> map = this.fsToPathMap(srcPaths);
        FileSystem destinationFs = destination.getFileSystem((Configuration)this.hiveConf);
        for (Map.Entry<FileSystem, List<Path>> entry : map.entrySet()) {
            FileSystem sourceFs = entry.getKey();
            List<ReplChangeManager.FileInfo> fileList = Lists.transform(entry.getValue(), path -> new ReplChangeManager.FileInfo(sourceFs, (Path)path, null));
            this.doCopyOnce(sourceFs, entry.getValue(), destinationFs, destination, this.regularCopy(destinationFs, sourceFs, fileList));
        }
    }

    private boolean regularCopy(FileSystem destinationFs, FileSystem sourceFs, List<ReplChangeManager.FileInfo> fileList) throws IOException {
        if (this.hiveInTest) {
            return true;
        }
        if (this.isLocal(sourceFs) || this.isLocal(destinationFs)) {
            return true;
        }
        long size = 0L;
        long numberOfFiles = 0L;
        for (ReplChangeManager.FileInfo fileInfo : fileList) {
            ContentSummary contentSummary;
            block5: {
                contentSummary = null;
                try {
                    contentSummary = sourceFs.getContentSummary(fileInfo.getEffectivePath());
                }
                catch (IOException e) {
                    if (!fileInfo.isUseSourcePath() || fileInfo.getCmPath() == null) break block5;
                    contentSummary = sourceFs.getContentSummary(fileInfo.getCmPath());
                    fileInfo.setIsUseSourcePath(false);
                }
            }
            if (contentSummary == null || !this.limitReachedForLocalCopy(size += contentSummary.getLength(), numberOfFiles += contentSummary.getFileCount())) continue;
            return false;
        }
        return true;
    }

    boolean limitReachedForLocalCopy(long size, long numberOfFiles) {
        boolean result;
        boolean bl = result = size > this.maxCopyFileSize && numberOfFiles > this.maxNumberOfFiles;
        if (result) {
            LOG.info("Source is {} bytes. (MAX: {})", (Object)size, (Object)this.maxCopyFileSize);
            LOG.info("Source is {} files. (MAX: {})", (Object)numberOfFiles, (Object)this.maxNumberOfFiles);
            LOG.info("going to launch distributed copy (distcp) job.");
        }
        return result;
    }

    private boolean isLocal(FileSystem fs) {
        return fs.getScheme().equals("file");
    }

    private Map<FileSystem, List<Path>> fsToPathMap(List<Path> srcPaths) throws IOException {
        HashMap<FileSystem, List<Path>> result = new HashMap<FileSystem, List<Path>>();
        for (Path path : srcPaths) {
            FileSystem fileSystem = path.getFileSystem((Configuration)this.hiveConf);
            if (!result.containsKey(fileSystem)) {
                result.put(fileSystem, new ArrayList());
            }
            ((List)result.get(fileSystem)).add(path);
        }
        return result;
    }

    private Map<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> fsToFileMap(List<ReplChangeManager.FileInfo> srcFiles, Path destRoot) throws IOException {
        HashMap<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> result = new HashMap<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>>();
        for (ReplChangeManager.FileInfo file : srcFiles) {
            FileSystem fileSystem = file.getSrcFs();
            if (!result.containsKey(fileSystem)) {
                result.put(fileSystem, new HashMap());
            }
            Path destination = this.getCopyDestination(file, destRoot);
            if (!((Map)result.get(fileSystem)).containsKey(destination)) {
                ((Map)result.get(fileSystem)).put(destination, new ArrayList());
            }
            ((List)((Map)result.get(fileSystem)).get(destination)).add(file);
        }
        return result;
    }

    private Path getCopyDestination(ReplChangeManager.FileInfo fileInfo, Path destRoot) {
        if (fileInfo.getSubDir() == null) {
            return destRoot;
        }
        String[] subDirs = fileInfo.getSubDir().split("/");
        Path destination = destRoot;
        for (String subDir : subDirs) {
            destination = new Path(destination, subDir);
        }
        return destination;
    }
}

