/*
 * Decompiled with CFR 0.152.
 */
package org.apache.orc.tools.convert;

import hive.com.google.gson.JsonArray;
import hive.com.google.gson.JsonElement;
import hive.com.google.gson.JsonObject;
import hive.com.google.gson.JsonStreamParser;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.util.Iterator;
import java.util.List;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.hive.common.type.HiveDecimal;
import org.apache.hadoop.hive.ql.exec.vector.BytesColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.ColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.DecimalColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.DoubleColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.ListColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.LongColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.StructColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.TimestampColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch;
import org.apache.orc.RecordReader;
import org.apache.orc.TypeDescription;
import org.threeten.bp.LocalDateTime;
import org.threeten.bp.ZoneId;
import org.threeten.bp.ZonedDateTime;
import org.threeten.bp.format.DateTimeFormatter;
import org.threeten.bp.temporal.TemporalAccessor;

public class JsonReader
implements RecordReader {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy[[-][/]]MM[[-][/]]dd[['T'][ ]]HH:mm:ss[ ][XXX][X]");
    private final TypeDescription schema;
    private final Iterator<JsonElement> parser;
    private final JsonConverter[] converters;
    private final long totalSize;
    private final FSDataInputStream input;
    private long rowNumber = 0L;

    static JsonConverter createConverter(TypeDescription schema) {
        switch (schema.getCategory()) {
            case BYTE: 
            case SHORT: 
            case INT: 
            case LONG: {
                return new LongColumnConverter();
            }
            case FLOAT: 
            case DOUBLE: {
                return new DoubleColumnConverter();
            }
            case CHAR: 
            case VARCHAR: 
            case STRING: {
                return new StringColumnConverter();
            }
            case DECIMAL: {
                return new DecimalColumnConverter();
            }
            case TIMESTAMP: {
                return new TimestampColumnConverter();
            }
            case BINARY: {
                return new BinaryColumnConverter();
            }
            case BOOLEAN: {
                return new BooleanColumnConverter();
            }
            case STRUCT: {
                return new StructColumnConverter(schema);
            }
            case LIST: {
                return new ListColumnConverter(schema);
            }
        }
        throw new IllegalArgumentException("Unhandled type " + schema);
    }

    public JsonReader(Reader reader, FSDataInputStream underlying, long size, TypeDescription schema) throws IOException {
        this((Iterator<JsonElement>)new JsonStreamParser(reader), underlying, size, schema);
    }

    public JsonReader(Iterator<JsonElement> parser, FSDataInputStream underlying, long size, TypeDescription schema) throws IOException {
        this.schema = schema;
        if (schema.getCategory() != TypeDescription.Category.STRUCT) {
            throw new IllegalArgumentException("Root must be struct - " + schema);
        }
        this.input = underlying;
        this.totalSize = size;
        this.parser = parser;
        List<TypeDescription> fieldTypes = schema.getChildren();
        this.converters = new JsonConverter[fieldTypes.size()];
        for (int c = 0; c < this.converters.length; ++c) {
            this.converters[c] = JsonReader.createConverter(fieldTypes.get(c));
        }
    }

    @Override
    public boolean nextBatch(VectorizedRowBatch batch) throws IOException {
        batch.reset();
        int maxSize = batch.getMaxSize();
        List<String> fieldNames = this.schema.getFieldNames();
        while (this.parser.hasNext() && batch.size < maxSize) {
            JsonObject elem = this.parser.next().getAsJsonObject();
            for (int c = 0; c < this.converters.length; ++c) {
                JsonElement field = elem.get(fieldNames.get(c));
                if (field == null) {
                    batch.cols[c].noNulls = false;
                    batch.cols[c].isNull[batch.size] = true;
                    continue;
                }
                this.converters[c].convert(field, batch.cols[c], batch.size);
            }
            ++batch.size;
        }
        this.rowNumber += (long)batch.size;
        return batch.size != 0;
    }

    @Override
    public long getRowNumber() throws IOException {
        return this.rowNumber;
    }

    @Override
    public float getProgress() throws IOException {
        long pos = this.input.getPos();
        return this.totalSize != 0L && pos < this.totalSize ? (float)pos / (float)this.totalSize : 1.0f;
    }

    @Override
    public void close() throws IOException {
        this.input.close();
    }

    @Override
    public void seekToRow(long rowCount) throws IOException {
        throw new UnsupportedOperationException("Seek is not supported by JsonReader");
    }

    static class ListColumnConverter
    implements JsonConverter {
        private JsonConverter childrenConverter;

        public ListColumnConverter(TypeDescription schema) {
            this.childrenConverter = JsonReader.createConverter(schema.getChildren().get(0));
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                ListColumnVector vector = (ListColumnVector)vect;
                JsonArray obj = value.getAsJsonArray();
                vector.lengths[row] = obj.size();
                vector.offsets[row] = vector.childCount;
                vector.childCount = (int)((long)vector.childCount + vector.lengths[row]);
                vector.child.ensureSize(vector.childCount, true);
                for (int c = 0; c < obj.size(); ++c) {
                    this.childrenConverter.convert(obj.get(c), vector.child, (int)vector.offsets[row] + c);
                }
            }
        }
    }

    static class StructColumnConverter
    implements JsonConverter {
        private JsonConverter[] childrenConverters;
        private List<String> fieldNames;

        public StructColumnConverter(TypeDescription schema) {
            List<TypeDescription> kids = schema.getChildren();
            this.childrenConverters = new JsonConverter[kids.size()];
            for (int c = 0; c < this.childrenConverters.length; ++c) {
                this.childrenConverters[c] = JsonReader.createConverter(kids.get(c));
            }
            this.fieldNames = schema.getFieldNames();
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                StructColumnVector vector = (StructColumnVector)vect;
                JsonObject obj = value.getAsJsonObject();
                for (int c = 0; c < this.childrenConverters.length; ++c) {
                    JsonElement elem = obj.get(this.fieldNames.get(c));
                    this.childrenConverters[c].convert(elem, vector.fields[c], row);
                }
            }
        }
    }

    static class DecimalColumnConverter
    implements JsonConverter {
        DecimalColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                DecimalColumnVector vector = (DecimalColumnVector)vect;
                vector.vector[row].set(HiveDecimal.create(value.getAsString()));
            }
        }
    }

    static class TimestampColumnConverter
    implements JsonConverter {
        TimestampColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                TimestampColumnVector vector = (TimestampColumnVector)vect;
                TemporalAccessor temporalAccessor = DATE_TIME_FORMATTER.parseBest(value.getAsString(), ZonedDateTime.FROM, LocalDateTime.FROM);
                if (temporalAccessor instanceof ZonedDateTime) {
                    vector.set(row, new Timestamp(((ZonedDateTime)temporalAccessor).toEpochSecond() * 1000L));
                } else if (temporalAccessor instanceof LocalDateTime) {
                    vector.set(row, new Timestamp(((LocalDateTime)temporalAccessor).atZone(ZoneId.systemDefault()).toEpochSecond() * 1000L));
                } else {
                    vect.noNulls = false;
                    vect.isNull[row] = true;
                }
            }
        }
    }

    static class BinaryColumnConverter
    implements JsonConverter {
        BinaryColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                BytesColumnVector vector = (BytesColumnVector)vect;
                String binStr = value.getAsString();
                byte[] bytes = new byte[binStr.length() / 2];
                for (int i = 0; i < bytes.length; ++i) {
                    bytes[i] = (byte)Integer.parseInt(binStr.substring(i * 2, i * 2 + 2), 16);
                }
                vector.setRef(row, bytes, 0, bytes.length);
            }
        }
    }

    static class StringColumnConverter
    implements JsonConverter {
        StringColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                BytesColumnVector vector = (BytesColumnVector)vect;
                byte[] bytes = value.getAsString().getBytes(StandardCharsets.UTF_8);
                vector.setRef(row, bytes, 0, bytes.length);
            }
        }
    }

    static class DoubleColumnConverter
    implements JsonConverter {
        DoubleColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                DoubleColumnVector vector = (DoubleColumnVector)vect;
                vector.vector[row] = value.getAsDouble();
            }
        }
    }

    static class LongColumnConverter
    implements JsonConverter {
        LongColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                LongColumnVector vector = (LongColumnVector)vect;
                vector.vector[row] = value.getAsLong();
            }
        }
    }

    static class BooleanColumnConverter
    implements JsonConverter {
        BooleanColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                LongColumnVector vector = (LongColumnVector)vect;
                vector.vector[row] = value.getAsBoolean() ? 1L : 0L;
            }
        }
    }

    static interface JsonConverter {
        public void convert(JsonElement var1, ColumnVector var2, int var3);
    }
}

