/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.store.bson;

import com.google.common.base.Preconditions;
import io.netty.buffer.DrillBuf;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.List;
import org.apache.drill.common.exceptions.DrillRuntimeException;
import org.apache.drill.common.expression.PathSegment;
import org.apache.drill.common.expression.SchemaPath;
import org.apache.drill.exec.expr.holders.BigIntHolder;
import org.apache.drill.exec.expr.holders.BitHolder;
import org.apache.drill.exec.expr.holders.Float8Holder;
import org.apache.drill.exec.expr.holders.IntHolder;
import org.apache.drill.exec.expr.holders.VarBinaryHolder;
import org.apache.drill.exec.expr.holders.VarCharHolder;
import org.apache.drill.exec.physical.base.GroupScan;
import org.apache.drill.exec.vector.complex.impl.MapOrListWriterImpl;
import org.apache.drill.exec.vector.complex.writer.BaseWriter;
import org.apache.drill.exec.vector.complex.writer.TimeStampWriter;
import org.bson.BsonBinary;
import org.bson.BsonReader;
import org.bson.BsonType;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BsonRecordReader {
    static final Logger logger = LoggerFactory.getLogger(BsonRecordReader.class);
    public static final int MAX_RECORD_SIZE = 131072;
    private final List<SchemaPath> columns;
    private boolean atLeastOneWrite = false;
    private final boolean readNumbersAsDouble;
    protected DrillBuf workBuf;
    private String currentFieldName;
    private BsonReader reader;

    public BsonRecordReader(DrillBuf managedBuf, boolean allTextMode, boolean readNumbersAsDouble) {
        this(managedBuf, GroupScan.ALL_COLUMNS, readNumbersAsDouble);
    }

    public BsonRecordReader(DrillBuf managedBuf, List<SchemaPath> columns, boolean readNumbersAsDouble) {
        assert (((List)Preconditions.checkNotNull(columns)).size() > 0) : "bson record reader requires at least a column";
        this.readNumbersAsDouble = readNumbersAsDouble;
        this.workBuf = managedBuf;
        this.columns = columns;
    }

    public void write(BaseWriter.ComplexWriter writer, BsonReader reader) throws IOException {
        this.reader = reader;
        reader.readStartDocument();
        BsonType readBsonType = reader.getCurrentBsonType();
        switch (readBsonType) {
            case DOCUMENT: {
                this.writeToListOrMap(reader, new MapOrListWriterImpl(writer.rootAsMap()), false, null);
                break;
            }
            default: {
                throw new DrillRuntimeException("Root object must be DOCUMENT type. Found: " + readBsonType);
            }
        }
    }

    private void writeToListOrMap(BsonReader reader, MapOrListWriterImpl writer, boolean isList, String fieldName) {
        writer.start();
        block18: while (reader.readBsonType() != BsonType.END_OF_DOCUMENT) {
            if (!isList) {
                fieldName = reader.readName();
            }
            BsonType currentBsonType = reader.getCurrentBsonType();
            switch (currentBsonType) {
                case INT32: {
                    int readInt32 = reader.readInt32();
                    if (this.readNumbersAsDouble) {
                        this.writeDouble(readInt32, writer, fieldName, isList);
                    } else {
                        this.writeInt32(readInt32, writer, fieldName, isList);
                    }
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case INT64: {
                    long readInt64 = reader.readInt64();
                    if (this.readNumbersAsDouble) {
                        this.writeDouble(readInt64, writer, fieldName, isList);
                    } else {
                        this.writeInt64(readInt64, writer, fieldName, isList);
                    }
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case ARRAY: {
                    reader.readStartArray();
                    this.writeToListOrMap(reader, (MapOrListWriterImpl)writer.list(fieldName), true, fieldName);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case BINARY: {
                    this.writeBinary(reader, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case BOOLEAN: {
                    boolean readBoolean = reader.readBoolean();
                    this.writeBoolean(readBoolean, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case DATE_TIME: {
                    long readDateTime = reader.readDateTime();
                    this.writeDateTime(readDateTime, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case DOCUMENT: {
                    reader.readStartDocument();
                    MapOrListWriterImpl _writer = writer;
                    _writer = !isList ? (MapOrListWriterImpl)writer.map(fieldName) : (MapOrListWriterImpl)writer.listoftmap(fieldName);
                    this.writeToListOrMap(reader, _writer, false, fieldName);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case DOUBLE: {
                    double readDouble = reader.readDouble();
                    this.writeDouble(readDouble, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case JAVASCRIPT: {
                    String readJavaScript = reader.readJavaScript();
                    this.writeString(readJavaScript, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case JAVASCRIPT_WITH_SCOPE: {
                    String readJavaScriptWithScopeString = reader.readJavaScriptWithScope();
                    this.writeString(readJavaScriptWithScopeString, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case NULL: {
                    reader.readNull();
                    continue block18;
                }
                case OBJECT_ID: {
                    this.writeObjectId(reader, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case STRING: {
                    String readString = reader.readString();
                    this.writeString(readString, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case SYMBOL: {
                    String readSymbol = reader.readSymbol();
                    this.writeString(readSymbol, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case TIMESTAMP: {
                    int time = reader.readTimestamp().getTime();
                    this.writeTimeStamp(time, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
                case DECIMAL128: {
                    BigDecimal readBigDecimalAsDecimal128 = reader.readDecimal128().bigDecimalValue();
                    this.writeDecimal128(readBigDecimalAsDecimal128, writer, fieldName, isList);
                    this.atLeastOneWrite = true;
                    continue block18;
                }
            }
            throw new DrillRuntimeException("UnSupported Bson type: " + currentBsonType);
        }
        if (!isList) {
            reader.readEndDocument();
        } else {
            reader.readEndArray();
        }
    }

    private void writeBinary(BsonReader reader, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        VarBinaryHolder vb = new VarBinaryHolder();
        BsonBinary readBinaryData = reader.readBinaryData();
        byte[] data = readBinaryData.getData();
        Byte type = readBinaryData.getType();
        switch (type.intValue()) {
            case 1: {
                this.writeDouble(ByteBuffer.wrap(data).getDouble(), writer, fieldName, isList);
                break;
            }
            case 2: {
                this.writeString(new String(data, StandardCharsets.UTF_8), writer, fieldName, isList);
                break;
            }
            case 8: {
                boolean boolValue = data == null || data.length == 0 ? false : data[0] != 0;
                this.writeBoolean(boolValue, writer, fieldName, isList);
                break;
            }
            case 9: {
                this.writeDateTime(ByteBuffer.wrap(data).getLong(), writer, fieldName, isList);
                break;
            }
            case 13: {
                this.writeString(new String(data, StandardCharsets.UTF_8), writer, fieldName, isList);
                break;
            }
            case 14: {
                this.writeString(new String(data, StandardCharsets.UTF_8), writer, fieldName, isList);
                break;
            }
            case 15: {
                this.writeString(new String(data, StandardCharsets.UTF_8), writer, fieldName, isList);
                break;
            }
            case 16: {
                this.writeInt32(ByteBuffer.wrap(data).getInt(), writer, fieldName, isList);
                break;
            }
            case 17: {
                this.writeTimeStamp(ByteBuffer.wrap(data).getInt(), writer, fieldName, isList);
                break;
            }
            case 18: {
                this.writeInt64(ByteBuffer.wrap(data).getInt(), writer, fieldName, isList);
                break;
            }
            default: {
                byte[] bytes = readBinaryData.getData();
                this.writeBinary(writer, fieldName, isList, vb, bytes);
            }
        }
    }

    private void writeTimeStamp(int timestamp, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        DateTime dateTime = new DateTime((long)timestamp * 1000L);
        TimeStampWriter t = !isList ? writer.map.timeStamp(fieldName) : writer.list.timeStamp();
        t.writeTimeStamp(dateTime.withZoneRetainFields(DateTimeZone.UTC).getMillis());
    }

    private void writeString(String readString, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        byte[] strBytes;
        try {
            strBytes = readString.getBytes("UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new DrillRuntimeException("Unable to read string value for field: " + fieldName, (Throwable)e);
        }
        int length = strBytes.length;
        this.ensure(length);
        this.workBuf.setBytes(0, strBytes);
        VarCharHolder vh = new VarCharHolder();
        vh.buffer = this.workBuf;
        vh.start = 0;
        vh.end = length;
        if (!isList) {
            writer.varChar(fieldName).write(vh);
        } else {
            writer.list.varChar().write(vh);
        }
    }

    private void writeObjectId(BsonReader reader, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        VarBinaryHolder vObj = new VarBinaryHolder();
        byte[] objBytes = reader.readObjectId().toByteArray();
        this.writeBinary(writer, fieldName, isList, vObj, objBytes);
    }

    private void writeDouble(double readDouble, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        Float8Holder f8h = new Float8Holder();
        f8h.value = readDouble;
        if (!isList) {
            writer.float8(fieldName).write(f8h);
        } else {
            writer.list.float8().write(f8h);
        }
    }

    private void writeDateTime(long readDateTime, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        DateTime date = new DateTime(readDateTime);
        TimeStampWriter dt = !isList ? writer.map.timeStamp(fieldName) : writer.list.timeStamp();
        dt.writeTimeStamp(date.withZoneRetainFields(DateTimeZone.UTC).getMillis());
    }

    private void writeBoolean(boolean readBoolean, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        BitHolder bit = new BitHolder();
        int n = bit.value = readBoolean ? 1 : 0;
        if (!isList) {
            writer.bit(fieldName).write(bit);
        } else {
            writer.list.bit().write(bit);
        }
    }

    private void writeBinary(MapOrListWriterImpl writer, String fieldName, boolean isList, VarBinaryHolder vb, byte[] bytes) {
        this.ensure(bytes.length);
        this.workBuf.setBytes(0, bytes);
        vb.buffer = this.workBuf;
        vb.start = 0;
        vb.end = bytes.length;
        if (!isList) {
            writer.binary(fieldName).write(vb);
        } else {
            writer.list.varBinary().write(vb);
        }
    }

    private void writeInt64(long readInt64, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        BigIntHolder bh = new BigIntHolder();
        bh.value = readInt64;
        if (!isList) {
            writer.bigInt(fieldName).write(bh);
        } else {
            writer.list.bigInt().write(bh);
        }
    }

    private void writeInt32(int readInt32, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        IntHolder ih = new IntHolder();
        ih.value = readInt32;
        if (!isList) {
            writer.integer(fieldName).write(ih);
        } else {
            writer.list.integer().write(ih);
        }
    }

    private void writeDecimal128(BigDecimal readBigDecimal, MapOrListWriterImpl writer, String fieldName, boolean isList) {
        if (isList) {
            writer.list.varDecimal().writeVarDecimal(readBigDecimal);
        } else {
            writer.varDecimal(fieldName, readBigDecimal.precision(), readBigDecimal.scale()).writeVarDecimal(readBigDecimal);
        }
    }

    public void ensureAtLeastOneField(BaseWriter.ComplexWriter writer) {
        if (!this.atLeastOneWrite) {
            SchemaPath sp = this.columns.get(0);
            PathSegment.NameSegment root = sp.getRootSegment();
            BaseWriter.MapWriter fieldWriter = writer.rootAsMap();
            while (root.getChild() != null && !root.getChild().isArray()) {
                fieldWriter = fieldWriter.map(root.getNameSegment().getPath());
                root = root.getChild();
            }
            fieldWriter.integer(root.getNameSegment().getPath());
        }
    }

    private void ensure(int length) {
        this.workBuf = this.workBuf.reallocIfNeeded(length);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("BsonRecordReader[");
        if (this.reader != null) {
            sb.append("Name=").append(this.reader.getCurrentName()).append(", Type=").append(this.reader.getCurrentBsonType());
        }
        sb.append(']');
        return sb.toString();
    }
}

