/**
 * Copyright 2019 Confluent Inc.
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at</p>
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0</p>
 *
 * <p>Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.</p>
 */

package io.confluent.rest.impersonation;

import io.confluent.rest.exceptions.RestServerErrorException;
import org.apache.hadoop.security.UserGroupInformation;

import java.io.IOException;
import java.lang.reflect.UndeclaredThrowableException;
import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;

public class ImpersonatedExecutor {

  public <T> T runAs(String username, PrivilegedAction<T> action) {
    UserGroupInformation ugi = getImpersonatedUser(username);
    try {
      return ugi.doAs(action);
    } catch (RestServerErrorException e) {
      throw Errors.serverLoginException("It is not possible to do this operation", e);
    }
  }

  public <T> T runThrowingAs(String username, PrivilegedExceptionAction<T> action)
      throws Exception {
    UserGroupInformation ugi = getImpersonatedUser(username);
    try {
      return ugi.doAs(action);
    } catch (RestServerErrorException e) {
      throw Errors.serverLoginException("It is not possible to do this operation", e);
    } catch (UndeclaredThrowableException e) {
      throw e.getCause() instanceof Exception ? (Exception) e.getCause() : e;
    }
  }

  public UserGroupInformation getImpersonatedUser(String username) {
    try {
      UserGroupInformation currentUser = UserGroupInformation.getCurrentUser();
      if (currentUser.getUserName().equals(username)) {
        return currentUser;
      } else {
        return UserGroupInformation.createProxyUser(username, UserGroupInformation.getLoginUser());
      }
    } catch (IOException e) {
      throw Errors.serverLoginException(e);
    }
  }
}
