/**
 * Copyright 2014 Confluent Inc.
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at</p>
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0</p>
 *
 * <p>Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.</p>
 */

package io.confluent.rest;

import com.mapr.fs.jni.MapRConstants;
import com.mapr.web.security.WebSecurityConfig;
import io.confluent.common.utils.TestUtils;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.junit.Assert;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.Map;
import java.util.Objects;

/**
 * We can't have different MAPR_HOME for different tests
 * and also some MapR classes like WebSecurityConfig save the path on initialization
 * so changing won't be reflected by them
 */
public class MaprHomeSupport {
  public static final File MAPR_HOME_FOLDER = TestUtils.tempDirectory();
  public static final String MAPR_HOME = MAPR_HOME_FOLDER.getPath();

  public static void activate() {
    if (!isActive()) {
      System.setProperty(MapRConstants.MAPR_PROPERTY_HOME, MAPR_HOME);
      changeEnvironmentVariable();
      assertWebSecurityUsesNewValue();
    }
    resetToDefaults();
  }

  public static boolean isActive() {
    return Objects.equals(System.getenv(MapRConstants.MAPR_ENV_VAR), MAPR_HOME);
  }

  private static void assertWebSecurityUsesNewValue() {
    Assert.assertEquals(
        "To correctly stub MAPR_HOME "
                + "call MaprHomeSupport#stubMaprHomeForApplication "
                + "before environment variable is used",
        MAPR_HOME, WebSecurityConfig.CONFIG.getMaprHome());
  }

  public static void resetToDefaults() {
    Assert.assertTrue("MaprHomeSupport isn't active", isActive());
    try {
      FileUtils.deleteQuietly(MAPR_HOME_FOLDER);
      MAPR_HOME_FOLDER.mkdirs();
      new File(MAPR_HOME_FOLDER, "conf").mkdirs();
      new File(WebSecurityConfig.CONFIG.getMaprClusterCfgLocation()).createNewFile();
      createSslXmlConfiguration("conf/ssl-server.xml", "server");
      createSslXmlConfiguration("conf/ssl-client.xml", "client");
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private static void createSslXmlConfiguration(String path, String role) throws IOException {
    final Configuration xml = new Configuration();
    xml.set("ssl." + role + ".truststore.location", role + ".jks");
    xml.set("ssl." + role + ".truststore.type", "jks");
    xml.set("ssl." + role + ".keystore.location", role + ".jks");
    xml.set("ssl." + role + ".keystore.type", "jks");
    xml.set("ssl." + role + ".keystore.keypassword", "mapr");
    xml.set("ssl." + role + ".keystore.password", "mapr");
    xml.set("ssl." + role + ".truststore.password", "mapr");
    try (FileOutputStream stream = new FileOutputStream(new File(MAPR_HOME_FOLDER, path))) {
      xml.writeXml(stream);
    }
  }

  private static void changeEnvironmentVariable() {
    try {
      Map<String, String> env = System.getenv();
      Field field = env.getClass().getDeclaredField("m");
      field.setAccessible(true);
      ((Map<String, String>) field.get(env)).put(MapRConstants.MAPR_ENV_VAR, MAPR_HOME);
    } catch (NoSuchFieldException | IllegalAccessException e) {
      throw new RuntimeException("Cannot mutate environment", e);
    }
  }
}
